""" Information about current user """

from __future__ import absolute_import

from sandbox.common import api

import sandbox.common.types.user as ctu

from .api import Api
from . import schemas


class User(Api.Path("/user/{login}")):
    class Get(api.Get):
        """ Detailed information about a user """

        login = api.String("User login", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            """ Success """
            schema = schemas.user.User


class UserCurrent(Api.Path("/user/current")):
    class Get(api.Get):
        """ Information about current user """

        class Ok(api.Response):
            """ Success """
            schema = schemas.user.User


class UserGroups(Api.Path("/user/{login}/groups")):
    class Get(api.Get):
        """ List of all known groups specified user is in """

        login = api.String("User login", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            """ Success """
            schema = api.Array(schemas.suggest.SuggestGroup)


class UserCurrentGroups(Api.Path("/user/current/groups")):
    class Get(api.Get):
        """ List of all known groups current user is in """

        class Ok(api.Response):
            """ Success """
            schema = api.Array(schemas.suggest.SuggestGroup)


class UserCurrentRobots(Api.Path("/user/current/robots")):
    class Get(api.Get):
        """ List of robots owned by the current user """

        class Ok(api.Response):
            """ Success """
            schema = api.Array(schemas.user.Robot)


class UserPreferences(Api.Path("/user/{login}/preferences/{subnode}")):
    class Get(api.Get):
        """
        Specified user's preferences
        Arbitrary data of size up to 64Kb, for example, filter settings
        """
        __security__ = ctu.Restriction.AUTHENTICATED

        login = api.String("User login", required=True, scope=api.Scope.PATH)
        subnode = api.String("Data node identifier", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            """ Success """
            schema = api.Object

    class Put(api.Put):
        """
        Create/update specified user's preferences
        Arbitrary data of size up to 64Kb
        """
        __security__ = ctu.Restriction.AUTHENTICATED

        login = api.String("User login", required=True, scope=api.Scope.PATH)
        subnode = api.String("Data node identifier", required=True, scope=api.Scope.PATH)
        data = api.Object("Preferences node data", required=True, scope=api.Scope.BODY)

        class Created(api.Response):
            """
            Success
            """
            location = api.Header("URL for created preferences node")


class UserCurrentPreferencesGroupAbcLink(Api.Path("/user/current/preferences/group_abc_link")):
    class Get(api.Get):
        """
        Return user sandbox groups without abc link
        """
        __security__ = ctu.Restriction.AUTHENTICATED

        class Ok(api.Response):
            """ Success """
            schema = api.Array(api.String, "Groups without abc links")


class UserCurrentPreferences(Api.Path("/user/current/preferences/{subnode}")):
    class Get(api.Get):
        """
        Specified user's preferences
        Arbitrary data of size up to 64Kb, for example, filter settings
        """
        __security__ = ctu.Restriction.AUTHENTICATED

        subnode = api.String("Data node identifier", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            """ Success """
            schema = api.Object

    class Put(api.Put):
        """
        Create/update specified user's preferences
        Arbitrary data of size up to 64Kb
        """
        __security__ = ctu.Restriction.AUTHENTICATED

        subnode = api.String("Data node identifier", required=True, scope=api.Scope.PATH)
        data = api.Object("Preferences node data", required=True, scope=api.Scope.BODY)

        class Created(api.Response):
            """ Success """
            location = api.Header("URL for created preferences node")
