""" Operations on vault objects """

from __future__ import absolute_import

from sandbox.common import api
from sandbox.common import rest as common_rest
import sandbox.common.types.user as ctu

from .api import Api
from .common import FilterBase
from . import schemas


class VaultList(Api.Path("/vault")):
    class Get(FilterBase):
        """ List of vault items """
        name = api.String("Filter by element's name substring")
        owner = api.String("Search by (part of) owner name")
        description = api.String("Description of vault item")
        shared = api.String("Search by (part of) name of user/group, the element is shared with")

        class Ok(api.Response):
            """ Search result """
            schema = schemas.vault.VaultList

    class Post(api.Post):
        """ Create a new vault item """
        data = schemas.vault.VaultCreate(scope=api.Scope.BODY)

        class Created(api.Response):
            """ Object has been created successfully """
            schema = schemas.vault.Vault
            location = api.Header("URL to fetch just created object's data")


class Vault(Api.Path("/vault/{id}")):
    class Get(api.Get):
        """ Vault item object fetch """
        id = api.Id(scope=api.Scope.PATH)

        class Ok(api.Response):
            """ Item object """
            schema = schemas.vault.Vault

        class NotFound(api.Response):
            """ No object with ID provided can be found """

    class Put(api.Put):
        """ Vault item object update """
        id = api.Id(scope=api.Scope.PATH)
        data = schemas.vault.VaultUpdate(scope=api.Scope.BODY)

        class Ok(api.Response):
            """
            In case of request header `X-Request-Updated-Data` was passed, the updated item's object will be returned
            """
            schema = schemas.vault.Vault

        class NoContent(api.Response):
            """ Object has been updated successfully """

        class NotFound(api.Response):
            """ No object with ID provided can be found """

    class Delete(api.Delete):
        """ Delete the given item """
        id = api.Id(scope=api.Scope.PATH)

        class NoContent(api.Response):
            """ An object has been deleted successfully """

        class NotFound(api.Response):
            """ No object with ID provided can be found """


class VaultData(Api.Path("/vault/{id}/data")):
    class Get(api.Get):
        """ Vault item data fetch """
        __security__ = ctu.Restriction.TASK

        id = api.Id(scope=api.Scope.PATH)

        class Ok(api.Response):
            """ Item's data fetcher (works only for session scope requests) """
            __content__ = common_rest.Client.BINARY

            schema = api.String


class VaultDataByQuery(Api.Path("/vault/data")):
    class Get(api.Get):
        """ Vault item data fetch """
        __security__ = ctu.Restriction.TASK

        owner = api.String()
        name = api.String(required=True)

        class Ok(api.Response):
            """ Item's data fetcher (works only for session scope requests) """
            schema = schemas.vault.VaultData
