""" Endpoints related to Yandex Vault """

from __future__ import absolute_import

from sandbox.common import api
from sandbox.common.types import user as ctu

from . import schemas
from .api import Api


class Status(Api.Path("/yav/status")):

    class Post(api.Post):
        """ Check if the secrets are delegated to Sandbox. """

        __security__ = ctu.Restriction.AUTHENTICATED
        data = schemas.yav.SecretList(scope=api.Scope.BODY)

        class Ok(api.Response):
            """ List of delegation statuses """
            schema = schemas.yav.SecretStatusList

        class BadRequest(api.Response):
            """ Invalid secret uuid """


class Tokens(Api.Path("/yav/tokens")):

    class Post(api.Post):
        """ Delegate secrets to Sandbox. """

        __security__ = ctu.Restriction.AUTHENTICATED
        data = schemas.yav.SecretList(scope=api.Scope.BODY)

        class Ok(api.Response):
            """ Results of the secrets delegation """
            schema = schemas.yav.DelegationResultList

        class BadRequest(api.Response):
            """ Invalid secret uuid """


class Data(Api.Path("/yav/data")):

    class Post(api.Post):
        """ Get encrypted secret values by secret uuid and an optional version. Task session is required. """

        __security__ = ctu.Restriction.TASK
        data = schemas.yav.SecretAndVersionList(scope=api.Scope.BODY)

        class Ok(api.Response):
            """ List of secret values """
            schema = schemas.yav.EncryptedSecretDataList

        class BadRequest(api.Response):
            """ Invalid secret uuid and/or version """

        class Forbidden(api.Response):
            """ Secret is not delegated to Sandbox or unavailable to the task author """


class Secrets(Api.Path("/yav/secrets")):

    class Get(api.Post):
        """ List of secrets matching the query for the current user """

        __security__ = ctu.Restriction.AUTHENTICATED
        query = api.String("Search query", default="")
        limit = api.Integer("Page size", default=10)
        page = api.Integer("Number of page", default=0)
        id = api.Array(api.String, "Secret uuids")

        class Ok(api.Response):
            """ List of secrets matching the query """
            schema = schemas.yav.SecretSuggestList

        class BadRequest(api.Response):
            """ Invalid parameters """


class SecretsData(Api.Path("/yav/secrets/data")):

    class Post(api.Post):
        """ Get secret values by secret uuid and an optional version. Task session is required. """

        __security__ = (ctu.Restriction.TASK, ctu.Restriction.EXTERNAL_SESSION)
        data = schemas.yav.SecretAndVersionList(scope=api.Scope.BODY)

        class Ok(api.Response):
            """ List of secret values """
            schema = schemas.yav.SecretDataList

        class BadRequest(api.Response):
            """ Invalid secret uuid and/or version """

        class Forbidden(api.Response):
            """ Secret is not delegated to Sandbox or unavailable to the task author """


class Secret(Api.Path("/yav/secrets/{id}")):

    class Get(api.Get):
        """ Detailed info for the given secret, including the list of its versions """

        __security__ = ctu.Restriction.AUTHENTICATED
        id = api.String("Secret uuid", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.yav.SecretInfo

        class BadRequest(api.Response):
            """ Invalid parameters """
