# coding: utf-8

import time
import json
import types
import urllib
import httplib
import logging

import sandbox.common.types.user as ctu
import sandbox.common.types.misc as ctm
import sandbox.common.types.database as ctd

from sandbox import common

from sandbox.yasandbox import context
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import controller
from sandbox.yasandbox.controller import user as user_controller

import sandbox.yasandbox.api.xmlrpc
import sandbox.yasandbox.api.json.registry
import sandbox.yasandbox.database.mapping.base

import sandbox.serviceapi.web.exceptions

import sandbox.web.helpers
import sandbox.web.response


logger = logging.getLogger(__name__)


registered = {}


class RESTCallWrapper(object):
    """
    This call wrapper is just a lambda with closure emulation to allow pickling of callback to pass it via
    :class:`multiprocessing.Queue`.
    """

    _logger = common.log.LogLazyLoader(common.log.get_server_log, ('rest', ))

    def __init__(self, func, *args):
        self.func = (func.im_self, func.__name__) if isinstance(func, types.MethodType) else func
        self.started = time.time()
        self.args = args

    def __call__(self, request):
        func = self.func
        args = self.args
        func_is_tuple = isinstance(func, tuple)
        method_name = request.remote_method
        handler = getattr(func[0], func[1]) if func_is_tuple else func
        if request.method in (ctm.RequestMethod.GET, ctm.RequestMethod.HEAD):
            if request.read_preference == ctd.ReadPreference.PRIMARY:
                request.read_preference = ctd.ReadPreference.PRIMARY_PREFERRED
        elif request.user != user_controller.User.anonymous or request.user.super_user:
            request.read_preference = ctd.ReadPreference.PRIMARY

        # log all debug info
        self._logger.debug(
            "Request: '%s', login '%s', method: '%s', arguments: %r, session: %r",
            request.id, request.user.login, method_name, args, request.session
        )
        try:
            sandbox.yasandbox.database.mapping.base.tls.request = request
            context.set_current(request.ctx)
            return handler(request, *args)
        except sandbox.web.response.HttpResponseBase as ex:
            return ex
        except Exception as ex:
            self._logger.error(
                "Request: '%s', login '%s', error calling method %r (args %r): %s",
                request.id, request.user.login, method_name, args, ex
            )
            if isinstance(ex, sandbox.serviceapi.web.exceptions.RETRIABLE_EXCEPTIONS):
                if sandbox.yasandbox.database.mapping.is_query_error(ex):
                    http_code = httplib.BAD_REQUEST
                else:
                    http_code = httplib.SERVICE_UNAVAILABLE
            else:
                http_code = httplib.INTERNAL_SERVER_ERROR
            return sandbox.web.response.HttpExceptionResponse(http_code)
        finally:
            ts = int((time.time() - self.started) * 1000)
            request.measures.processing_time = ts
            request.handler = request.remote_method
            context.set_current(None)
            self._logger.info(
                "Request '%s' login '%s' from client '%s' to method '%s' processed in %sms",
                request.id, request.user.login, request.client_address, method_name, ts
            )


def check_restrictions(restrictions, request):
    restrictions = list(common.utils.chain(restrictions))

    # set individual reply for each condition if it's the only one to check
    cond_reply = {
        ctu.Restriction.ANY: (True, (None, None)),  # just a stub
        ctu.Restriction.AUTHENTICATED: (request.is_authenticated, (httplib.UNAUTHORIZED, "Unauthorized")),
        ctu.Restriction.ADMIN: (request.user.super_user, (httplib.FORBIDDEN, "Forbidden")),
        ctu.Restriction.TASK: (request.session, (httplib.FORBIDDEN, "Allowed only with task session token")),
    }

    if any(
        cond_reply[restriction][0]
        for restriction in restrictions
    ):
        return
    code, message = (
        cond_reply[restrictions[0]][1]
        if len(restrictions) == 1 else
        (
            httplib.FORBIDDEN,
            "Your request matches none of these conditions: {}".format(", ".join(map(str.lower, restrictions)))
        )
    )
    return sandbox.web.helpers.response_error(code, message)


def dispatch(path, req):
    if path == "/sandbox/xmlrpc":
        if req.method == ctm.RequestMethod.POST:
            m = sandbox.yasandbox.api.xmlrpc.XMLRPCDispatcher(req)
        else:
            return sandbox.web.helpers.response_error(httplib.METHOD_NOT_ALLOWED, "Method not allowed, post required")
    else:
        # Support for legacy path prefix.
        if path.startswith('/sandbox/'):
            path = path[8:]
        if path.endswith('/'):
            path = path[:-1]
        m = registered.get(path)

    if not m and path.startswith(sandbox.yasandbox.api.json.registry.COMMON_PREFIX):
        path = path[len(sandbox.yasandbox.api.json.registry.COMMON_PREFIX) + 1:]
        pparts = path.split('/', 1)
        if len(pparts) > 1 and pparts[0] in ("json", "v1.0", "v1.1"):
            api_version = pparts[0]
            m = sandbox.yasandbox.api.json.registry.api_docs(pparts[1], api_version)
            if m:
                return m
            # Now we should search for all the registered handlers as regular expressions
            matched = None
            ret = {req.method} if req.method == ctm.RequestMethod.OPTIONS else None
            for e in sandbox.yasandbox.api.json.registry.REGISTERED_JSON:
                match = e.regexp.match(pparts[1])
                if match and (e.version == api_version or not e.version):
                    if ret:
                        ret.add(e.method)
                    elif e.method == req.method:
                        if (
                            e.method != ctm.RequestMethod.GET and not e.allow_ro and
                            controller.Settings.mode() == controller.Settings.OperationMode.READ_ONLY
                        ):
                            return sandbox.web.helpers.response_error(
                                503,
                                json.dumps({
                                    "reason": "Sandbox is in READ_ONLY mode",
                                    "server": common.config.Registry().this.fqdn
                                }),
                                "application/json",
                            )
                        check_restrictions(e.restriction, req)
                        args = [urllib.unquote(arg).decode('utf8') for arg in match.groups()]

                        # requests from web pages have their referer set, unlike these sent directly to API
                        if not req.session and "Referer" not in req.headers:
                            req.source = req.Source.API
                        req.remote_method = '{}.{}'.format(
                            e.handler.im_self.__name__, e.handler.__name__
                        ) if isinstance(e.handler, types.MethodType) else e.handler.__name__

                        return RESTCallWrapper(e.handler, *args)
                    else:
                        matched = e
            if ret:
                return (
                    sandbox.web.helpers.response_error(httplib.NOT_FOUND, "Not Found", "application/json")
                    if len(ret) == 1 else
                    sandbox.web.helpers.response_ok("application/json", headers={"Allow": ','.join(sorted(ret))})
                )
            elif matched:
                logging.warn("Method Not Allowed (path is '%s', entry %r)", path, matched)
                sandbox.web.helpers.response_error(httplib.METHOD_NOT_ALLOWED, "Method Not Allowed")
        else:
            logging.warn("Wrong REST API version '%s' (path is '%s')", pparts[0], path)
    elif req.method == ctm.RequestMethod.OPTIONS:
        logging.error("HTTP method %r is not allowed for path %r", req.method, path)
        return sandbox.web.helpers.response_error(httplib.METHOD_NOT_ALLOWED, "Method Not Allowed")
    elif req.user and req.user.login != controller.User.anonymous.login:
        if 'refresh_params' in req.params:
            controller.User.update_path_params(req.user.login, req.path, req.params)
        cached_params = controller.User.get_path_params(req.user.login, req.path)
        cached_params.update(req.params)
        req.params = cached_params
    return m


def registered_view(path):
    """
        Декоратор для добавления пути в web-интерфейс
        @path: путь в виде строки, не должен начинаться со слеша
    """
    def decorator(func):
        url = '/%s' % (path[:-1] if path.endswith('/') else path)
        registered[url] = func
        return func
    return decorator


@registered_view('http_check')
def http_check(request):
    """
        ручка которую дёргает балансер.
        Если 200 - то бекенд считается рабочим
        если 500 - то отрубается на фермы на 5 - 10 секунд
    """
    # проверяем, что есть доступ в базу
    try:
        conn = mapping.get_connection(ctd.ReadPreference.SECONDARY)
        conn.admin.command("ping")
    except Exception:
        logger.exception("http_check failed")
        sandbox.web.helpers.response_error(500, "")
    else:
        logger.debug("HTTP/DB check successful.")
        sandbox.web.helpers.response_ok()
