import os
import logging
import aniso8601
import traceback
import datetime as dt

import jinja2

from sandbox import common
import sandbox.web.response

ARCADIA_ROOT = reduce(lambda p, _: os.path.dirname(p), xrange(3), os.path.abspath(__file__))
logger = logging.getLogger(__name__)
env = None


def load_template(template_name):
    if common.system.inside_the_binary():
        from library.python import resource

        data = resource.find(template_name)
        if data is not None:
            data = data.decode("utf-8")
    else:
        templates_path = os.path.join(ARCADIA_ROOT, template_name)
        try:
            with open(templates_path, "r") as f:
                data = f.read().decode("utf-8")
        except IOError:
            data = None

    return data


def get_template(template_name):
    global env
    if env is None:
        env = jinja2.Environment(loader=jinja2.FunctionLoader(load_template))
        env.filters["timestamp_delta_format"] = timestamp_delta_format
        env.filters["timestampformat"] = timestampformat
        env.filters["staff_user"] = staff_user
        env.filters["size2str"] = common.utils.size2str
        env.filters["kib2str"] = lambda x: common.utils.size2str(x << 10) if x is not None else "None"
        env.filters["object_id"] = lambda x: str(x)
        env.filters["cutstring"] = cutstring
    try:
        template = env.get_template(template_name)
    except Exception as error:
        logger.exception(error)
        raise common.errors.ViewError("Cannot load template {}: {}\n".format(template_name, traceback.format_exc(100)))
    return template


def redirect_to(redirect_url, local=True, set_content_disposition=False):
    if local:
        redirect_url = '/sandbox/%s' % redirect_url
    raise sandbox.web.response.HttpRedirect(redirect_url, set_content_disposition)


def response_raw(content_type='text/plain', content=''):
    raise sandbox.web.response.HttpResponse(content_type, content)


def response_error(code, msg='http error response raised', content_type='text/html', headers=None):
    raise sandbox.web.response.HttpErrorResponse(code, msg, content_type, headers)


def response_ok(content_type='text/html', content='', headers=None):
    raise sandbox.web.response.HttpResponse(content_type, content, headers=headers)


def send_response(http_response):
    raise http_response


def response_created(location, content=None, content_type=None):
    response = sandbox.web.response.HttpResponse(code=201)
    response.set_header('location', location)
    if content_type:
        response.content_type = content_type
    if content:
        response.content = content
    return send_response(response)


def draw_template(template_name, ctx=None):
    settings = common.config.Registry()
    if ctx is None:
        ctx = {}
    template = get_template(template_name)

    ctx['_hostname'] = settings.server.web.address.host
    ctx['settings'] = settings
    result = template.render(ctx)

    return result


def timestamp_delta_format(delta):
    if delta < 0:
        delta = 0
    hours, remainder = divmod(delta, 3600)
    minutes, seconds = divmod(remainder, 60)
    return '%d:%02d:%02d' % (hours, minutes, seconds)


def timestampformat(timestamp, format='%d.%m.%y, %H:%M:%S'):
    if timestamp == 0:
        return ''
    date_time = dt.datetime.fromtimestamp(timestamp)
    return date_time.strftime(format)


utcdt2iso = common.utils.utcdt2iso


def staff_user(user):
    from sandbox.yasandbox import controller
    if controller.Group.exists(user):
        return '<a href="/sandbox/admin/groups/view?group_name={}">{}</a>'.format(user, user)
    else:
        return '<a class="auto-person-card" lang="{}" href="//staff.yandex-team.ru/{}">{}</a>'.format(user, user, user)


def cutstring(string, line_len):
    if not isinstance(string, unicode):
        string = unicode(string or "", "utf-8", "replace")
    return string[0:min(line_len, len(string))] if string else string


def datetime_couple(arg):
    """ Parses given string as a couple of two strings in ISO 8601 dates separated by `..`. """
    since, to = arg.split('..')
    return aniso8601.parse_datetime(since), aniso8601.parse_datetime(to)
