import sys
import json
import inspect
import logging
import requests
import traceback as tb

import six
from six.moves import http_client as httplib

from sandbox import common

logger = logging.getLogger(__name__)


# note that appropriate response object is raised by helpers (see below)
# and dispatched with try...except (see server.py)
class HttpResponseBase:
    def __init__(self, headers):
        self._headers = headers or {}

    def set_header(self, header_name, header_value):
        self._headers[header_name] = header_value

    @property
    def headers(self):
        return self._headers

    def __encode__(self):
        return self.__class__.__name__, (self._headers,)

    @classmethod
    def __decode__(cls, data):
        class_name, args = data
        cls_ = globals().get(six.ensure_str(class_name))
        if cls_ is None or not inspect.isclass(cls_) or not issubclass(cls_, cls):
            return None
        return cls_(*args)

    def raise_for_status(self):
        pass


class HttpResponse(HttpResponseBase):
    def __init__(self, content_type='text/html', content='', code=200, headers=None):
        HttpResponseBase.__init__(self, headers)
        self.content_type = content_type
        self.content = content
        self.code = code

    def __repr__(self):
        return 'HttpResponse "%s"' % self.content_type

    def json(self):
        return json.loads(self.content)

    def __encode__(self):
        return self.__class__.__name__, (self.content_type, self.content, self.code, self._headers)


class HttpErrorResponse(HttpResponseBase):
    def __init__(self, code, msg='error response raised', content_type='text/html', headers=None):
        HttpResponseBase.__init__(self, headers)
        self.code = code
        self.msg = msg
        self.content_type = content_type
        logger.error("Replying code %s: %s", code, msg)

    @property
    def text(self):
        return self.msg

    @property
    def status_code(self):
        return self.code

    def __repr__(self):
        return "%s (%s: %s)" % (self.__class__.__name__, self.code, self.msg)

    def __encode__(self):
        return self.__class__.__name__, (self.code, self.msg, self.content_type, self._headers)

    def raise_for_status(self):
        """Raises common.rest.Client.HTTPError in a manner requests.Response.raise_for_status does."""

        http_error_msg = u""
        if 400 <= self.code < 500:
            http_error_msg = u"{} Client Error: {}".format(self.code, self.msg)
        elif 500 <= self.code < 600:
            http_error_msg = u"{} Server Error: {}".format(self.code, self.msg)

        if http_error_msg:
            raise common.rest.Client.HTTPError(requests.HTTPError(http_error_msg, response=self))


class HttpExceptionResponse(HttpErrorResponse):
    def __init__(self, code=httplib.INTERNAL_SERVER_ERROR, headers=None, msg=None):
        if msg is None:
            ei = sys.exc_info()
            ex = ei[1]
            msg = json.dumps({
                "reason": str(ex.args[0]) if ex.args else ex.__class__.__name__,
                "traceback": "\n".join(tb.format_exception(*ei)),
                "server": common.config.Registry().this.fqdn,
            })
        HttpErrorResponse.__init__(self, code, msg, "application/json", headers)

    def __encode__(self):
        return self.__class__.__name__, (self.code, self._headers, self.msg)


class HttpRedirect(HttpResponseBase):
    def __init__(self, redirect_url, set_content_disposition=False):
        HttpResponseBase.__init__(self, None)
        self.redirect_url = redirect_url
        self.set_content_disposition = set_content_disposition

    def __repr__(self):
        return 'HttpRedirect to %s' % self.redirect_url

    def __encode__(self):
        return self.__class__.__name__, (self.redirect_url, self.set_content_disposition)


class HTTPError(HttpErrorResponse):
    """ Error class for compatibility with common.rest.Client """

    # noinspection PyMissingConstructor
    def __init__(self, ex):
        self.__source = ex

    def __getattr__(self, item):
        return None if item == "_HTTPError__source" else getattr(self.__source, item)

    @property
    def status(self):
        return self.__source.code
