"""
we need to import all modules here for register urls in registry module
yasandbox.api.json.registry.REGISTERED_JSON must be filled completely after it
"""

import abc
import json
import types
import httplib
import functools as ft
import collections

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt

from sandbox.yasandbox.api.json import registry

__all__ = [
    "task",
    "resource",
    "release",
    "scheduler",
    "statistics",
    "suggest",
    "user",
    "group",
    "vault",
    "batch",
    "misc",
    "client",
    "service",
    "template",
    "authenticate",
    "semaphore"
]


class Base(object):
    """
    """
    __metaclass__ = abc.ABCMeta

    Model = abc.abstractproperty()

    # Query arguments" name mapping to
    # request parameter name, query builder method's argument name, database field name for ordering and
    # type validator method.
    QueryMapping = collections.namedtuple("QueryMapping", ("arg", "kwarg", "mearg", "type"))

    # A list of list operation query parameters mapping.
    LIST_QUERY_MAP = abc.abstractproperty()

    @classmethod
    def request_needs_updated_data(cls, request):
        return ctm.HTTPHeader.WANT_UPDATED_DATA in request.headers

    @abc.abstractproperty
    class ListItemEntry:
        pass

    @abc.abstractproperty
    class Entry:
        pass

    @classmethod
    def _handle_args(cls, request, multi_order=False):
        kwargs = {
            e.kwarg: (sum(map(e.type, v), []) if len(v) > 1 else e.type(v[0]))
            for e, v in (
                (_, [__ for __ in request[_.arg] if __ and __.strip()])
                for _ in cls.LIST_QUERY_MAP if _.arg in request
            ) if v
        }
        limit = kwargs.pop('limit', None)
        offset = kwargs.pop('offset', 0)

        # Do not forget to map ordering field name.
        order_by = kwargs.get("order_by", "")
        orders = []
        for order in order_by.split(","):
            order = order.strip()
            if not order:
                continue

            if order[0] in ("+", "-"):
                sign = order[0]
                order = order[1:]
            else:
                sign = "+"

            if order:
                order = next(
                    (entry.mearg for entry in cls.LIST_QUERY_MAP if entry.mearg and order == entry.arg),
                    None
                ) or order
                orders.append(sign + order)

        if orders:
            if multi_order:
                kwargs["order_by"] = orders
            else:
                if len(orders) > 1:
                    raise ValueError("Only one order field is supported: {}".format(orders))
                kwargs["order_by"] = orders[0]
        else:
            kwargs.pop("order_by", None)

        priority = kwargs.get("priority")
        if priority:
            ctt.Priority.make(priority)
        return kwargs, offset, limit

    @classmethod
    def _id(cls, obj_id):
        try:
            return int(obj_id)
        except ValueError as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Path parameter validation error: " + str(ex))

    @classmethod
    def _document(cls, obj_id, lite=False):
        query = cls.Model.objects
        if lite:
            query = query.lite()
        doc = query.with_id(cls._id(obj_id))
        if not doc:
            return misc.json_error(httplib.NOT_FOUND, "{} #{} not found.".format(cls.__name__, obj_id))
        return doc

    @abc.abstractmethod
    def list(self, request):
        pass

    @classmethod
    def get(cls, request, obj_id):
        return misc.response_json(cls.Entry(request.user, request.uri, cls._document(obj_id)))


class PathBase(object):
    api_path = None
    Model = abc.abstractproperty()

    # noinspection PyPep8Naming
    class __metaclass__(type):
        def __new__(mcs, name, bases, namespace):
            base_path = namespace.pop("base_path", False)
            cls = type.__new__(mcs, name, bases, namespace)
            # noinspection PyUnresolvedReferences
            api_path = cls.api_path
            if api_path is None or base_path:
                return cls
            for request_index, request in enumerate(api_path.requests):
                name = request.__name__.lower()
                method = getattr(cls, name, None)
                path = api_path.path
                assert (
                    method and isinstance(method, types.MethodType) and method.__self__
                ), "{} {} is not implemented".format(name.upper(), path)
                for param in request.__parameters__:
                    if param.scope != common.api.Scope.PATH:
                        continue
                    path = path.replace("{{{}}}".format(param.__param_name__), "({})".format(param.__re__))
                # noinspection PyProtectedMember
                # noinspection PyUnresolvedReferences
                registry.registered_json(
                    path[1:] if path.startswith("/") else path,
                    method=name.upper(),
                    restriction=request.__security__,
                    allow_ro=request.__allow_ro__
                )(ft.wraps(method)(ft.partial(cls._request_parser, method, api_path.path, request_index)))
            return cls

    def __new__(cls, api_path):
        return type(cls)(cls.__name__, (PathBase,), dict(api_path=api_path, base_path=True))

    @classmethod
    def _document(cls, obj_id):
        # noinspection PyUnresolvedReferences
        doc = cls.Model.objects.with_id(obj_id)
        if not doc:
            return misc.json_error(httplib.NOT_FOUND, "{} #{} not found.".format(cls.__name__, obj_id))
        return doc

    @classmethod
    def _request_parser(cls, method, path, request_index, request, *args):
        from yasandbox.api.json import misc
        args_iter = iter(args)
        arguments = []
        query = {}
        body = None
        parameters = cls.api_path.api.__paths__[path].requests[request_index].__parameters__
        for param in parameters:
            if param.scope == common.api.Scope.PATH:
                arguments.append(param.decode(args_iter.next()))
            elif param.scope == common.api.Scope.BODY:
                try:
                    data = json.loads(request.raw_data)
                except ValueError as ex:
                    return misc.json_error(httplib.BAD_REQUEST, "Unable to parse input data: {}".format(ex))
                assert body is None, "Cannot be more then one parameter with scope BODY"
                try:
                    body = param.decode(data)
                except (ValueError, TypeError) as ex:
                    return misc.json_error(httplib.BAD_REQUEST, "Error parsing {}: {}".format(param.__name__, ex))
            elif param.scope == common.api.Scope.QUERY:
                param_name = param.__param_name__
                value = request.get(param_name, None)
                try:
                    query[param_name] = param.decode(value and value.strip())
                except (ValueError, TypeError) as ex:
                    return misc.json_error(httplib.BAD_REQUEST, "Error parsing {}: {}".format(param_name, ex))
            else:
                assert False, "Unsupported scope"
        if body is not None:
            arguments.append(body)
        if query:
            arguments.append(query)
        return method(request, *arguments)


def list_arg_parser(_type):
    def _parser(argument):
        return map(_type, filter(lambda _: _, argument.split(",")))
    return _parser


def priority_parser(priority):
    return priority.upper().split(":")


from yasandbox.api.json import task
from yasandbox.api.json import resource
from yasandbox.api.json import release
from yasandbox.api.json import vault
from yasandbox.api.json import scheduler
from yasandbox.api.json import statistics
from yasandbox.api.json import suggest
from yasandbox.api.json import batch
from yasandbox.api.json import user
from yasandbox.api.json import group
from yasandbox.api.json import misc
from yasandbox.api.json import client
from yasandbox.api.json import service
from yasandbox.api.json import template
from yasandbox.api.json import authenticate
from yasandbox.api.json import semaphore
