import time
import base64
import struct
import logging
import hashlib
import urlparse
import binascii
import collections

import requests

import paramiko.rsakey
import paramiko.dsskey
import paramiko.message

import sandbox.common.types.user as ctu
import sandbox.common.types.misc as ctm

from sandbox import common

from sandbox.yasandbox import controller
from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import registry

import sandbox.web.response


class SSHKey(object):
    @classmethod
    def get_user_keys(cls, login):
        persons = controller.User.staff_api.persons.read(login=login, _fields="keys").get("result")
        return persons[0].get("keys", []) if persons else []

    @classmethod
    def gen_oauth_token(cls, request, login, fingerprint):
        login = login.strip().lower()
        try:
            fingerprint = binascii.unhexlify(fingerprint.strip().lower())
        except (ValueError, TypeError) as ex:
            return misc.json_error(400, "Invalid fingerprint provided: {}".format(ex))
        try:
            cnonce = binascii.unhexlify(request.get("cnonce").strip().lower())
            cnow = struct.unpack("L", cnonce)[0] >> 32
            now = int(time.time())
            # Check the salt has been created recently.
            if cnow < now - 43200 or cnow > now + 43200:
                raise ValueError("Client time does not match current date")
        except (ValueError, TypeError, AttributeError, struct.error) as ex:
            return misc.json_error(400, "Invalid cnonce provided: {}".format(ex))

        keys = cls.get_user_keys(login)
        if not keys:
            return misc.json_error(404, "No SSH keys found for user '{}'".format(login))

        pubkey = None
        for k in keys:
            try:
                # get the second field from the public key file.
                keydata = base64.b64decode(k["key"].split(None)[1])
                msg = paramiko.message.Message(keydata)
                ktype = msg.get_string()
                msg.rewind()

                if ktype == "ssh-rsa":
                    pubkey = paramiko.rsakey.RSAKey(msg=msg)
                elif ktype == "ssh-dss":
                    pubkey = paramiko.dsskey.DSSKey(msg=msg)
                else:
                    raise ValueError("Unknown key format {}".format(ktype))

                if fingerprint == pubkey.get_fingerprint():
                    break
            except Exception as ex:
                logging.error("Error loading user's '%s' key #%s (%s): %s", login, k["id"], k["description"], ex)
                continue

        if not pubkey:
            return misc.json_error(
                404,
                "No user's '{}' public key '{}' found.".format(login, binascii.hexlify(fingerprint))
            )

        h = hashlib.sha1()
        h.update(login)
        h.update(fingerprint)
        h.update(cnonce)

        if not pubkey.verify_ssh_sig(h.digest(), paramiko.message.Message(request.raw_data)):
            return misc.json_error(
                400,
                "Provided signature of {} bytes length cannot verify user's '{}' "
                "digest '{}' with key '{}' and cnonce '{}'.".format(
                    len(request.raw_data), login,
                    h.hexdigest(), binascii.hexlify(fingerprint), binascii.hexlify(cnonce)
                )
            )
        token = controller.OAuthCache.refresh(login, source=ctu.TokenSource.SSH_KEY).token
        return sandbox.web.response.HttpResponse(code=201, content=token, content_type="text/plain")


registry.registered_json("authenticate/ssh-key/([-\w]+)/(\w+)", method=ctm.RequestMethod.POST)(SSHKey.gen_oauth_token)


class OAuth(object):
    __client_token_cache = None

    @common.utils.classproperty
    def __client_token(cls):
        if cls.__client_token_cache is None:
            settings = common.config.Registry()
            cls.__client_token_cache = (
                common.utils.read_settings_value_from_file(settings.server.auth.oauth.client_config)
                if settings.server.auth.enabled else
                ':'
            )
        return collections.namedtuple('ClientToken', ('id', 'secret'))(*cls.__client_token_cache.split(':'))

    @classmethod
    def _request_token(cls, url, data):
        url += '/token'
        data['client_id'], data['client_secret'] = cls.__client_token
        headers = {
            'Host': urlparse.urlparse(url).hostname,
            'Content-type': 'application/x-www-form-urlencoded',
        }
        resp = requests.post(url, data=data, headers=headers).json()
        token = resp.get('access_token')
        if not token:
            raise Exception('OAuth server returned no token: {!r}'.format(resp))
        return token

    @classmethod
    def client(cls, request):
        return misc.response_json({
            "client": cls.__client_token.id,
            "url": common.config.Registry().client.auth.oauth_url,
        })

    @classmethod
    def generate(cls, request):
        settings = common.config.Registry()
        data = misc.request_data(request)

        if not settings.server.auth.enabled:
            token = "d41d8cd98f00b204e9800998ecf8427e"  # empty MD5 checksum
        elif "login" in data and "password" in data:
            login, passwd = map(data.get, ("login", "password"))
            logging.debug("User '%s' requested OAuth token via password authentication.", login)
            try:
                token = cls._request_token(
                    settings.client.auth.oauth_url,
                    {'grant_type': 'password', 'username': login, 'password': passwd},
                )
            except Exception as ex:
                return misc.json_error(400, "OAuth server returned no token: " + str(ex))
        else:
            try:
                code = data["code"]
                logging.debug("User '%s' grant OAuth token via code '%s'.", code)
                token = cls._request_token(
                    settings.client.auth.oauth_url,
                    {'grant_type': 'authorization_code', 'code': code},
                )
            except Exception as ex:
                return misc.json_error(400, "OAuth server returned no token: " + str(ex))

            logging.info("%r OAuth token: %r", request.user.login, token)
        return sandbox.web.response.HttpResponse(code=201, content=token, content_type="text/plain")

registry.registered_json("authenticate/oauth/client")(OAuth.client)
registry.registered_json("authenticate/oauth/token", method=ctm.RequestMethod.POST)(OAuth.generate)
