""" The file encapsulates all the functionality handlers related to "/batch" prefixed REST API resources. """

import logging
import datetime as dt
import itertools as it
import collections

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.user as ctu
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc

from sandbox.yasandbox.proxy import resource as resource_proxy

import sandbox.yasandbox.manager
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import context

from sandbox.serviceapi.web import exceptions

import sandbox.web.response

from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import registry
from sandbox.yasandbox.api.json import resource as resource_api
from sandbox.yasandbox.api.json import scheduler as scheduler_api


class BatchResult(common.patterns.Abstract):
    Status = ctm.BatchResultStatus

    __slots__ = ("status", "message", "id")
    __defs__ = (None,) * 3


class BatchOperation(object):
    Messages = collections.namedtuple("Message", ("success", "failure"))

    @staticmethod
    def _handle_args(request, id_type=int):
        data = misc.request_data(request, (dict, list, basestring, id_type))
        ids, comment = map(data.get, ("id", "comment")) if isinstance(data, dict) else (data, None)
        try:
            ids = map(id_type, common.utils.chain(ids))
        except (ValueError, TypeError) as ex:
            return misc.json_error(400, "Unable to parse input data: " + str(ex))
        return ids, comment

    @staticmethod
    def _log_operation(event, objtype, objects, comment=None):
        logging.info("Batch operation %r on %s %r; comment: %r", event, objtype, objects, comment)


class BatchTask(BatchOperation):
    """ Batch operations on tasks objects. """

    FAILED_STATUSES = tuple(ctt.Status.Group.BREAK) + (ctt.Status.FAILURE,)

    @classmethod
    def _op(cls, request, mname, event, messages, obj_ids):
        cls._log_operation(mname, "tasks", obj_ids, event)
        getattr(controller.TaskWrapper, mname)
        ret = []
        sync = request.headers.get(ctm.HTTPHeader.COMPONENT) != ctm.Component.SERVICE
        for i in obj_ids:
            try:
                task = controller.TaskWrapper(controller.Task.get(i))

                if mname in ("suspend", "resume") and task.author_mismatch():
                    ret.append(BatchResult(
                        BatchResult.Status.ERROR,
                        "Cannot {} yav-enabled task {} with author {}: "
                        "it is not created by you ({}) or a robot you own".format(
                            mname, task.model.id, task.model.author, request.user.login
                        ),
                        i
                    ))
                    continue

                # a special case for tasks being started either for the first time,
                # or after all their dependencies are met (that is, WAIT_*).
                # we also allow users to forcibly wake tasks up from WAIT_TIME: SANDBOX-4618

                # this is a "fresh start"; on real restart (another branch), TaskWrapper.restart() is called,
                # which resets task's resources, calling sandbox.yasandbox.manager.task.TaskManager.restart_task()
                if (
                    mname == "restart" and
                    task.status in ctt.Status.Group.WAIT + ctt.Status.Group.DRAFT
                ):
                    # intended to be used from TaskStateSwitcher
                    if (
                        task.status in (ctt.Status.WAIT_OUT, ctt.Status.WAIT_TASK, ctt.Status.WAIT_RES) and
                        not request.user.super_user
                    ):
                        ret.append(BatchResult(
                            BatchResult.Status.WARNING,
                            "You must be an admin to start a task in '{}' status".format(task.status),
                            i
                        ))
                        continue

                    priority = task.model.priority
                    allowed_priority = controller.Group.allowed_priority(request, task.model.owner)
                    message = event
                    success_message = messages.success
                    op_status = BatchResult.Status.SUCCESS

                    if priority > int(allowed_priority):
                        message = "Task priority {!r} lowered to {!r} for owner {!r}".format(
                            ctt.Priority.make(priority), allowed_priority, task.model.owner
                        )
                        success_message = " ".join((messages.success, message))
                        op_status = BatchResult.Status.WARNING
                        task.model.priority = int(allowed_priority)
                        task.model.save()

                    try:
                        if controller.TaskQueue.enqueue_task(task, message, sync=sync):
                            if task.model.execution.status != ctt.Status.WAIT_TIME:
                                try:
                                    controller.TimeTrigger.get(i).delete()
                                except controller.TimeTrigger.NotExists:
                                    pass
                                else:
                                    logging.info("Time trigger for task #%s removed", i)

                            if task.model.execution.status in cls.FAILED_STATUSES:
                                ret.append(
                                    BatchResult(BatchResult.Status.ERROR, "Task #{} was not enqueued".format(i), i)
                                )
                            else:
                                ret.append(BatchResult(op_status, success_message, i))
                        else:
                            msg = "Failed to enqueue the task due to race condition"
                            ret.append(BatchResult(BatchResult.Status.WARNING, msg, i))
                    except common.errors.IncorrectStatus as ex:
                        ret.append(BatchResult(BatchResult.Status.ERROR, str(ex), i))

                else:
                    ret.append(
                        BatchResult(BatchResult.Status.SUCCESS, messages.success, i)
                        if getattr(task, mname)(event=event) else
                        BatchResult(
                            BatchResult.Status.WARNING,
                            messages.failure.format(task.id, task.model.execution.status),
                            i
                        )
                    )
            except exceptions.RETRIABLE_EXCEPTIONS:
                raise
            except Exception as ex:
                logging.exception("Batch %r operation error on task #%s: %s", mname, i, str(ex))
                ret.append(BatchResult(BatchResult.Status.ERROR, str(ex), i))

        if mname == "restart":
            mapping.Task.objects(id__in=[_.id for _ in ret if _.status != BatchResult.Status.ERROR]).update(
                set__execution__disk_usage__max=0,
                set__execution__disk_usage__last=0
            )
        return misc.response_json(map(dict, ret))

    @classmethod
    @context.timer_decorator("task_start", reset=True, loglevel=logging.INFO)
    def start(cls, request):
        ids, comment = cls._handle_args(request)
        comment = comment or ("Batch start" if len(ids) > 1 else "Task start")
        return cls._op(
            request, "restart", comment,
            cls.Messages("Task started successfully.", "Task #{} in status {} cannot be started."),
            ids,
        )

    @classmethod
    def stop(cls, request):
        ids, comment = cls._handle_args(request)
        comment = comment or ("Batch stop" if len(ids) > 1 else "Task stop")
        return cls._op(
            request, "stop", comment,
            cls.Messages("Task scheduled for stop.", "Task #{} in status {} cannot be stopped."),
            ids,
        )

    @classmethod
    def delete(cls, request):
        ids, comment = cls._handle_args(request)
        comment = comment or ("Batch delete" if len(ids) > 1 else "Task delete")
        return cls._op(
            request, "delete", comment,
            cls.Messages("Task deleted.", "Task #{} in status {} cannot be deleted."),
            ids,
        )

    @classmethod
    def suspend(cls, request):
        ids, comment = cls._handle_args(request)
        comment = comment or ("Batch suspend" if len(ids) > 1 else "Task suspend")
        return cls._op(
            request, "suspend", comment,
            cls.Messages("Task scheduled for suspending.", "Task #{} in status {} cannot be suspended."),
            ids,
        )

    @classmethod
    def resume(cls, request):
        ids, comment = cls._handle_args(request)
        comment = comment or ("Batch resume" if len(ids) > 1 else "Task resume")
        return cls._op(
            request, "resume", comment,
            cls.Messages("Task scheduled for resuming.", "Task #{} in status {} cannot be resumed."),
            ids,
        )

    @classmethod
    def expire(cls, request):
        ids, comment = cls._handle_args(request)
        comment = comment or ("Batch expire" if len(ids) > 1 else "Task expire")
        return cls._op(
            request, "expire", comment,
            cls.Messages("Task has been expired.", "Task #{} in status {} cannot be expired."),
            ids,
        )

    @classmethod
    def increase_priority(cls, request):
        ret = []
        ids, comment = cls._handle_args(request, int)
        cls._log_operation("increase_priority", "tasks", ids, comment)
        for i in ids:
            try:
                task = controller.TaskWrapper(controller.Task.get(i))
                priority = ctt.Priority.make(task.model.priority).next
                controller.TaskQueue.set_priority(request, task, priority)
                ret.append(
                    BatchResult(BatchResult.Status.SUCCESS, "Task's priority has been increased", i)
                )
            except ValueError as ex:
                ret.append(BatchResult(BatchResult.Status.WARNING, str(ex), i))
            except Exception as ex:
                ret.append(BatchResult(BatchResult.Status.ERROR, str(ex), i))

        return misc.response_json(map(dict, ret))


class Resource(BatchOperation):
    """ Batch operations on resource objects. """
    API = resource_api.Resource
    Proxy = resource_proxy.Resource

    @classmethod
    def _op(cls, request, mname, messages):
        method = getattr(cls.Proxy, mname)

        ret = []
        ids, comment = cls._handle_args(request)
        cls._log_operation(mname, "resources", ids, comment)
        for i in ids:
            try:
                resource = cls.Proxy._restore(cls.API.Model.objects.with_id(cls.API._id(i)))
                if not resource:
                    raise Exception("Resource #{} not found.".format(i))
                if not resource.user_has_permission(request.user):
                    raise Exception("You have no access to modify resource #{}".format(i))
                ret.append(
                    BatchResult(BatchResult.Status.SUCCESS, messages.success, i)
                    if method(resource) else
                    BatchResult(BatchResult.Status.WARNING, messages.failure, i))
            except Exception as ex:
                logging.error("Batch %r operation error on resource #%s: %s", mname, i, str(ex))
                ret.append(BatchResult(BatchResult.Status.ERROR, str(ex), i))

        return misc.response_json(map(dict, ret))

    @classmethod
    def restore(cls, request):
        return cls._op(
            request, "restore", cls.Messages("Resource scheduled for restore.", "Resource cannot be restored.")
        )

    @classmethod
    def delete(cls, request):
        resource_manager = sandbox.yasandbox.manager.resource_manager
        ret = []
        messages = cls.Messages('The resource is successfully deleted.', 'The resource cannot be deleted.')
        ids, comment = cls._handle_args(request)
        cls._log_operation("delete", "resources", ids, comment)
        for i in ids:
            try:
                resource = resource_manager.load(cls.API._id(i))
                if not resource:
                    raise Exception("Resource #{} not found.".format(i))
                if not resource.user_has_permission(request.user):
                    raise Exception("You have no access to modify resource #{}".format(i))
                result = resource_manager.delete_resource(resource.id, ignore_last_usage_time=True)
                ret.append(
                    BatchResult(BatchResult.Status.SUCCESS, messages.success, i)
                    if result is None else
                    BatchResult(BatchResult.Status.WARNING, result, i)
                )
            except Exception as ex:
                ret.append(BatchResult(BatchResult.Status.ERROR, str(ex), i))

        return misc.response_json(map(dict, ret))

    @classmethod
    def _mark_flag(cls, request, fname, fvalue, messages, override=True):
        ret = []
        ids, comment = cls._handle_args(request)
        cls._log_operation((fname, fvalue), "resources", ids, comment)
        for i in ids:
            try:
                resource = cls.Proxy._restore(cls.API.Model.objects.with_id(cls.API._id(i)))
                if not resource:
                    raise Exception("Resource #{} not found.".format(i))
                if not resource.user_has_permission(request.user):
                    raise Exception("You have no access to modify resource #{}".format(i))
                if override or fname not in resource.attrs:
                    resource.attrs[fname] = str(fvalue)
                    resource._update()
                ret.append(BatchResult(BatchResult.Status.SUCCESS, messages.success, i))
            except Exception as ex:
                ret.append(BatchResult(BatchResult.Status.ERROR, str(ex), i))
        return misc.response_json(map(dict, ret))

    @classmethod
    def backup(cls, request):
        return cls._mark_flag(
            request, "backup_task", True,
            cls.Messages(
                "Resource successfully marked for backup.",
                "Resource cannot be marked for backup."
            ),
            override=False
        )

    @classmethod
    def do_not_remove(cls, request):
        return cls._mark_flag(request, "ttl", "inf", cls.Messages(
            "Resource's TTL is set to infinite'.", "Resource's TTL cannot be set to infinite."))

    @classmethod
    def touch(cls, request):
        ids, comment = cls._handle_args(request)
        cls._log_operation("touch", "resources", ids, comment)
        updated_count = cls.Proxy.Model.objects(id__in=ids).update(set__time__accessed=dt.datetime.utcnow())

        not_existing_ids = set()
        if updated_count != len(ids):
            not_existing_ids = set(ids) - set(cls.Proxy.Model.objects(id__in=ids).scalar("id"))

        response = list()
        for rid in ids:
            if rid in not_existing_ids:
                response.append(BatchResult(BatchResult.Status.ERROR, "Resource #{} not found.".format(rid), rid))
            else:
                response.append(BatchResult(BatchResult.Status.SUCCESS, "Resource updated", rid))

        return misc.response_json(map(dict, response))


class Client(BatchOperation):
    """ Batch operations on clients objects. """

    @classmethod
    def op(cls, request, kind):
        ret = []
        ids, comment = cls._handle_args(request, str)
        cls._log_operation(kind, "clients", ids, comment)
        for i in ids:
            try:
                client = mapping.Client.objects.with_id(i)
                if not client or not client.arch:
                    ret.append(BatchResult(BatchResult.Status.ERROR, "Client not found", i))
                    continue

                if not request.user.super_user and request.user.login not in client.info.get("owners", []):
                    ret.append(
                        BatchResult(
                            BatchResult.Status.ERROR,
                            "Operation is not permitted for user {}".format(request.user.login),
                            i
                        )
                    )
                    continue

                controller.Client.reload(client, kind, request.user.login, comment)
                ret.append(BatchResult(BatchResult.Status.SUCCESS, "Operation scheduled successfully", i))
            except Exception as ex:
                ret.append(BatchResult(BatchResult.Status.ERROR, str(ex), i))

        return misc.response_json(map(dict, ret))

    @classmethod
    def reboot(cls, request):
        return cls.op(request, ctc.ReloadCommand.REBOOT)

    @classmethod
    def reload(cls, request):
        return cls.op(request, ctc.ReloadCommand.RESTART)

    @classmethod
    def shutdown(cls, request):
        return cls.op(request, ctc.ReloadCommand.SHUTDOWN)

    @classmethod
    def cleanup(cls, request):
        return cls.op(request, ctc.ReloadCommand.CLEANUP)

    @classmethod
    def poweroff(cls, request):
        return cls.op(request, ctc.ReloadCommand.POWEROFF)


class Scheduler(BatchOperation):
    """ Batch operations on scheduler objects. """

    @classmethod
    def _op(cls, request, method_name, messages):
        ids, comment = cls._handle_args(request)
        cls._log_operation(method_name, "schedulers", ids, comment)
        method = getattr(controller.Scheduler, method_name)

        ret = []
        for sch_id in ids:
            try:
                sch = scheduler_api.Scheduler._document(sch_id)
                if not controller.user_has_permission(request.user, [sch.owner]):
                    raise Exception(
                        "User {} has no permission to manage scheduler #{}".format(request.user.login, sch_id)
                    )
                ret.append(
                    BatchResult(BatchResult.Status.SUCCESS, messages.success, sch_id)
                    if method(sch) else
                    BatchResult(BatchResult.Status.WARNING, messages.failure, sch_id)
                )
            except (Exception, sandbox.web.response.HttpResponseBase) as ex:
                logging.error("Batch %r operation error on scheduler #%s: %s", method_name, sch_id, str(ex))
                ret.append(BatchResult(BatchResult.Status.ERROR, str(ex), sch_id))

        return misc.response_json(map(dict, ret))

    @classmethod
    def start(cls, request):
        return cls._op(
            request, "restart", cls.Messages("Scheduler started successfully.", "Scheduler cannot be started."),
        )

    @classmethod
    def stop(cls, request):
        return cls._op(
            request, "stop", cls.Messages("Scheduler stopped.", "Scheduler cannot be stopped."),
        )

    @classmethod
    def delete(cls, request):
        return cls._op(
            request, "delete", cls.Messages("Scheduler deleted.", "Scheduler cannot be deleted."),
        )


class Bucket(BatchOperation):
    """ Batch operations on bucket resources """

    @classmethod
    def cleanup(cls, request):
        ids, comment = cls._handle_args(request, str)
        cls._log_operation("cleanup", "clients", ids, comment)
        ret = []
        for bucket_id in ids:
            try:
                bucket_stats = common.mds.S3.bucket_stats(bucket_id)
                if bucket_stats is None:
                    ret.append(BatchResult(BatchResult.Status.ERROR, "Bucket not found", bucket_id))
                    continue
                abc_id = bucket_id.replace("sandbox-", "")  # TODO: use bucket object after (SANDBOX-8766)
                responsible_users = it.chain(
                    common.abc.users_by_service_and_role(abc_id, [common.abc.RESOURCE_MANAGER]),
                    common.abc.service_responsibles(abc_id)
                )
                if not request.user.super_user and request.user.login not in responsible_users:
                    ret.append(
                        BatchResult(
                            BatchResult.Status.ERROR,
                            "Operation is not permitted for user {}".format(request.user.login),
                            bucket_id
                        )
                    )
                    continue
                controller.Resource.mark_force_cleanup_mds_bucket_resources(bucket_id)
                ret.append(BatchResult(BatchResult.Status.SUCCESS, "Operation scheduled successfully", bucket_id))
            except Exception as ex:
                ret.append(BatchResult(BatchResult.Status.ERROR, str(ex), bucket_id))
        return misc.response_json(map(dict, ret))


registry.registered_json("batch/tasks/start", method=ctm.RequestMethod.PUT)(BatchTask.start)
registry.registered_json("batch/tasks/stop", method=ctm.RequestMethod.PUT)(BatchTask.stop)
registry.registered_json("batch/tasks/delete", method=ctm.RequestMethod.PUT)(BatchTask.delete)
registry.registered_json("batch/tasks/suspend", method=ctm.RequestMethod.PUT)(BatchTask.suspend)
registry.registered_json("batch/tasks/resume", method=ctm.RequestMethod.PUT)(BatchTask.resume)
registry.registered_json("batch/tasks/expire", method=ctm.RequestMethod.PUT)(BatchTask.expire)
registry.registered_json("batch/tasks/increase_priority", method=ctm.RequestMethod.PUT)(BatchTask.increase_priority)

registry.registered_json(
    "batch/clients/reboot", method=ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED, allow_ro=True
)(Client.reboot)
registry.registered_json(
    "batch/clients/reload", method=ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED, allow_ro=True
)(Client.reload)
registry.registered_json(
    "batch/clients/shutdown", method=ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED, allow_ro=True
)(Client.shutdown)
registry.registered_json(
    "batch/clients/cleanup", method=ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED, allow_ro=True
)(Client.cleanup)
registry.registered_json(
    "batch/clients/poweroff", method=ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED, allow_ro=True
)(Client.poweroff)

registry.registered_json("batch/resources/restore", method=ctm.RequestMethod.PUT)(Resource.restore)
registry.registered_json("batch/resources/delete", method=ctm.RequestMethod.PUT)(Resource.delete)
registry.registered_json("batch/resources/backup", method=ctm.RequestMethod.PUT)(Resource.backup)
registry.registered_json("batch/resources/do_not_remove", method=ctm.RequestMethod.PUT)(Resource.do_not_remove)
registry.registered_json(
    "batch/resources/touch", method=ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED
)(Resource.touch)

registry.registered_json(
    "batch/schedulers/start", method=ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED
)(Scheduler.start)
registry.registered_json(
    "batch/schedulers/stop", method=ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED
)(Scheduler.stop)
registry.registered_json(
    "batch/schedulers/delete", method=ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED
)(Scheduler.delete)

registry.registered_json(
    "batch/buckets/cleanup", method=ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED
)(Bucket.cleanup)
