import logging
import httplib
import distutils.util
import datetime as dt
import itertools as it

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.user as ctu
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

from sandbox.yasandbox.api.json import Base
from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import registry
from sandbox.yasandbox.api.json import list_arg_parser

import sandbox.web.helpers
import sandbox.web.response


###########################################################
# API Version 1.0
###########################################################
class Client(Base):
    """
    The class encapsulates all the logic related to REST API representation of any entities related to client object.
    """

    # Shortcuts for database models.
    Model = mapping.Client
    TaskModel = mapping.Task

    logger = logging.getLogger("RESTAPI_Client")

    # Drop sessions older than 5 minutes which are not repored by a client
    STALE_SESSION_AGE = 5

    LIST_QUERY_MAP = (
        Base.QueryMapping('id', 'hostname', 'id', list_arg_parser(str)),
        Base.QueryMapping('platform', 'platform', None, str),
        Base.QueryMapping('alive', 'alive', None, distutils.util.strtobool),
        Base.QueryMapping('alive_offset', 'alive_offset', None, int),
        Base.QueryMapping('tags', 'tags', None, str),
        Base.QueryMapping('limit', 'limit', None, int),
        Base.QueryMapping('offset', 'offset', None, int),
    )

    class Entity(dict):

        def __init__(self, base_url, client, tasks):
            context_info = client.info
            system_info = context_info.get('system', {})
            net_unreach_ts = context_info.get('net_unreach_ts')
            super(Client.Entity, self).__init__({
                'uuid': client['uuid'],
                'disk': {
                    'status': system_info.get('disk_status', ctc.DiskStatus.OK),
                    'total_space': system_info.get('total_space', 0) << 20,
                    'free_space': system_info.get('free_space', 0) << 20,
                },
                'tags': list(client.mapping().pure_tags),
                'alive': client.is_alive(),
                'availability': client.get_availability(),
                'id': client.hostname,
                'dc': system_info.get('dc'),
                'lxc': client.lxc,
                'platform': common.platform.get_platform_alias(system_info.get('platform')),
                'platforms': common.platform.LXC_PLATFORMS if client.lxc else None,
                'os': {
                    'name': system_info.get('arch'),
                    'version': system_info.get('os_version'),
                },
                'ram': client.ram << 20,
                'ncpu': client.ncpu,
                'cpu': client.model,
                'fqdn': client.fqdn,
                'fileserver': system_info.get('fileserver'),
                'url': '{}/client/{}'.format(base_url, client.hostname),
                'msg': context_info.get('msg', ''),
                'last_activity': sandbox.web.helpers.utcdt2iso(dt.datetime.utcfromtimestamp(client.update_ts)),
                'net_unreachable': net_unreach_ts and sandbox.web.helpers.utcdt2iso(
                    dt.datetime.utcfromtimestamp(net_unreach_ts)
                ),
                'pending_commands': sorted(client.pending_service_commands()),
            })
            if tasks:
                self.update(
                    task={  # TODO: remove when GUI is not required for it
                        "task_id": tasks[0].id,
                        "owner": tasks[0].owner,
                        "task_type": tasks[0].type,
                        "url": "{}/task/{}".format(base_url, tasks[0].id)
                    },
                    tasks=[
                        {
                            "id": task.id,
                            "owner": task.owner,
                            "type": task.type,
                            "url": "{}/task/{}".format(base_url, task.id)
                        }
                        for task in tasks
                    ]
                )

    @common.utils.singleton_classproperty
    def tags_owners(self):
        _tags_owners = common.config.Registry().server.tags_owners
        return {
            owner: tags
            for owners, tags in it.imap(lambda _: (_["owners"], set(_["tags"])), _tags_owners)
            for owner in owners
        }

    @common.utils.singleton_classproperty
    def __local_mode(self):
        return common.config.Registry().common.installation == ctm.Installation.LOCAL

    @common.utils.singleton_classproperty
    def __local_or_test_mode(self):
        return common.config.Registry().common.installation in ctm.Installation.Group.LOCAL

    @classmethod
    def _update_data(cls, data):
        try:
            os = data["os"]
            disk = data["disk"]
            slots = data.get("slots", {})
            update = {
                "system": {
                    "arch": os["name"],
                    "os_version": os["version"],
                    "physmem": data["ram"],
                    "cpu_model": data["cpu"],
                    "ncpu": data["ncpu"],
                    "total_space": disk["total_space"] >> 20,
                    "free_space": disk["free_space"] >> 20,
                    "used_space": 100 - float(disk["free_space"]) / disk["total_space"] * 100,
                    "used_space_value": (disk["total_space"] - disk["free_space"]) >> 20,
                    "reserved_space": disk.get("locked_space", 0) >> 20,
                    "disk_status": disk["status"],
                    "platform": data["platform"],
                    "lxc": bool(data.get("lxc")),
                    "porto": bool(data.get("porto")),
                    "root": data["root"],
                    "fqdn": data["fqdn"],
                    "dc": data["dc"],
                    "fileserver": data["fileserver"],
                    "tasks_dir": data["tasks_dir"],
                    "total_slots": slots.get("total"),
                    "used_slots": slots.get("used"),
                },
                "revisions": {
                    _: data.get(_) for _ in ("sdk", "client", "tasks", "venv", "server")
                }
            }
            age = data["age"]
        except (KeyError, AttributeError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Error on update: {!r}. Data: {!r}".format(ex, data))
        return update, age

    @classmethod
    def _reset_data(cls, data):
        update, age = cls._update_data(data)
        try:
            tags = set(data["tags"])
            cuuid = data["uuid"]
            assert len(cuuid) == 32, "UUID should be 32 symbols length"
            update["system"]["inetfs"] = [ctm.Network.Family.V6] + (
                [ctm.Network.Family.V4] if ctc.Tag.IPV4 in tags else []
            )
        except (KeyError, TypeError, AttributeError, AssertionError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Error on update: {!r}. Data: {!r}".format(ex, data))
        return update, age, cuuid, tags

    @classmethod
    def reset(cls, request, client_id):
        client = controller.Client.get(client_id, create=True)
        update, age, cuuid, tags = cls._reset_data(misc.request_data(request))
        client.info["revisions"] = update.pop("revisions")

        old_uuid = client.info.get("uuid")
        new = old_uuid != cuuid and not cls.__local_mode

        if new:
            cls.logger.info("REGISTERING client %r v%s.", client_id, age)
            if old_uuid:
                cls.logger.warning("UUID for client %r was changed from %r to %r", client_id, old_uuid, cuuid)

            dup = next((_ for _ in controller.Client.list() if _.info.get("uuid") == cuuid), None)
            if dup:
                return misc.json_error(
                    httplib.BAD_REQUEST,
                    "UUID {!r} is already assigned for client {!r}".format(cuuid, dup.hostname)
                )

            client.info["uuid"] = cuuid
            if not old_uuid:
                tags.add(ctc.Tag.NEW)

            cls.logger.info("%r is probably a new client, running CLEANUP_2 task...", client_id)
            controller.TaskQueue.HostsCache()
            params = dict(
                run_in_dry_mode=False,
                force_update_tests_data=False,
                semaphore="sandbox/cleanup/new"
            )
            # use non-binary task in tests: creating binary in tests is overhead
            if cls.__local_or_test_mode:
                params["binary_executor_release_type"] = "none"
            controller.Task.create_service_task(
                description="Cleanup new host {}".format(client_id),
                task_type="CLEANUP_2",
                requirements=dict(host=client_id),
                priority=ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.HIGH),
                parameters=params
            )

        else:
            cls.logger.info("Resetting client %r v%s.", client_id, age)
            controller.Client.reset_host_tasks(cls.logger, [client_id])

        now = dt.datetime.utcnow()
        client.info["idle"] = {"since": now, "last_update": now}
        controller.Client.update(client, update)

        if tags is not None:
            cls.logger.debug("Setting tags %r for client %r", tags, client_id)
            controller.Client.update_tags(client, tags, controller.Client.TagsOp.SET)

        return sandbox.web.response.HttpResponse(code=httplib.CREATED if new else httplib.NO_CONTENT)

    @classmethod
    def reset_many(cls, request):
        clients = misc.request_data(request, expects=list, expects_each=basestring)
        cls.logger.info("Resetting clients %r", clients)
        restarted = controller.Client.reset_host_tasks(cls.logger, clients)
        return misc.response_json([{"id": t.id, "host": t.host} for t in restarted])

    @classmethod
    def update_comment(cls, request, client_id):
        client = mapping.Client.objects.with_id(client_id)
        if not request.user.super_user:
            return misc.json_error(
                httplib.FORBIDDEN,
                "User '{}' is not permitted to update comments".format(request.user.login)
            )
        if client is None:
            return misc.json_error(httplib.NOT_FOUND, "Client with id '{}' not found".format(client_id))
        try:
            update_data = request.raw_data.decode('utf8')
        except ValueError:
            return misc.json_error(httplib.BAD_REQUEST, "Invalid data: {!r}".format(request.raw_data))

        client.info['msg'] = update_data
        controller.Client.update(client)

        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def update_tags(cls, request, client_id):
        client = mapping.Client.objects.with_id(client_id)
        if client is None:
            return misc.json_error(httplib.NOT_FOUND, "Client with id '{}' not found".format(client_id))
        tags = cls.tags_owners.get(request.user.login)
        if not request.user.super_user and (not tags or not tags & client.tags_set):
            return misc.json_error(
                httplib.FORBIDDEN,
                "User '{}' is not permitted to update tags in '{}'".format(request.user.login, client_id)
            )

        def is_tag_allowed(tag):
            return any((
                tag in ctc.Tag.Group.CUSTOM,
                tag in ctc.Tag.Group.USER,
                request.user.super_user and tag in ctc.Tag.Group.SERVICE,
            ))

        new_tags = set(it.ifilter(
            is_tag_allowed,
            it.imap(
                str,
                common.utils.chain(misc.request_data(request, (list, basestring)))
            )
        ))
        current_tags = set(it.ifilter(is_tag_allowed, client.tags_set))
        controller.Client.update_tags(client, new_tags - current_tags, controller.Client.TagsOp.ADD)
        controller.Client.update_tags(client, current_tags - new_tags, controller.Client.TagsOp.REMOVE)
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def known_resources(cls, request, client_id):
        rc = controller.Resource
        limit = request.get("limit")
        kind = request.get("kind")
        try:
            limit = int(limit) if limit else None
        except (ValueError, TypeError):
            return misc.json_error(httplib.BAD_REQUEST, "Invalid limit value: {!r}".format(limit))

        if kind == ctc.RemovableResources.DELETED:
            ret = [_.id for _ in rc.resources_to_remove(client_id, limit=limit, replicated=False)]
        elif kind == ctc.RemovableResources.EXTRA:
            ret = [_.id for _ in rc.extra_resources_to_drop(client_id, limit=limit)]
        elif kind == ctc.RemovableResources.REPLICATED:
            ret = [_.id for _ in rc.resources_to_remove(client_id, limit=limit, replicated=True)]
        else:
            query = mapping.Resource.objects(
                hosts_states__host=client_id
            ).order_by("+id").scalar("id")
            ret = list(query if not limit else query.limit(limit))
        return misc.response_json(ret)


registry.registered_json(
    "client/([\w\-]+)/service/resources", restriction=ctu.Restriction.ADMIN
)(Client.known_resources)
registry.registered_json("client/([\w\-]+)/comment", ctm.RequestMethod.PUT)(Client.update_comment)
registry.registered_json("client/([\w\-]+)/tags", ctm.RequestMethod.PUT)(Client.update_tags)
registry.registered_json("client", ctm.RequestMethod.POST, ctu.Restriction.ADMIN)(Client.reset_many)
registry.registered_json("client/([\w\-]+)", ctm.RequestMethod.POST, ctu.Restriction.ADMIN)(Client.reset)
