import re
import json
import itertools as it
import mongoengine as me

from six.moves import http_client as httplib

import sandbox.common.types.user as ctu
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm

from sandbox import common

import sandbox.yasandbox.controller
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.controller import user as user_controller

import sandbox.web.helpers
import sandbox.web.response

from sandbox.yasandbox.api.json import Base
from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import registry

import sandbox.serviceapi.handlers.common


###########################################################
# API Version 1.0
###########################################################
class Group(Base):
    """
    The class encapsulates all the logic related to REST API representation of any entities related to resource object.
    """

    MAX_USER_TAG_COUNT = 5  # Max number of client tags for group

    Model = mapping.Group
    USER_TAG_RE = re.compile("\AUSER_[A-Z0-9_]+\Z")  # Regular expression to check user tags

    LIST_QUERY_MAP = (
        Base.QueryMapping('limit', 'limit', None, int),
        Base.QueryMapping('offset', 'offset', None, int),
        Base.QueryMapping('order', 'order_by', None, str),
        Base.QueryMapping('name', 'name', 'name', str),
        Base.QueryMapping('user', 'users', None, str),
        Base.QueryMapping('user_tag', 'user_tags__name', None, str),
    )

    class BaseEntry(dict):
        def __init__(self, user, base_url, doc, quotas=None):  # type: (mapping.User, str, mapping.Group, dict) -> None
            write_access = ctu.Rights.get(user.login in doc.users or user.super_user)
            pl = doc.priority_limits or user_controller.Group.regular.priority_limits
            if quotas is None:
                quotas = sandbox.yasandbox.controller.TaskQueue.owners_rating(owner=doc.name)
            sources = doc.get_sources()
            super(Group.BaseEntry, self).__init__({
                "name": doc.name,
                "url": '{}/{}'.format(base_url, doc.id),
                "abc": doc.abc,
                "email": doc.email,
                "members": sorted(doc.users or []),
                "rights": write_access,
                "priority_limits": {
                    attr: dict(zip(("class", "subclass"), (
                        getattr(ctu.DEFAULT_PRIORITY_LIMITS, attr)
                        if getattr(pl, attr) is None else
                        ctt.Priority().__setstate__(getattr(pl, attr))
                    ).__getstate__()))
                    for attr in ("ui", "api")
                },
                "sources": [
                    {
                        "source": source.source,
                        "group": source.group,
                    }
                    for source in sources
                ],
                # TODO: compatibility with old UI (SANDBOX-5061)
                "sync": {
                    attr: getattr(sources[0], attr)
                    for attr in ("source", "group")
                } if sources else None,
                "quota": sandbox.serviceapi.handlers.common.quota_info(doc.name, quotas=quotas),
                "user_tags": [_.name for _ in doc.user_tags or []],
                "messenger_chat_id": doc.messenger_chat_id if write_access == ctu.Rights.WRITE else None
            })

    class ListItemEntry(BaseEntry):
        pass

    class Entry(BaseEntry):
        pass

    @staticmethod
    def __check_user_permission(user, group_doc):
        if user.login not in group_doc.users and not user.super_user:
            return misc.json_error(
                httplib.FORBIDDEN,
                "User {} is not permitted to view or edit group {}".format(user.login, group_doc.name)
            )

    @staticmethod
    def __check_members(members):
        try:
            user_controller.validate_credentials(members)
        except Exception as ex:
            return misc.json_error(httplib.BAD_REQUEST, "User validation error: {}".format(ex))

    @classmethod
    def _id(cls, obj_id):
        try:
            return str(obj_id)
        except ValueError as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Path parameter validation error: " + str(ex))

    @classmethod
    def _document(cls, obj_id):
        return (
            user_controller.Group.anonymous
            if obj_id == user_controller.Group.anonymous.name else
            super(Group, cls)._document(obj_id)
        )

    @classmethod
    def list(cls, request):
        # Parse query arguments and form them as keyword arguments to database query builder
        try:
            kwargs, offset, limit = cls._handle_args(request)
        except (TypeError, ValueError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Query parameter validation error: " + str(ex))
        if limit is None:
            return misc.json_error(httplib.BAD_REQUEST, "Required parameter 'limit' not provided.")
        order = kwargs.pop('order_by', None)
        query = cls.Model.objects(**kwargs)

        total = query.count()
        if order:
            query = query.order_by(order)
        docs = list((query if not offset else query.skip(offset)).limit(limit))
        quotas = sandbox.yasandbox.controller.TaskQueue.owners_rating()
        return misc.response_json({
            "limit": limit,
            "offset": offset,
            "total": total,
            "items": [
                cls.ListItemEntry(request.user, request.uri, doc, quotas=quotas)
                for doc in docs
            ]
        })

    @classmethod
    def get(cls, request, obj_id):
        doc = cls._document(obj_id)
        if not doc:
            return misc.json_error(httplib.NOT_FOUND, "Group with ID '{}' not found.".format(obj_id))
        return misc.response_json(
            cls.Entry(request.user, request.uri.rsplit("/", 1)[0], doc))

    @staticmethod
    def __extract_users_and_groups(sync):
        entities = set(x.strip() for x in sync.get("group", "").split(","))
        if sync.get("source") == ctu.GroupSource.ABC:
            return set(), entities

        groups = set(it.ifilterfalse(user_controller.User.valid, entities))
        return entities - groups, groups

    @staticmethod
    def __parse_group_sources(data):
        syncs = []
        if "sources" in data:
            syncs = data.get("sources") or []
        elif data.get("sync"):
            syncs = [data.get("sync")]

        for sync in syncs:
            if sync and sync.get("source", ctm.NotExists) not in it.chain((None,), ctu.GroupSource):
                return misc.json_error(httplib.BAD_REQUEST, "sync.source '{}' is invalid".format(sync.get("source")))
        return syncs

    @classmethod
    def __process_group_sources(cls, users, syncs, update=False):
        for i, sync in enumerate(syncs):
            additional_users, groups = cls.__extract_users_and_groups(sync)
            users |= additional_users
            sync_users = users if i == 0 and not update else additional_users
            if sync.get("source") == ctu.GroupSource.ABC:
                sync_users = set("@{}".format(user) for user in sync_users)
            sync["group"] = ", ".join(sorted(sync_users | groups))

    @classmethod
    def validate_user_tags(cls, user_tags):
        if user_tags is None:
            return
        if len(user_tags) > cls.MAX_USER_TAG_COUNT:
            misc.json_error(
                httplib.BAD_REQUEST, "Maximum {} tags allowed for group.".format(cls.MAX_USER_TAG_COUNT)
            )

        unknown_tags = [tag for tag in user_tags if cls.USER_TAG_RE.match(tag) is None]

        if unknown_tags:
            return misc.json_error(
                httplib.BAD_REQUEST,
                "Tags {} must have prefix 'USER_' and consist of upper case letters, digits and '_'.".format(
                    ", ".join(unknown_tags)
                )
            )

        return

    @classmethod
    def create(cls, request):
        data = misc.request_data(request)

        if "name" not in data:
            return misc.json_error(httplib.BAD_REQUEST, "Field 'name' is required")
        name, users, abc = data["name"], data.get("members", []), data.get("abc")
        users = set(users + [request.user.login])

        syncs = cls.__parse_group_sources(data)
        cls.__process_group_sources(users, syncs)
        cls.__check_members(users)
        user_tags_validation = cls.validate_user_tags(data.get("user_tags"))
        if user_tags_validation is not None:
            return user_tags_validation
        try:
            group = user_controller.Group.create(
                cls.Model(
                    abc=abc,
                    name=name,
                    users=users,
                    email=data.get("email"),
                    messenger_chat_id=data.get("messenger_chat_id"),
                    sources=[cls.Model.SyncSource(**sync) for sync in syncs],
                    user_tags=[cls.Model.UserTag(name=tag) for tag in set(data.get("user_tags", []))]
                )
            )
        except (ValueError, me.errors.ValidationError, me.errors.NotUniqueError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, str(ex))
        return sandbox.web.helpers.response_created(
            "{}/{}".format(request.uri, group.id),
            content_type="application/json",
            content=json.dumps(
                cls.Entry(request.user, request.uri, group),
                ensure_ascii=False, encoding="utf-8", cls=common.rest.Client.CustomEncoder
            )
        )

    @classmethod
    def update(cls, request, group_id):
        data = misc.request_data(request)
        doc = cls._document(group_id)
        if not doc:
            return misc.json_error(httplib.NOT_FOUND, "Document {} not found".format(group_id))
        cls.__check_user_permission(request.user, doc)

        syncs = cls.__parse_group_sources(data)
        users = set(doc.users)

        # forbid addition of dismissed users
        if data.get("members"):
            cls.__check_members(set(data["members"]) - users)

        if syncs:
            cls.__process_group_sources(users, syncs, update=True)
            data["sources"] = [cls.Model.SyncSource(**sync) for sync in syncs if sync["group"]]
            if any(sync.get("source") for sync in syncs):
                data["members"] = list(users)

        if "members" in data:
            data["users"] = filter(user_controller.User.validate, data["members"])
            cls.__check_members(data["users"])

        if data.get("users") and not request.user.super_user and request.user.login not in data["users"]:
            return misc.json_error(
                httplib.BAD_REQUEST,
                "User {} cannot remove themself from group".format(request.user.login)
            )

        for field in ("users", "email", "sources", "abc", "messenger_chat_id"):
            value = data.get(field, ctm.NotExists)
            if value is not ctm.NotExists:
                setattr(doc, field, value)

        if data.get("user_tags") is not None:
            user_tags_validation = cls.validate_user_tags(data.get("user_tags"))
            if user_tags_validation is not None:
                return user_tags_validation
            doc.user_tags = [cls.Model.UserTag(name=tag) for tag in set(data["user_tags"])]

        if doc._get_changed_fields():
            try:
                if not doc.get_sources():
                    user_controller.Group.edit(doc)
                else:
                    user_controller.Group.sync(doc)
            except (ValueError, me.errors.ValidationError, me.errors.NotUniqueError) as ex:
                return misc.json_error(httplib.BAD_REQUEST, str(ex))

        return (
            misc.response_json(cls.Entry(request.user, request.uri, cls._document(group_id)))
            if cls.request_needs_updated_data(request) else
            sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)
        )

    @classmethod
    def delete(cls, request, group_id):
        doc = cls._document(group_id)
        if not doc:
            return misc.json_error(404, "Document {} not found".format(group_id))
        cls.__check_user_permission(request.user, doc)
        doc.delete()
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)


registry.registered_json("group")(Group.list)
registry.registered_json("group", ctm.RequestMethod.POST, ctu.Restriction.AUTHENTICATED)(Group.create)
registry.registered_json("group/([\w\-]+)", restriction=ctu.Restriction.AUTHENTICATED)(Group.get)
registry.registered_json(
    "group/([\w\-]+)", ctm.RequestMethod.PUT, ctu.Restriction.AUTHENTICATED)(Group.update)
registry.registered_json(
    "group/([\w\-]+)", ctm.RequestMethod.DELETE, ctu.Restriction.AUTHENTICATED)(Group.delete)
