import json
import cPickle
import datetime
import itertools as it
import collections

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu

from sandbox.common import config, utils, patterns
from sandbox.yasandbox.database import mapping

from sandbox.yasandbox import controller

from . import base as base_mapper


__all__ = ("TaskMapper",)


registry = config.Registry()


class TaskParametersMetaInfo(object):
    __slots__ = ("name", "output", "complex", "default")

    def __init__(self, **kwargs):
        for name in self.__slots__:
            setattr(self, name, kwargs.get(name, None))

    def to_dict(self):
        return {name: getattr(self, name) for name in self.__slots__}


TaskReport = collections.namedtuple("TaskReport", "label title")


class TaskMetaInfo(object):
    def __init__(self, sdk_type, parameters, reports):
        self.sdk_type = sdk_type
        self.parameters = [TaskParametersMetaInfo(**param) for param in parameters]
        self.reports = [TaskReport(**report) for report in reports]


class TaskMapperContext(object):

    def __init__(self, mapper, doc, legacy=True):
        """
        :type mapper: TaskMapper
        :type doc: mapping.Task
        """
        self.mapper = mapper
        self.doc = doc
        self.legacy = legacy

    @classmethod
    def get_field_mappers_defs(cls):
        return {
            "id": cls.id,
            "url": cls.url,
            "priority": cls.priority,
            "important": cls.important,
            "type": cls.type,
            "scores": cls.scores,
            "description": cls.description,
            "owner": cls.owner,
            "author": cls.author,
            "status": cls.status,
            "context": cls.context,

            "time": cls.time,
            "time.created": cls.time_created,
            "time.updated": cls.time_updated,

            "results.info": cls.results_info,
            "parent": cls.parent,

            "kill_timeout": cls.kill_timeout,
            "fail_on_any_error": cls.fail_on_any_error,
            "tasks_archive_resource": cls.tasks_archive_resource,
            "dump_disk_usage": cls.dump_disk_usage,
            "tcpdump_args": cls.tcpdump_args,
            "tags": cls.tags,
            "hints": cls.hints,
            "hidden": cls.hidden,
            "se_tags": cls.se_tags,
            "se_tag": cls.se_tag,
            "max_restarts": cls.max_restarts,
            "uniqueness": cls.uniqueness,
            "expires": cls.expires,
            "scheduler": cls.scheduler,
            "template_alias": cls.template_alias,
            "enable_yav": cls.enable_yav,
            "suspend_on_status": cls.suspend_on_status,
            "push_tasks_resource": cls.push_tasks_resource,
            "score": cls.score,

            "requirements": cls.requirements,
            "requirements.disk_space": cls.requirements_disk_space,
            "requirements.platform": cls.requirements_platform,
            "requirements.cpu_model": cls.requirements_cpu_model,
            "requirements.cores": cls.requirements_cores,
            "requirements.host": cls.requirements_host,
            "requirements.ram": cls.requirements_ram,
            "requirements.privileged": cls.requirements_privileged,
            "requirements.dns": cls.requirements_dns,
            "requirements.client_tags": cls.requirements_client_tags,
            "requirements.semaphores": cls.requirements_semaphores,
            "requirements.caches": cls.requirements_caches,
            "requirements.resources": cls.requirements_resources,
            "requirements.ramdrive": cls.requirements_ramdrive,
            "requirements.porto_layers": cls.requirements_porto_layers,
            "requirements.container_resource": cls.requirements_container_resource,
            "requirements.resources_space_reserve": cls.requirements_resources_space_reserve,

            "notifications": cls.notifications,
            "input_parameters": cls.input_parameters,
            "output_parameters": cls.output_parameters,
            "release": cls.release,
            "execution": cls.execution,
            "intervals": cls.intervals,

            "reports": cls.reports,
            "rights": cls.rights,
            "short_task_result": cls.short_task_result,
        }

    @utils.singleton_property
    def weather(self):
        if self.mapper.weather_map is None:
            return mapping.Weather.objects(type=self.doc.type).first()
        return self.mapper.weather_map.get(self.doc.type)

    def id(self):
        return self.doc.id

    def url(self):
        return "{}/{}".format(self.mapper.base_url, self.doc.id)

    def priority(self):
        return ctt.Priority.make(self.doc.priority).as_dict()

    def important(self):
        return self.doc.flagged

    def type(self):
        return self.doc.type

    def scores(self):
        return self.weather.data.weather if self.weather else 10

    def description(self):
        return self.doc.description

    def owner(self):
        return self.doc.owner

    def author(self):
        return self.doc.author

    def status(self):
        if self.mapper.semaphore_waiters and self.doc.id in self.mapper.semaphore_waiters:
            return ctt.Status.WAIT_MUTEX
        if hasattr(self.doc, "execution"):
            return self.doc.execution.status
        return ctt.Status.DRAFT

    def context(self):
        if self.doc.context:
            return cPickle.loads(self.doc.context)
        return None

    def time(self):
        return {
            "created": self.time_created(),
            "updated": self.time_updated(),
        }

    def time_created(self):
        return utils.utcdt2iso(self.doc.time.created)

    def time_updated(self):
        return utils.utcdt2iso(self.doc.time.updated)

    def results_info(self):
        if hasattr(self.doc, "execution"):
            return self.doc.execution.description
        return None

    def parent(self):
        if getattr(self.doc, "parent_id", None):
            return {
                "id": self.doc.parent_id,
                "url": "{}/{}".format(self.mapper.base_url, self.doc.parent_id)
            }
        return None

    def hints(self):
        return self.doc.hints or []

    def hidden(self):
        return self.doc.hidden

    def se_tags(self):
        return []

    def se_tag(self):
        return self.doc.se_tag

    def max_restarts(self):
        if self.doc.max_restarts is not None:
            return self.doc.max_restarts
        return registry.common.task.execution.max_restarts

    def uniqueness(self):
        return dict(key=self.doc.unique_key)

    def expires(self):
        return self.doc.expires_delta

    def scheduler(self):
        if self.doc.scheduler:
            return {
                "url": "{}/scheduler/{}".format(self.mapper.base_url.rsplit("/", 2)[0], abs(self.doc.scheduler)),
                "id": self.doc.scheduler,
            }
        return None

    def template_alias(self):
        return self.doc.template_alias

    def enable_yav(self):
        return self.doc.enable_yav

    def suspend_on_status(self):
        return self.doc.suspend_on_status

    def push_tasks_resource(self):
        return self.doc.push_tasks_resource

    def score(self):
        return self.doc.score

    def requirements(self):
        return {
            "disk_space": self.requirements_disk_space(),
            "platform": self.requirements_platform(),
            "cpu_model": self.requirements_cpu_model(),
            "cores": self.requirements_cores(),
            "host": self.requirements_host(),
            "ram": self.requirements_ram(),
            "privileged": self.requirements_privileged(),
            "dns": self.requirements_dns(),
            "client_tags": self.requirements_client_tags(),
            "semaphores": self.requirements_semaphores(),
            "caches": self.requirements_caches(),
            "resources": self.requirements_resources(),
            "ramdrive": self.requirements_ramdrive(),
            "porto_layers": self.requirements_porto_layers(),
            "tasks_resource": self.requirements_tasks_resource(),
            "container_resource": self.requirements_container_resource(),
            "resources_space_reserve": self.requirements_resources_space_reserve(),
        }

    def requirements_disk_space(self):
        return self.doc.requirements.disk_space << 20

    def requirements_platform(self):
        return self.doc.requirements.platform

    def requirements_cpu_model(self):
        return self.doc.requirements.cpu_model

    def requirements_cores(self):
        return self.doc.requirements.cores or 0

    def requirements_host(self):
        return self.doc.requirements.host

    def requirements_ram(self):
        return self.doc.requirements.ram << 20

    def requirements_privileged(self):
        return bool(self.doc.requirements.privileged)

    def requirements_dns(self):
        return self.doc.requirements.dns or ctm.DnsType.DEFAULT

    def requirements_semaphores(self):
        if self.doc.requirements.semaphores:
            return ctt.Semaphores(**self.doc.requirements.semaphores.to_mongo()).to_dict(with_public=False)
        return None

    def requirements_caches(self):
        if self.doc.requirements.caches is None:
            return None
        return {c.key: c.value for c in self.doc.requirements.caches}

    def requirements_resources(self):
        return {
            "url": "{}/{}/requirements".format(self.mapper.base_url, self.doc.id),
            "count": len(self.doc.requirements.resources),
            "ids": self.doc.requirements.resources,
        }

    def requirements_ramdrive(self):
        if self.doc.requirements.ramdrive:
            return {
                "type": self.doc.requirements.ramdrive.type,
                "size": self.doc.requirements.ramdrive.size << 20
            }
        return None

    def requirements_porto_layers(self):
        return self.doc.requirements.porto_layers

    def requirements_tasks_resource(self):
        return self.doc.requirements.tasks_resource and self.doc.requirements.tasks_resource.id

    def requirements_container_resource(self):
        return self.doc.requirements.container_resource

    def requirements_resources_space_reserve(self):
        if self.doc.requirements.resources_space_reserve:
            return {item.bucket: item.size for item in self.doc.requirements.resources_space_reserve}
        return None

    def notifications(self):
        return [
            {
                "statuses": list(ctt.Status.Group.collapse(n.statuses)),
                "recipients": list(n.recipients),
                "transport": n.transport,
                "check_status": n.check_status,
                "juggler_tags": list(n.juggler_tags)
            }
            for n in self.doc.notifications
        ]

    def release(self):
        if getattr(self.doc, "release", None):  # Schedulers has no "release" sub-document
            return {
                "url": "{}/release/{}".format(self.mapper.base_url.rsplit("/", 1)[0], self.doc.id),
                "type": self.doc.release.status,
                "id": self.doc.id,
            }
        return {}

    def execution(self):
        result = {}
        ex = self.doc.execution
        ex_ = None

        if ex.status in it.chain(ctt.Status.Group.REALEXECUTE, (ctt.Status.FINISHING,)):
            started = ex.last_execution_start

            ex_ = result = {
                "estimated": self.weather.data.eta if self.weather else 0,
                "current": (self.mapper.now - (started or self.mapper.now)).total_seconds(),
            }

        elif ex.status not in tuple(ctt.Status.Group.DRAFT) + tuple(ctt.Status.Group.QUEUE):
            started = ex.last_execution_start
            finished = ex.last_execution_finish

            ex_ = result = {
                "estimated": self.weather.data.eta if self.weather else 0,
                "current": (finished - started).total_seconds() if finished and started else 0,
                "started": utils.utcdt2iso(started),
                "finished": utils.utcdt2iso(finished),
            }

        if ex_ and ex.host:
            ex_["client"] = {
                "id": ex.host,
                "url": "{}/client/{}".format(self.mapper.base_url.rsplit("/", 1)[0], ex.host)
            }

        return result

    def intervals(self):
        result = {}
        if self.doc.execution.intervals:
            for interval_type in ctt.IntervalType:
                if getattr(self.doc.execution.intervals, interval_type):
                    result[interval_type] = []
                    for interval in getattr(self.doc.execution.intervals, interval_type):
                        item = {
                            "start": utils.utcdt2iso(interval.start),
                            "duration": interval.duration
                        }
                        if interval.pool is not None:
                            item["pool"] = interval.pool
                        if interval.consumption is not None:
                            item["consumption"] = interval.consumption
                        result[interval_type].append(item)
        return result

    def rights(self):
        from sandbox.yasandbox import controller
        return ctu.Rights.get(controller.user_has_permission(
            self.mapper.user, (self.doc.author, self.doc.owner)
        ))

    def kill_timeout(self):
        return controller.Task.kill_timeout(self.doc)

    def fail_on_any_error(self):
        if self.doc.fail_on_any_error is not None:
            return bool(self.doc.fail_on_any_error)

        if controller.Task.is_sdk1_type(self.doc, self.mapper.task_meta.get(self.doc.type)):
            return bool(self.doc.ctx.get("fail_on_any_error"))
        return bool(self.doc.fail_on_any_error)

    def tasks_archive_resource(self):
        return controller.Task.tasks_archive_resource(self.doc)

    def dump_disk_usage(self):
        if self.doc.dump_disk_usage is not None:
            return bool(self.doc.dump_disk_usage)

        if controller.Task.is_sdk1_type(self.doc, self.mapper.task_meta.get(self.doc.type)):
            return bool(self.doc.ctx.get("dump_disk_usage", True))
        return bool(self.doc.dump_disk_usage)

    def tcpdump_args(self):
        return self.doc.tcpdump_args

    def tags(self):
        return self.doc.tags or []

    def requirements_client_tags(self):
        return str(controller.Task.client_tags(self.doc))

    def input_parameters(self):
        params = {}
        if self.doc.parameters:
            params, _ = controller.Task.parameters_dicts(
                self.doc, output=False, task_meta=self.mapper.task_meta.get(self.doc.type)
            )
        return params

    def output_parameters(self):
        params = {}
        if self.doc.parameters:
            _, params = controller.Task.parameters_dicts(
                self.doc, input=False, task_meta=self.mapper.task_meta.get(self.doc.type)
            )
        return params

    def reports(self):
        if self.doc.reports is not None:
            reports = self.doc.reports
        else:
            if self.mapper.task_meta.get(self.doc.type) is not None:
                reports = self.mapper.task_meta.get(self.doc.type).reports
            else:
                reports = []
        return [
            {
                "title": report.title,
                "label": report.label,
                "url": "{}/{}/reports/{}".format(self.mapper.base_url, self.doc.id, report.label),
            }
            for report in reports
        ]

    def short_task_result(self):
        try:
            if controller.Task.is_sdk1_type(self.doc, self.mapper.task_meta.get(self.doc.type)):
                short_res = self.doc.ctx.get("task_short_result")
                json.dumps(short_res)
                return short_res
        except Exception as ex:
            return "Can't get short task result or serialize it: {}".format(ex)


class TaskMapper(base_mapper.BaseMapper):
    """
    Mapper for mapping task model to task json API representation.
    It implements all fields required for creation of `sdk2.Task` instance.
    """
    mapper_context = TaskMapperContext

    def __init__(
        self, fields, user, base_url, semaphore_waiters, weather_map, template=False, task_docs=None,
        types_resolve_func=None
    ):
        super(TaskMapper, self).__init__()
        self.user = user
        self.base_url = base_url.rstrip("/")
        self.semaphore_waiters = semaphore_waiters
        self.weather_map = weather_map
        self.now = datetime.datetime.utcnow()
        self.task_docs = task_docs
        self.types_resolve_func = types_resolve_func

        if fields is None:
            fields = self.get_base_fields()
        if template:
            fields -= {"execution", "intervals"}

        self._field_mappers = self._build_field_mappers(fields)

    @patterns.singleton_property
    def task_types(self):
        return list(set(doc.type for doc in self.task_docs))

    def update_parameters_meta_fields(self):
        tasks_parameters_meta = dict()
        docs_with_meta = []

        for doc in self.task_docs:
            get_bson_value = getattr(doc, "get_bson_value", None)
            if get_bson_value is not None:
                meta_id = get_bson_value("parameters_meta")
                if meta_id is not None:
                    tasks_parameters_meta[meta_id] = None
                    docs_with_meta.append(doc)

        if tasks_parameters_meta:
            parameters_meta_list = mapping.ParametersMeta.objects(id__in=tasks_parameters_meta.keys()).lite()
            for param in parameters_meta_list:
                tasks_parameters_meta[param.id] = param

        completed_tasks_meta = (len(self.task_docs) == len(docs_with_meta))
        for doc in docs_with_meta:
            meta_object = tasks_parameters_meta[doc.get_bson_value("parameters_meta")]
            if meta_object is None:
                completed_tasks_meta = False
            doc.update_cache_value(
                "parameters_meta", "pm", meta_object
            )
        return completed_tasks_meta

    @patterns.singleton_property
    def task_meta(self):
        if not self.update_parameters_meta_fields():
            task_classes = self.types_resolve_func(self.task_types or [])
        else:
            task_classes = {}
        return {task_type: TaskMetaInfo(**value) for task_type, value in task_classes.iteritems()}

    @classmethod
    def get_base_fields(cls):
        return {
            "id", "priority", "type",
            "description", "owner", "author", "status", "time", "parent", "scheduler", "template_alias", "enable_yav",

            "kill_timeout", "fail_on_any_error", "tasks_archive_resource",
            "dump_disk_usage", "tcpdump_args", "tags", "hints",
            "max_restarts", "uniqueness",

            "requirements", "notifications",
            "input_parameters", "output_parameters",

            "url", "rights", "important", "scores",
            "results.info",

            "hidden", "se_tags", "se_tag",
            "expires",

            "release", "reports",
            "execution", "intervals",

            "suspend_on_status", "push_tasks_resource", "score"
        }
