import collections

from . import base as base_mapper

from sandbox import common
import sandbox.common.types.task as ctt

from sandbox.taskbox.model import update as model_update


class TemplateMapperContext(object):
    def __init__(self, mapper, doc, legacy=True, **kwargs):
        """
        :type mapper: TemplateMapper or None
        :type doc: mapping.TaskTemplate

        """
        self.mapper = mapper
        self.doc = doc
        self.legacy = legacy

    @classmethod
    def get_field_mappers_defs(cls):
        return {
            "author": cls.author,
            "status": cls.status,
            "alias": cls.alias,
            "description": cls.description,
            "shared_with": cls.shared_with,
            "task": cls.task,
            "task.type": cls.task_type,
            "task.parameters": cls.task_parameters,
            "task.parameters.input": cls.task_parameters_input,
            "task.parameters.common": cls.task_parameters_common,
            "task.requirements": cls.task_requirements,
            "time": cls.time,
            "time.created": cls.time_created,
            "time.updated": cls.time_updated,
            "favorite": cls.favorite,
        }

    def author(self):
        return self.doc.author

    def status(self):
        return self.doc.status

    def alias(self):
        return self.doc.alias

    def description(self):
        return self.doc.description

    def shared_with(self):
        return self.doc.shared_with

    def view_parameter_meta(self, pm):
        value = pm.value
        value = model_update.decode_parameter(value, pm.complex)
        # Multi-select field must be a list of strings (selected items), but it is false for sdk1 tasks.
        if pm.type == ctt.ParameterType.MULTISELECT and isinstance(value, basestring):
            value = value.split()

        if pm.name == "priority" and value is not None:
            value = common.types.task.Priority.make(value).as_dict(legacy=self.legacy)

        field = {
            "name": pm.name,
            "required": pm.required,
            "title": pm.title,
            "description": pm.description,
            "output": pm.output,
            "modifiers": pm.modifiers,
            "context": pm.context,
            "hide": pm.hide,
            "filter": pm.filter,
            "default": model_update.decode_parameter(pm.default, pm.complex),
            "default_from_code": pm.default_from_code
        }
        if pm.type:
            field["type"] = pm.type
        if pm.type != ctt.ParameterType.BLOCK:
            if pm.sub_fields:
                field["sub_fields"] = pm.sub_fields

        parameter_view = {
            "meta": field,
            "value": value
        }

        return parameter_view

    def view_parameters_meta(self, parameters_meta):
        result = collections.OrderedDict()
        for idx, pm in enumerate(parameters_meta):
            parameter_view = self.view_parameter_meta(pm)
            parameter_view["meta"]["order"] = idx
            result[pm.name] = parameter_view

        return result

    def task_type(self):
        return self.doc.task.type

    def task_parameters_input(self):
        return self.view_parameters_meta(self.doc.task.input_parameters)

    def task_parameters_common(self):
        return self.view_parameters_meta(self.doc.task.common_parameters)

    def task_requirements(self):
        return self.view_parameters_meta(self.doc.task.requirements)

    def task_parameters(self):
        return {
            "input": self.task_parameters_input(),
            "common": self.task_parameters_common()
        }

    def task(self):
        return {
            "type": self.task_type(),
            "parameters": self.task_parameters(),
            "requirements": self.task_requirements()
        }

    def time_created(self):
        return common.format.utcdt2iso(self.doc.time.created)

    def time_updated(self):
        return common.format.utcdt2iso(self.doc.time.updated)

    def time(self):
        return {
            "created": self.time_created(),
            "updated": self.time_updated()
        }

    def favorite(self):
        return self.mapper.user.login in self.doc.favorites


class TemplateMapper(base_mapper.BaseMapper):
    mapper_context = TemplateMapperContext

    def __init__(self, fields, user):
        super(TemplateMapper, self).__init__()
        self.user = user
        if fields is None:
            fields = self.get_base_fields()
        fields = set(fields) | {"alias"}
        self._field_mappers = self._build_field_mappers(fields)

    @classmethod
    def get_base_fields(cls):
        return {"author", "status", "alias", "description", "shared_with", "task", "time"}


class TemplateAuditMapper(object):
    def __init__(self, legacy=True):
        self.mapper_context = TemplateMapperContext(None, None, legacy=legacy)

    def view_parameter_meta(self, parameter):
        return parameter if parameter is None else self.mapper_context.view_parameter_meta(parameter)

    def dump_parameters(self, doc_parameter_array):
        result = {}
        for parameter_audit in doc_parameter_array:
            name = parameter_audit.new.name if parameter_audit.new else parameter_audit.old.name
            result[name] = {}
            result[name]["old"] = self.view_parameter_meta(parameter_audit.old)
            result[name]["new"] = self.view_parameter_meta(parameter_audit.new)
        return result

    def dump_properties(self, doc_properties):
        result = {}
        for name, property_audit in doc_properties.iteritems():
            result[name] = {}
            result[name]["old"] = property_audit.old
            result[name]["new"] = property_audit.new
        return result

    def dump(self, doc):
        return {
            "template_alias": doc.template_alias,
            "author": doc.author,
            "date": doc.date,
            "task": {
                "parameters": {
                    "input": self.dump_parameters(doc.input_parameters),
                    "common": self.dump_parameters(doc.common_parameters)
                },
                "requirements": self.dump_parameters(doc.requirements)
            },
            "properties": self.dump_properties(doc.properties)
        }
