import json
import httplib

import six

from sandbox import common

import sandbox.web.response


def response_json(message):
    """ Produce JSON-formatted reply from `message` """

    try:
        message = json.dumps(message, ensure_ascii=False, encoding="utf-8", cls=common.rest.Client.CustomEncoder)
    except Exception as error:
        raise ValueError(u"Cannot dump json message: {}. Error: ({}) {}".format(message, type(error), error))
    raise sandbox.web.response.HttpResponse(
        "application/json", message, httplib.OK,
    )


def response_json_ex(code, message, headers=None):
    """
    Extended version of `response_json`.

    :param code:    HTTP code to respond
    :param message: Data to be encoded to JSON and returned as response content
    :param headers: HTTP headers to be returned with the response
    """
    try:
        message = json.dumps(message, ensure_ascii=False, encoding="utf-8", cls=common.rest.Client.CustomEncoder)
    except Exception as error:
        raise ValueError("Cannot dump json message", error)
    headers = headers or {}
    raise sandbox.web.response.HttpResponse("application/json", message, code, headers)


def json_error(code, message, headers=None):
    """
    Respond with an error message to REST API request.

    :param code:    HTTP code to respond
    :param message: Error message to be returned. If the message is string, it will be returned as
                    dictionary value for "reason" key.
    :param headers: HTTP headers to return with the response
    """
    try:
        message = json.dumps(
            {"reason": message} if isinstance(message, six.string_types) else message,
            ensure_ascii=False, encoding="utf-8", cls=common.rest.Client.CustomEncoder
        )
    except Exception as error:
        raise ValueError(u"Cannot dump json message: {}. Error: ({}) {}".format(message, type(error), error))
    headers = headers or {}
    raise sandbox.web.response.HttpErrorResponse(code, message, "application/json", headers)


def json_parser(data, expects=dict, expects_each=None):
    try:
        data = json.loads(data)
    except ValueError as ex:
        return json_error(httplib.BAD_REQUEST, "Unable to parse input data: " + str(ex))
    if not isinstance(data, expects):
        return json_error(
            httplib.BAD_REQUEST,
            "Wrong input data {} - expects {}".format(type(data), expects)
        )
    if expects_each is not None:
        for key in data:
            if not isinstance(key, expects_each):
                return json_error(
                    httplib.BAD_REQUEST,
                    "Wrong key {} - expects {}".format(type(key), expects_each)
                )
    return data


def request_data(request, expects=dict, expects_each=None):
    """ Parses given request's data as JSON and also validates it is of correct type. """
    return json_parser(request.raw_data, expects=expects, expects_each=expects_each)
