import json
import httplib
import logging
import pymongo
import datetime as dt
import distutils.util

import sandbox.web.helpers
import sandbox.web.response
from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr

from sandbox import sdk2

from sandbox.yasandbox import manager
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.api.json import Base
from sandbox.yasandbox.api.json import registry
from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import list_arg_parser


###########################################################
# Service
###########################################################

@registry.registered_json("release/list")
def list_releases(request):
    resource_type = request.get("resource_type")
    release_status = request.get("release_status")
    task_id = request.getint("task_id")
    arch = request.get("arch")
    order_by = request.get("order_by")
    creation_ts_gte = request.getint("creation_ts_gte")
    creation_ts_lte = request.getint("creation_ts_lte")
    creation_ts_bound = None
    if creation_ts_gte and creation_ts_lte:
        creation_ts_gte = dt.datetime.utcfromtimestamp(creation_ts_gte)
        creation_ts_lte = dt.datetime.utcfromtimestamp(creation_ts_lte)
        creation_ts_bound = (creation_ts_gte, creation_ts_lte)
    limit = request.get("limit") or 20
    offset = request.get("offset")
    ret = manager.release_manager.list_releases(
        resource_type=resource_type,
        release_status=release_status,
        task_id=task_id,
        arch=arch,
        order_by=order_by,
        creation_ts_bound=creation_ts_bound,
        limit=limit, offset=offset,
        resource_as_dict=True,
    )
    return misc.response_json(ret)


###########################################################
# API Version 1.0
###########################################################

class Release(Base):
    """
    The class encapsulates all the logic related to REST API representation of any entities related to release object.
    """

    logger = logging.getLogger("RESTAPI_Release")

    # Shortcuts for database models.
    Model = mapping.Task
    ResourceModel = mapping.Resource

    # A list of list operation query parameters mapping.
    LIST_QUERY_MAP = (
        Base.QueryMapping("resource_type", "resource_type", "resource_type", str),
        Base.QueryMapping("task_id", "task_id", "task_id", list_arg_parser(int)),
        Base.QueryMapping("type", "type", None, list_arg_parser(str)),
        Base.QueryMapping("owner", "owner", None, str),
        Base.QueryMapping("created", "created", "release__creation_time", sandbox.web.helpers.datetime_couple),
        Base.QueryMapping("include_broken", "include_broken", None, distutils.util.strtobool),
        Base.QueryMapping("arch", "arch", None, str),
        Base.QueryMapping("limit", "limit", None, int),
        Base.QueryMapping("offset", "offset", None, int),
        Base.QueryMapping("order", "order_by", None, str),
    )

    class BaseEntry(dict):
        def __init__(self, _, doc):
            super(Release.BaseEntry, self).__init__(
                task_id=doc.id,
                author=doc.release.author,
                type=doc.release.status,
                created=sandbox.web.helpers.utcdt2iso(doc.release.creation_time),
                subject=doc.release.message.subject
            )

    class ListItemEntry(BaseEntry):
        def __init__(self, base_url, doc):
            super(Release.ListItemEntry, self).__init__(base_url, doc)

    class Entry(BaseEntry):
        def __init__(self, base_url, doc):
            super(Release.Entry, self).__init__(base_url, doc)
            self["message"] = doc.release.message.body
            self["resources"] = [
                {
                    "resource_id": res.id,
                    "type": res.type,
                    "description": res.name,
                    "releasers": res_type.releasers
                }
                for res, res_type in (
                    (_, sdk2.Resource[_.type] if _.type in sdk2.Resource else sdk2.service_resources.UnknownResource)
                    for _ in Release.ResourceModel.objects(task_id=doc.id)
                )
                if res_type.releasable
            ]

    @classmethod
    def _released_resources(
        cls, resource_type=None, status=None, include_broken=False, task_id=None, arch=None, limit=None, offset=None
    ):
        match = {
            "state": {
                "$in": (
                    ctr.State.READY,
                    ctr.State.BROKEN
                )
            }
            if include_broken else
            ctr.State.READY
        }
        if not status:
            status = list(iter(ctt.ReleaseStatus))
        match["attrs"] = {"$elemMatch": {"k": "released", "v": status[0] if len(status) == 1 else {"$in": status}}}
        if resource_type:
            match["type"] = resource_type
        if task_id:
            match["tid"] = task_id[0] if len(task_id) == 1 else {"$in": task_id}
        if arch:
            match["arch"] = {"$in": (arch, "any")}

        pipeline = [
            {"$match": match},
            {"$group": {
                "_id": "$tid",
                "res": {"$addToSet": "$_id"},
            }},
            {"$sort": {"_id": pymongo.DESCENDING}},
        ]
        if offset:
            pipeline.append({"$skip": int(offset)})
        if limit:
            pipeline.append({"$limit": int(limit)})
        return [(r["_id"], r["res"]) for r in cls.ResourceModel.aggregate(pipeline)]

    @classmethod
    def list(cls, request):
        # Parse query arguments and form them as keyword arguments to database query builder
        try:
            kwargs, offset, limit = cls._handle_args(request)
        except (TypeError, ValueError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Query parameter validation error: " + str(ex))
        if limit is None:
            return misc.json_error(httplib.BAD_REQUEST, "Required parameter 'limit' not provided.")

        released_resources = cls._released_resources(
            resource_type=kwargs.get("resource_type"),
            status=kwargs.get("type"),
            include_broken=kwargs.get("include_broken"),
            task_id=kwargs.get("task_id"),
            arch=kwargs.get("arch"),
            limit=limit,
            offset=offset
        )
        cond = {
            "id__in": [x[0] for x in released_resources],
            "execution__status": ctt.Status.RELEASED,
            "release__exists": True,
        }
        created = kwargs.get("created")
        if created:
            cond["release__creation_time__gte"] = created[0]
            cond["release__creation_time__lte"] = created[1]
        query = cls.Model.objects(**cond).order_by(kwargs.get("order_by"))
        return misc.response_json({
            "limit": limit,
            "offset": offset,
            "total": query.count(),
            "items": [cls.ListItemEntry(request.uri, doc) for doc in query]
        })

    @classmethod
    def get(cls, request, obj_id):
        doc = cls._document(obj_id)
        if doc.execution.status != ctt.Status.RELEASED:
            return misc.json_error(httplib.NOT_FOUND, "Release does not exist")
        return misc.response_json(cls.Entry(request.uri, doc))

    @classmethod
    def create(cls, request):
        data = json.loads(request.raw_data)
        message = None
        try:
            task = controller.TaskWrapper(cls._document(data["task_id"]))

            status = data["type"]
            author = request.user.login
            custom_to = data.get("to")
            addresses_to = list(common.utils.chain(custom_to)) if custom_to else (list(task.release_to or []))
            addresses_cc = list(common.utils.chain(data.get("cc", [])))
            settings = common.config.Registry()

            if task.model.execution.status == ctt.Status.RELEASING:
                if mapping.Audit.objects(
                    task_id=task.id,
                    status=ctt.Status.RELEASING,
                    request_id=request.id
                ).first():
                    return sandbox.web.response.HttpResponse(code=httplib.ACCEPTED)
                raise common.errors.ReleaseError("Releasing of task #{} is already in progress".format(task.id))
            if not ctt.Status.can_switch(task.model.execution.status, ctt.Status.RELEASING):
                raise common.errors.ReleaseError("Task #{} is not finished successfully".format(task.id))

            if status not in ctt.ReleaseStatus:
                raise common.errors.ReleaseError("Invalid release status {!r}".format(status))

            task_resources = [r for r in task.resources() if r.type.releasable]
            if not task_resources:
                raise common.errors.ReleaseError("No resources for release")

            bad_resources = [r for r in task_resources if r.state in (ctr.State.BROKEN, ctr.State.DELETED)]
            if bad_resources:
                raise common.errors.ReleaseError(
                    "Cannot release task #{}: the following resources are broken/deleted: {}".format(
                        task.id, ", ".join((str(res.id) for res in bad_resources))
                    )
                )

            for resource in task_resources if settings.server.auth.enabled else []:
                user = controller.User.get(author)
                releasers = resource.type.releasers
                if not (user and user.super_user) and releasers and author not in controller.Group.unpack(releasers):
                    raise common.errors.ReleaseError(
                        "Access denied. User {} cannot release resource #{}, type {}. "
                        "To fix this, you should add this user to 'releasers' property of this resource".format(
                            author, resource.id, resource.type
                        )
                    )
            cls.logger.info("Creating release for #%s author %s with status %s", task.id, author, status)

            for resource in task_resources:
                if resource.type.release_subscribers:
                    addresses_cc.extend(resource.type.release_subscribers)

            params = data.get("params", {})
            params.setdefault("release_status", status)
            params.setdefault("release_subject", data["subject"])
            params.setdefault("release_comments", data.get("message"))
            params["releaser"] = author
            params["email_notifications"] = {"to": addresses_to, "cc": addresses_cc}
            task.model.execution.release_params = params
            task.model.save()
            try:
                task.set_status(
                    ctt.Status.RELEASING, event=controller.Group.check_abc(task.model), set_last_action_user=True
                )
            except common.errors.UpdateConflict:
                if request and mapping.Audit.objects(
                    task_id=task.id,
                    status=ctt.Status.RELEASING,
                    request_id=request.id
                ).first():
                    return sandbox.web.response.HttpResponse(
                        content_type="application/json",
                        content="{}",
                        code=httplib.ACCEPTED
                    )
                raise
        except KeyError as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Field is required '{}'".format(ex))
        except common.errors.SandboxException as ex:
            return misc.json_error(httplib.BAD_REQUEST, ex.message)
        return misc.response_json_ex(httplib.ACCEPTED, {"message": message})


registry.registered_json("release")(Release.list)
registry.registered_json("release", ctm.RequestMethod.POST)(Release.create)
registry.registered_json("release/(\d+)")(Release.get)
