import json
import re
import math
import httplib
import logging
import datetime as dt
import distutils.util

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.user as ctu
import sandbox.common.types.database as ctd
import sandbox.common.types.resource as ctr

from sandbox import sdk2

import sandbox.web.helpers
import sandbox.web.response
import sandbox.web.server.request
from sandbox.yasandbox import controller
import yasandbox.manager
import sandbox.yasandbox.proxy.resource as proxy_resource
import sandbox.yasandbox.database.mapping as mp

from sandbox.web import api
from sandbox.yasandbox.api.json import Base
from sandbox.yasandbox.api.json import PathBase
from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import registry
from sandbox.yasandbox.api.json import list_arg_parser


###########################################################
# API Version 1.0
###########################################################
class Resource(Base):
    """
    The class encapsulates all the logic related to REST API representation of any entities related to resource object.
    """

    logger = logging.getLogger("RESTAPI_Resource")

    # Shortcuts for database models.
    Model = mp.Resource

    # A list of list operation query parameters mapping.
    LIST_QUERY_MAP = (
        Base.QueryMapping('id', 'id', 'id', list_arg_parser(int)),
        Base.QueryMapping('type', 'resource_type', 'type', list_arg_parser(str)),
        Base.QueryMapping('arch', 'arch', 'arch', str),
        Base.QueryMapping('state', 'state', 'state', list_arg_parser(str)),
        Base.QueryMapping('owner', 'owner', 'owner', str),
        Base.QueryMapping('client', 'host', 'hosts__h', str),
        Base.QueryMapping('task_id', 'task_id', 'task_id', list_arg_parser(int)),
        Base.QueryMapping('dependant', 'dependant', 'task_id', int),
        Base.QueryMapping('accessed', 'accessed', 'time__at', sandbox.web.helpers.datetime_couple),
        Base.QueryMapping('created', 'created', 'time__ct', sandbox.web.helpers.datetime_couple),
        Base.QueryMapping('updated', 'updated', 'time__up', sandbox.web.helpers.datetime_couple),
        Base.QueryMapping('attr_name', 'attr_name', None, str),
        Base.QueryMapping('attr_value', 'attr_value', None, str),
        Base.QueryMapping('any_attr', 'any_attr', None, lambda _: bool(distutils.util.strtobool(_))),
        Base.QueryMapping('attrs', 'attrs', None, str),
        Base.QueryMapping('limit', 'limit', None, int),
        Base.QueryMapping('offset', 'offset', None, int),
        Base.QueryMapping('order', 'order_by', None, str),
    )

    class BaseEntry(dict):
        @classmethod
        def resource_meta_from_model(cls, meta):
            return dict(
                name=meta.name,
                parent=meta.parent,
                releaseable=meta.releaseable,
                releasable=meta.releasable,
                any_arch=meta.any_arch,
                releasers=meta.releasers,
                executable=meta.executable,
                auto_backup=meta.auto_backup,
                calc_md5=meta.calc_md5,
                on_restart=meta.on_restart,
                release_subscribers=meta.release_subscribers,
                share=meta.share,
                default_attribute=meta.default_attribute,
                attributes=meta.attributes
            )

        def __init__(self, user, base_url, doc, resource=None, lightweight=False, resource_meta_hashes=None):
            resource = resource or proxy_resource.Resource._restore(doc)
            write_access = resource.user_has_permission(user) if not lightweight else False
            try:
                ttl = resource.attrs.get("ttl", 14)
                ttl = int(ttl) if ttl != "inf" else None
            except:
                ttl = 14

            resource_meta = None
            if resource_meta_hashes and doc.resource_meta:
                resource_meta = [
                    self.resource_meta_from_model(resource_meta_hashes[meta_hash].resource_meta)
                    for meta_hash in doc.resource_meta
                ]

            TAR_ARCHIVES_REGEX = re.compile("\w+\.t((gz)|(ar(\.gz)?))$")

            observable = doc.multifile or bool(TAR_ARCHIVES_REGEX.search(doc.path))

            super(Resource.BaseEntry, self).__init__({
                "id": doc.id,
                "rights": ctu.Rights.get(write_access),
                "url": '{}/{}'.format(base_url, doc.id),
                "type": doc.type,
                "owner": doc.owner,
                "arch": doc.arch,
                "state": doc.state,
                "description": doc.name,
                "size": doc.size << 10,
                "time": {
                    "created": sandbox.web.helpers.utcdt2iso(doc.time.created),
                    "accessed": sandbox.web.helpers.utcdt2iso(doc.time.accessed),
                    "expires": (
                        sandbox.web.helpers.utcdt2iso(doc.time.accessed + dt.timedelta(days=ttl))
                        if ttl else
                        None
                    ),
                    "updated": sandbox.web.helpers.utcdt2iso(doc.time.updated),
                },
                "task": {
                    "id": doc.task_id,
                    "url": '{}/task/{}'.format(base_url.rsplit('/', 1)[0], doc.task_id),
                },
                "http": {
                    "links": resource.get_http_links() if not lightweight else [],
                    "proxy": resource.proxy_url
                },
                "attributes": {attr.key: attr.value for attr in doc.attributes},
                "skynet_id": resource.skynet_id,
                "md5": resource.file_md5 or None,
                "file_name": doc.path,
                "multifile": doc.multifile,
                "executable": doc.executable,
                "observable": observable,
                "resource_meta": resource_meta,
                "system_attributes": {
                    "linux_platform": doc.system_attributes.linux_platform,
                    "osx_platform": doc.system_attributes.osx_platform,
                    "osx_arm_platform": doc.system_attributes.osx_arm_platform,
                    "win_nt_platform": doc.system_attributes.win_nt_platform,
                } if doc.system_attributes else None
            })

    class ListItemEntry(BaseEntry):
        pass

    class Entry(BaseEntry):
        def __init__(self, user, base_url, doc, lightweight=False, resource_meta_hashes=None):
            resource = proxy_resource.Resource._restore(doc)
            super(Resource.Entry, self).__init__(
                user, base_url.rsplit('/', 1)[0], doc, resource, lightweight=lightweight,
                resource_meta_hashes=resource_meta_hashes
            )
            self.update(
                rsync={"links": resource.get_rsync_links() if not lightweight else []},
                sources=(
                    [_.hostname for _ in resource.sources]
                    if not lightweight else
                    [h.host for h in doc.hosts_states if h.state == ctr.HostState.OK]
                ),
            )
            task_query = mp.Task.objects(id=doc.task_id)
            self["task"]["status"] = task_query.scalar("execution__status").first() or ""
            mds = doc.mds
            if mds:
                self["mds"] = api.v1.schemas.resource.MDS.create(
                    key=mds.key,
                    namespace=mds.namespace,
                    url=common.mds.s3_link(mds.key, mds.namespace),
                    backup_url=(
                        common.mds.s3_link(mds.key, mds.backup_namespace)
                        if mds.backup_namespace else
                        ""
                    ),
                    __filter_empty__=True,
                )
            else:
                self["mds"] = None

    @staticmethod
    def _lightweight(r):
        # TODO: TESTENV-1416: A special crutch to speedup requests from TestEnv
        return (
            common.config.Registry().server.auth.enabled and
            not r.session and r.user.login in ("robot-testenv", "guest")
        ) or ctm.HTTPHeader.NO_LINKS in r.headers

    @staticmethod
    def _resource_meta(r):
        return ctm.HTTPHeader.RESOURCE_META in r.headers

    @classmethod
    def get(cls, request, obj_id):
        doc = cls._document(obj_id)
        if ctm.HTTPHeader.TOUCH_RESOURCE in request.headers:
            yasandbox.manager.resource_manager.touch(doc.id)
        resource_meta_hashes = {
            doc.hash: doc
            for doc in
            (
                mp.ResourceMeta.objects(hash__in=doc.resource_meta)
                if cls._resource_meta(request) and doc.resource_meta
                else []
            )
        }
        return misc.response_json(cls.Entry(
            request.user, request.uri, doc, lightweight=cls._lightweight(request),
            resource_meta_hashes=resource_meta_hashes
        ))

    @classmethod
    def _list(cls, request, default_order=None, forced_limit=None):
        # Parse query arguments and form them as keyword arguments to database query builder
        try:
            kwargs, offset, limit = cls._handle_args(request, multi_order=True)
            if forced_limit:
                limit = forced_limit
        except (TypeError, ValueError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Query parameter validation error: " + str(ex))
        if limit is None:
            return misc.json_error(httplib.BAD_REQUEST, "Required parameter 'limit' not provided.")
        if limit > 3000 and not request.user.super_user:
            return misc.json_error(httplib.BAD_REQUEST, "Too big amount of data requested.")

        order = kwargs.pop('order_by', None) or (None if 'id' in kwargs else default_order)
        attrs = {}
        raw_attrs = kwargs.pop('attrs', None)
        if raw_attrs:
            try:
                attrs = json.loads(raw_attrs)
            except ValueError as ex:
                return misc.json_error(
                    httplib.BAD_REQUEST,
                    'Bad json value for "attrs": {}, Error: {}'.format(raw_attrs, ex)
                )

            if isinstance(attrs, list):
                attrs = type("", (type(attrs),), {"iteritems": lambda _: iter(_)})(attrs)
            elif not isinstance(attrs, dict):
                return misc.json_error(
                    httplib.BAD_REQUEST,
                    'Parameter "attrs" must be dict or list of pairs, not: {!r}'.format(type(attrs))
                )

        attr_name = kwargs.pop('attr_name', None)
        attr_value = kwargs.pop('attr_value', None)
        if kwargs.get("owner", "") == "SUGGEST" or kwargs.get("resource_type", "") == "LM_DUMPS_LIST":
            # KORUM: TODO: Send heavy list requests to secondary database nodes.
            cls.logger.info("Send SUGGEST/LM_DUMPS_LIST resources list to SECONDARY")
            request.read_preference = ctd.ReadPreference.SECONDARY

        if attr_name:
            attrs[attr_name] = attr_value
        any_attr = kwargs.pop('any_attr', False)
        if any_attr:
            kwargs['any_attrs'] = attrs
        elif attrs:
            kwargs['all_attrs'] = attrs

        dependant = kwargs.pop('dependant', None)
        if dependant:
            ids = list(mp.Task.objects(id=dependant).scalar('requirements__resources'))
            if ids and ids[0]:
                kwargs['id'] = ids[0]
        # Fetch all the document records here
        query = cls.Model.objects(**yasandbox.manager.resource_manager.list_query(**kwargs))

        if cls._lightweight(request):
            # Sources are only needed for http/rsync links, they are not displayed in lightweight mode.
            query = query.exclude("hosts_states")

        total = query.count()
        if order:
            query = query.order_by(*common.utils.chain(order))
        return limit, offset, total, list((query if not offset else query.skip(offset)).limit(limit))

    @classmethod
    def list(cls, request):
        limit, offset, total, docs = cls._list(request, default_order='-id')
        resource_meta_hashes = None
        if cls._resource_meta(request):
            hashes = set()
            for doc in docs:
                if doc.resource_meta:
                    hashes.update(doc.resource_meta)
            resource_meta_hashes = {
                doc.hash: doc
                for doc in mp.ResourceMeta.objects(hash__in=list(hashes))
            }

        items = [
            cls.ListItemEntry(
                request.user, request.uri, doc, lightweight=cls._lightweight(request),
                resource_meta_hashes=resource_meta_hashes
            )
            for doc in docs
        ]

        task_query = mp.Task.objects(id__in=set(doc.task_id for doc in docs))
        task_statuses = {tid: status for (tid, status) in task_query.scalar("id", "execution__status")}

        for item in items:
            item["task"]["status"] = task_statuses.get(item["task"]["id"], "")

        return misc.response_json({
            "limit": limit,
            "offset": offset,
            "total": total,
            "items": items,
        })

    @classmethod
    def create(cls, request):
        if not request.session:
            return misc.json_error(httplib.FORBIDDEN, "The method is allowed only for task session scope.")

        data = misc.request_data(request)
        data.setdefault("arch", ctm.OSFamily.ANY)
        data.setdefault("state", ctr.State.NOT_READY)

        task_obj = mp.Task.objects.with_id(request.session.task)

        attributes = data.get("attributes")
        for_parent = data.get("for_parent")

        if for_parent:
            if not task_obj.parent_id:
                return misc.json_error(
                    httplib.FORBIDDEN,
                    "Cannot create resource for parent of task #{}: task has no parent".format(request.session.task)
                )
            task_id = task_obj.parent_id
        else:
            task_id = request.session.task

        resource_meta = data.get("resource_meta", None)
        if resource_meta is None:
            if data.get("type") not in sdk2.Resource:
                return misc.json_error(httplib.BAD_REQUEST, "Unknown resource type {!r}.".format(data.get("type")))
            resource_meta = sdk2.Resource[data["type"]].__getstate__()
        try:
            resource = controller.Resource.create(
                data.get("description"), data.get("file_name", ""), data.get("md5"), data.get("type"), task_id,
                resource_meta, task_obj,
                skynet_id=data.get("skynet_id"),
                attrs=attributes,
                state=data["state"],
                arch=data["arch"],
                mds=data.get("mds"),
                multifile=bool(data.get("multifile")),
                executable=bool(data.get("executable")),
                system_attributes=data.get("system_attributes")
            )
        except common.errors.TaskError as ex:
            return misc.json_error(httplib.BAD_REQUEST, str(ex))
        return sandbox.web.helpers.response_created(
            "{}/{}".format(request.uri, resource.id),
            content_type="application/json",
            content=json.dumps(
                cls.Entry(request.user, request.uri, resource),
                ensure_ascii=False, encoding="utf-8", cls=common.rest.Client.CustomEncoder
            )
        )

    @classmethod
    def _update_fields(cls, resource, data, request, full_update=True, db_update=True):
        checks = (
            ("description", "name", lambda _: str(_) or True, None),
            (
                "attributes",
                "attrs",
                lambda _: dict(_) or True,
                None
            ),
        )
        if full_update:
            checks += (
                ("skynet_id", None, str, None),
                ("file_name", None, str, None),
                ("md5", "file_md5", str, None),
                ("state", None, lambda _: _ in ctr.State, None),
                ("arch", None, lambda _: _ in ctm.OSFamily, None),
                ("size", None, int, int),
                ("mds", None, api.v1.schemas.resource.MDS.decode, None),
                ("multifile", None, bool, bool),
                ("executable", None, bool, bool),
            )

        new_data = {}

        for key, attr_name, validate_func, prepare_func in checks:
            prepare_func = prepare_func or (lambda _: _)
            attr_name = attr_name or key
            try:
                if key in data and validate_func(data[key]):
                    new_data[key] = prepare_func(data[key])
            except (TypeError, ValueError) as ex:
                return misc.json_error(
                    httplib.BAD_REQUEST,
                    "Error validating property {!r} ({!r}, {!r}): {}.".format(key, attr_name, data[key], ex)
                )

        resource_meta = None
        if full_update:
            resource_meta = controller.Resource.resource_meta(resource)
            if resource_meta is None:
                resource.hashes, resource_meta = controller.Resource.update_meta_objects(
                    sdk2.Resource[resource.type].__getstate__()
                )
                resource_meta = resource_meta[0]
                resource.save()
        client_id = request.session.client if request.session else None
        return controller.Resource.update(
            resource, new_data, client_id, resource_meta, full_update=full_update, db_update=db_update
        )

    @classmethod
    def update(cls, request, res_id):
        obj = cls._document(res_id)
        data = misc.request_data(request) if request.raw_data else {}
        if not data:
            yasandbox.manager.resource_manager.touch(obj.id)
            return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

        allowed_fields = ("attributes", "description")
        full_update = any(_ not in allowed_fields for _ in data)

        if not request.session and request.user.super_user:
            request.session = sandbox.web.server.request.TaskSession(
                None, obj.task_id, "", ""
            )

        if not request.session:
            if not controller.user_has_permission(request.user, (obj.owner,)):
                return misc.json_error(
                    httplib.FORBIDDEN,
                    "User `{}` is not permitted to modify resource #{}".format(request.user.login, res_id)
                )

            if full_update:
                return misc.json_error(
                    httplib.FORBIDDEN,
                    "It's not allowed to modify {} fields beyond task session scope. "
                    "Modification of {} is permitted.".format(
                        sorted(set(data) - set(allowed_fields)), sorted(allowed_fields)
                    )
                )

        if request.session and full_update and not request.user.super_user:
            allowed_ids = mp.Task.objects(id=request.session.task).scalar("id", "parent_id")[0]
            if obj.task_id not in allowed_ids:
                return misc.json_error(
                    httplib.FORBIDDEN,
                    "Resource #{} created by task #{} cannot be changed in scope of task #{}.".format(
                        obj.id, obj.task_id, request.session.task
                    )
                )

            if obj.state != ctr.State.NOT_READY:
                try:
                    updated = cls._update_fields(obj, data, request, db_update=False)
                    if updated:
                        return misc.json_error(
                            httplib.BAD_REQUEST,
                            "Modification of resource #{} in state {!r} is not allowed.".format(obj.id, obj.state)
                        )

                    else:
                        cls.logger.warning("Resource #%s re-modification detected.", obj.id)
                        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)
                except (common.errors.TaskError, TypeError, ValueError) as ex:
                    return misc.json_error(httplib.BAD_REQUEST, str(ex))

        try:
            cls._update_fields(obj, data, request, full_update=full_update)
        except (common.errors.TaskError, TypeError, ValueError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, str(ex))
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def delete_source(cls, request, obj_id, host):
        resource = proxy_resource.Resource._restore(cls._document(obj_id))
        if not resource.user_has_permission(request.user):
            return misc.json_error(
                httplib.FORBIDDEN,
                'User `{}` is not permitted to delete resource #{}'.format(request.user.login, obj_id)
            )
        cls.Model.objects(id=cls._id(obj_id), hosts_states__host=host).update_one(
            set__hosts_states__S__state=cls.Model.HostState.State.MARK_TO_DELETE)
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def backup(cls, request, obj_id):
        obj = cls._document(obj_id)
        try:
            data = misc.request_data(request) if request.raw_data else {}
        except ValueError:
            # FIXME: Tempoary till SDK update on cluster
            data = {}
        task_id = controller.Resource.force_backup(obj.id, dc=data.get("dc"), size=data.get("size"))
        return misc.response_json({'id': task_id})

    @classmethod
    def __resource_hosts_response(cls, hosts, link_builder):
        storage_hosts = set(common.config.Registry().server.storage_hosts)
        return [
            {
                'host': host.hostname,
                'url': link_builder(host),
                'storage': host.hostname in storage_hosts,
                'alive': host.is_alive(),
                'state': ctr.HostState.OK,
            }
            for host in hosts
        ]

    @classmethod
    def get_resource_http_urls(cls, _, obj_id):
        doc = cls._document(obj_id)
        obj = proxy_resource.Resource._restore(doc)
        return misc.response_json(cls.__resource_hosts_response(obj.sources, obj.http_url))

    @classmethod
    def get_resource_rsync_urls(cls, _, obj_id):
        doc = cls._document(obj_id)
        obj = proxy_resource.Resource._restore(doc)
        return misc.response_json(cls.__resource_hosts_response(obj.sources, obj.rsync_url))


class ResourceAttribute(Base):
    """
    The class encapsulates all the logic related to REST API representation
    of any entities related to resource attribute object.
    """

    # Shortcuts for database models.
    Model = mp.Resource

    LIST_QUERY_MAP = None

    ListItemEntry = None

    INVALID_TTL_ERROR = "Attribute 'ttl' is not valid: should be either 'inf' or a positive number < 1000"

    class Entry(dict):
        def __init__(self, doc):
            super(ResourceAttribute.Entry, self).__init__({'name': doc.key, 'value': doc.value})

    @staticmethod
    def _is_valid_ttl(value):
        if value:
            try:
                value = float(value)
            except ValueError:
                return False
            return math.isinf(value) or 1 <= value < 1000
        return False

    @staticmethod
    def cast(value):
        return value.encode("utf-8") if isinstance(value, unicode) else str(value)

    @classmethod
    def get(cls, request, obj_id):
        doc = cls._document(obj_id)
        return misc.response_json([cls.Entry(attr) for attr in doc.attributes])

    @classmethod
    def create(cls, request, obj_id):
        attr = misc.request_data(request)
        if not attr or set(attr.keys()) ^ {'name', 'value'}:
            return misc.json_error(httplib.BAD_REQUEST, "Invalid parameter 'attr': {!r}".format(attr))
        attr_name = attr['name'].strip()
        attr_value = cls.cast(attr['value']).strip()
        res_doc = cls._document(obj_id)
        if any(a.key == attr_name for a in res_doc.attributes):
            return misc.json_error(httplib.CONFLICT, "Attribute '{}' already exists".format(attr_name))
        if attr_name == "ttl" and not cls._is_valid_ttl(attr_value):
            return misc.json_error(httplib.BAD_REQUEST, cls.INVALID_TTL_ERROR)

        attr_doc = cls.Model.Attribute(key=attr_name, value=attr_value)
        updated = cls.Model.objects(id=res_doc.id, attributes__key__ne=attr_name).update_one(
            push__attributes=attr_doc,
            set__time__updated=dt.datetime.utcnow()
        )
        if not updated:
            return misc.json_error(httplib.CONFLICT, "Attribute '{}' already exists".format(attr_name))

        return misc.response_json(cls.Entry(attr_doc))

    @classmethod
    def update(cls, request, obj_id, attr_name):
        data = misc.request_data(request)
        res_doc = cls._document(obj_id)
        attr_doc = next((a for a in res_doc.attributes if a.key == attr_name), None)
        if not attr_doc:
            return misc.json_error(httplib.NOT_FOUND, "Attribute '{}' does not exist".format(attr_name))
        if data.get("name"):
            attr_doc.key = data["name"]
        if "value" in data:
            attr_doc.value = cls.cast(data["value"])
        if attr_doc.key == "ttl" and not cls._is_valid_ttl(attr_doc.value):
            return misc.json_error(httplib.BAD_REQUEST, cls.INVALID_TTL_ERROR)

        updated = cls.Model.objects(id=res_doc.id, attributes__key=attr_name).update_one(
            set__attributes__S=attr_doc,
            set__time__updated=dt.datetime.utcnow()
        )
        if not updated:
            return misc.json_error(httplib.NOT_FOUND, "Attribute '{}' does not exist".format(attr_name))

        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def delete(cls, request, obj_id, attr_name):
        res_doc = cls._document(obj_id)
        if not any(a.key == attr_name for a in res_doc.attributes):
            return misc.json_error(httplib.NOT_FOUND, "Attribute '{}' does not exist".format(attr_name))
        cls.Model.objects(
            id=res_doc.id,
            attributes__key=attr_name
        ).update_one(
            pull__attributes__key=attr_name,
            set__time__updated=dt.datetime.utcnow()
        )
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)


class ResourceStorageInsufficientRedundancy(PathBase(api.v1.resource.ResourceInsufficientRedundancy)):
    @classmethod
    def get(cls, request, host):
        query = controller.Resource.insufficient_resources_query(host)
        total = query.count()
        offset = request.get("offset")
        limit = request.get("limit")
        offset, limit = map(lambda x: (int(x) if x is not None else x), (offset, limit))

        if offset:
            query = query.skip(offset)
        if limit is not None and limit < total - (offset if offset else 0):
            query = query.limit(limit)

        items = [
            api.v1.schemas.resource.ResourseInsufficientRedundancyItem.create(id=r[0], type=r[1], size=r[2] << 10)
            for r in query.scalar("id", "type", "size")
        ]

        return misc.response_json({
            "offset": offset,
            "limit": limit,
            "total": total,
            "items": items,
        })


class ResourceMeta(PathBase(api.v1.resource.ResourceMeta)):
    @classmethod
    def get(cls, _, rtype):
        try:
            resource_class = sdk2.Resource[rtype]
        except common.errors.UnknownResourceType:
            return misc.json_error(httplib.NOT_FOUND, "Resource type {} not found.".format(rtype))
        return misc.response_json(resource_class.__getstate__()[0])


class ResourceMetaList(PathBase(api.v1.resource.ResourceMetaList)):
    @classmethod
    def get(cls, _, rtype):
        try:
            resource_class = sdk2.Resource[rtype]
        except common.errors.UnknownResourceType:
            return misc.json_error(httplib.NOT_FOUND, "Resource type {} not found.".format(rtype))
        return misc.response_json(resource_class.__getstate__())


registry.registered_json('resource')(Resource.list)
registry.registered_json('resource', method=ctm.RequestMethod.POST)(Resource.create)
registry.registered_json('resource/(\d+)')(Resource.get)
registry.registered_json('resource/(\d+)', method=ctm.RequestMethod.PUT)(Resource.update)
registry.registered_json('resource/(\d+)/attribute')(ResourceAttribute.get)
registry.registered_json('resource/(\d+)/attribute', method=ctm.RequestMethod.POST)(ResourceAttribute.create)
registry.registered_json('resource/(\d+)/attribute/(\w+)', method=ctm.RequestMethod.PUT)(ResourceAttribute.update)
registry.registered_json('resource/(\d+)/attribute/(\w+)', method=ctm.RequestMethod.DELETE)(ResourceAttribute.delete)
registry.registered_json('resource/(\d+)/data/http')(Resource.get_resource_http_urls)
registry.registered_json('resource/(\d+)/data/rsync')(Resource.get_resource_rsync_urls)
# TODO: SANDBOX-2869: Drop this method after complete clients upgrade
registry.registered_json('resource/(\d+)/data/([\w\-]+)', method=ctm.RequestMethod.DELETE)(Resource.delete_source)
registry.registered_json('resource/(\d+)/source/([\w\-]+)', method=ctm.RequestMethod.DELETE)(Resource.delete_source)
registry.registered_json('resource/(\d+)/backup', method=ctm.RequestMethod.POST, restriction=ctu.Restriction.TASK)(
    Resource.backup
)
