import json
import pytz
import datetime as dt
import aniso8601
import itertools
import functools
import collections
import distutils.util

from six.moves import http_client as httplib

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu
import sandbox.common.types.scheduler as cts

import sandbox.yasandbox.api.json.task as api_task

from sandbox.yasandbox import controller
from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import Base
from sandbox.yasandbox.api.json import registry
from sandbox.yasandbox.api.json import list_arg_parser
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.controller import user as user_controller

import sandbox.web.helpers
import sandbox.web.response


SchedulerTask = collections.namedtuple("SchedulerTask", ["id", "status"])


class Scheduler(Base):

    Model = mapping.Scheduler
    TaskModel = mapping.Task

    MULTI_SELECT_FIELDS = ("id", "type", "status")

    LIST_QUERY_MAP = (
        Base.QueryMapping("id", "id", "id", list_arg_parser(int)),
        Base.QueryMapping("task_type", "type", None, list_arg_parser(str)),
        Base.QueryMapping("status", "status", None, list_arg_parser(str)),
        Base.QueryMapping("owner", "owner", None, str),
        Base.QueryMapping("author", "author", None, str),
        Base.QueryMapping("limit", "limit", None, int),
        Base.QueryMapping("offset", "offset", None, int),
        Base.QueryMapping("order", "order_by", None, str),
        Base.QueryMapping('tags', 'tags', 'tags', list_arg_parser(lambda x: str(x).upper())),
        Base.QueryMapping('all_tags', 'all_tags', None, distutils.util.strtobool),
    )

    @classmethod
    def _handle_args(cls, request):
        kwargs, offset, limit = super(Scheduler, cls)._handle_args(request)
        for field in cls.MULTI_SELECT_FIELDS:
            if field in kwargs:
                kwargs["{}__in".format(field)] = kwargs.pop(field)
        return kwargs, offset, limit

    class BaseEntry(dict):
        def __init__(self, user, base_url, doc):
            write_access = user_controller.user_has_permission(user, [doc.owner])
            super(Scheduler.BaseEntry, self).__init__({
                "id": doc.id,
                "rights": "write" if write_access else "read",
                "url": "{}/{}".format(base_url, doc.id),
                "author": doc.author,
                "owner": doc.owner,
                "status": doc.status,
                "schedule": Scheduler._get_schedule(doc),
                "scheduler_notifications": [
                    {
                        "statuses": list(ctt.Status.Group.collapse(n.statuses)),
                        "recipients": list(n.recipients),
                        "transport": n.transport,
                        "check_status": n.check_status,
                        "juggler_tags": list(n.juggler_tags)
                    }
                    for n in (doc.scheduler_notifications.notifications if doc.scheduler_notifications else [])
                ],
                "time": {
                    "created": sandbox.web.helpers.utcdt2iso(doc.time.created),
                    "updated": sandbox.web.helpers.utcdt2iso(doc.time.updated),
                    "next": sandbox.web.helpers.utcdt2iso(doc.time.next_run) if doc.time.next_run else None,
                    "last": sandbox.web.helpers.utcdt2iso(doc.time.last_run) if doc.time.last_run else None,
                }
            })

    class ListItemEntry(BaseEntry):
        def __init__(self, user, base_url, doc, last_task):
            super(Scheduler.ListItemEntry, self).__init__(user, base_url, doc)
            self.update({
                "task": {
                    "type": doc.type,
                    "description": common.utils.force_unicode_safe(doc.description),
                    "tags": doc.tags or [],
                    "last": {
                        "id": last_task.id,
                        "url": "{}/task/{}".format(base_url.rsplit("/", 1)[0], last_task.id),
                        "status": last_task.status,
                    } if last_task else None
                }
            })

    class Entry(ListItemEntry):
        def __init__(self, user, base_url, doc):
            last_task = mapping.Task.objects(scheduler=doc.id).order_by("-id").scalar("id", "execution__status").first()
            super(Scheduler.Entry, self).__init__(user, base_url, doc, last_task and SchedulerTask(*last_task))
            priority = ctt.Priority.make(doc.priority)
            task = controller.TaskWrapper(doc)

            # TODO: remove custom_fields from the response when SANDBOX-7793 will be done
            custom_fields = []
            try:
                custom_fields = api_task.Task.custom_fields_views(doc, input=True)
            except sandbox.web.response.HttpErrorResponse as ex:
                if ex.status_code != httplib.NOT_FOUND:
                    raise

            self["task"].update({
                "sdk_version": task.sdk_version,
                "owner": doc.owner,
                "priority": {
                    "class": priority.Class.val2str(priority.cls),
                    "subclass": priority.Subclass.val2str(priority.scls),
                },
                "tags": task.tags,
                "requirements": {
                    "disk_space": (doc.requirements.disk_space or 0) << 20,
                    "platform": doc.requirements.platform,
                    "cpu_model": doc.requirements.cpu_model,
                    "cores": doc.requirements.cores,
                    "host": doc.requirements.host,
                    "ram": (doc.requirements.ram or 0) << 20,
                    "ramdrive": (
                        {"type": doc.requirements.ramdrive.type, "size": doc.requirements.ramdrive.size << 20}
                        if doc.requirements.ramdrive else
                        None
                    ),
                    "container_resource": doc.requirements.container_resource,
                    "privileged": bool(doc.requirements.privileged),
                    "dns": doc.requirements.dns or ctm.DnsType.DEFAULT,
                    "client_tags": str(task.client_tags or ""),
                    "semaphores": (
                        ctt.Semaphores(**doc.requirements.semaphores.to_mongo()).to_dict()
                        if doc.requirements.semaphores else
                        None
                    )
                },
                "suspend_on_status": doc.suspend_on_status,
                "score": doc.score,
                "kill_timeout": task.kill_timeout,
                "fail_on_any_error": task.fail_on_any_error,
                "tasks_archive_resource": task.tasks_archive_resource,
                "hidden": doc.hidden,
                "notifications": [
                    {
                        "statuses": list(ctt.Status.Group.collapse(n.statuses)),
                        "recipients": list(n.recipients),
                        "transport": n.transport,
                        "check_status": n.check_status,
                        "juggler_tags": list(n.juggler_tags)
                    }
                    for n in doc.notifications
                ],
                "custom_fields": custom_fields,
            })

    @staticmethod
    def _get_schedule(scheduler):
        repetition = retry = None

        repetition_mode = scheduler.plan.repetition
        if repetition_mode == cts.Repetition.INTERVAL:
            repetition = {"interval": scheduler.plan.interval}
        elif repetition_mode == cts.Repetition.WEEKLY:
            days_of_week = scheduler.plan.days_of_week
            repetition = {"weekly": [i for i in range(7) if days_of_week & (1 << i)]}
        retry_mode = scheduler.plan.retry
        if retry_mode == cts.Retry.NO:
            retry = {"ignore": True}
        elif retry_mode == cts.Retry.INTERVAL:
            retry = {"interval": scheduler.plan.retry_interval}

        start_time = None
        if scheduler.plan.start_mode == cts.StartMode.SET:
            start_time = scheduler.plan.start_time
        return {
            "repetition": repetition,
            "retry": retry,
            "fail_on_error": (retry_mode == cts.Retry.FAILURE),
            "sequential_run": scheduler.plan.sequential_run,
            "start_time": start_time and sandbox.web.helpers.utcdt2iso(start_time),
        }

    @classmethod
    def get_last_tasks(cls, schedulers):
        schedulers_task = cls.TaskModel.last_task_per_scheduler([o.id for o in schedulers])
        tasks_statuses = dict(cls.TaskModel.objects(
            id__in=schedulers_task.values()).scalar("id", "execution__status"))
        return {key: SchedulerTask(value, tasks_statuses[value]) for key, value in schedulers_task.iteritems()}

    @classmethod
    def list(cls, request):
        try:
            kwargs, offset, limit = cls._handle_args(request)
        except (TypeError, ValueError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Query parameter validation error: " + str(ex))
        if limit is None:
            return misc.json_error(httplib.BAD_REQUEST, "Required parameter 'limit' not provided.")
        order = kwargs.pop("order_by", "-id")
        if "status__in" not in kwargs:
            kwargs["status__ne"] = "DELETED"

        tags = kwargs.pop("tags", None)
        all_tags = kwargs.pop("all_tags", False)
        if tags:
            tag_op = "__all" if all_tags else "__in"
            kwargs["tags" + tag_op] = list(common.utils.chain(tags))

        query = cls.Model.objects(**kwargs)
        total = query.count()
        if order:
            query = query.order_by(order)
        docs = (query if not offset else query.skip(offset)).limit(limit)
        schedulers_last_tasks = cls.get_last_tasks(docs)
        return misc.response_json({
            "limit": limit,
            "offset": offset,
            "total": total,
            "items": [
                cls.ListItemEntry(request.user, request.uri, doc, schedulers_last_tasks.get(doc.id))
                for doc in docs
            ]
        })

    @classmethod
    def get(cls, request, obj_id):
        doc = cls._document(obj_id)
        return misc.response_json(cls.Entry(request.user, request.uri.rsplit("/", 1)[0], doc))

    @classmethod
    def _check_right_to_act_on_behalf_of(cls, user, requested_login):  # type: (mapping.User, str) -> None
        if not user_controller.user_has_right_to_act_on_behalf_of(user, requested_login):
            return misc.json_error(
                httplib.FORBIDDEN,
                "It's allowed to change scheduler's author to yourself or a robot you own (ABC scope robots_management)"
            )

    @classmethod
    def create(cls, request):
        data = misc.request_data(request)

        task_type, source, up_data = map(data.get, ("task_type", "source", "data"))
        if not task_type and not source:
            return misc.json_error(httplib.BAD_REQUEST, "No task type or source ID specified")
        if task_type and source:
            return misc.json_error(httplib.BAD_REQUEST, "Either task type or source ID should be specified")

        author = (up_data or {}).get("author")
        if author:
            cls._check_right_to_act_on_behalf_of(request.user, author)
        else:
            author = request.user.login

        if source:
            new_sch = controller.Scheduler.copy(source, author)
            cls.Model.objects.filter(id=new_sch.id).update(
                set__notifications=cls.Model.objects.scalar("notifications").with_id(source)
            )
        else:
            new_sch = controller.Scheduler.create(task_type, author, author, up_data)

        if up_data:
            cls._update_scheduler(new_sch, up_data)

        return sandbox.web.helpers.response_created(
            "{}/{}".format(request.uri, new_sch.id),
            content_type="application/json",
            content=json.dumps(
                cls.Entry(request.user, request.uri, cls.Model.objects.with_id(new_sch.id)),
                ensure_ascii=False, encoding="utf-8"
            )
        )

    @classmethod
    def update(cls, request, sch_id):
        scheduler = cls._document(sch_id)

        # TODO: forbid to update in case of `scheduler.author != request.user` SANDBOX-9507
        if not user_controller.user_has_permission(request.user, [scheduler.owner]):
            return misc.json_error(
                httplib.FORBIDDEN, "User {} not permitted to edit scheduler".format(request.user.login)
            )

        data = misc.request_data(request)

        author = data.get("author")
        if author and author != scheduler.author:
            cls._check_right_to_act_on_behalf_of(request.user, author)

        cls._update_scheduler(scheduler, data)
        mapping.Scheduler.objects(id=scheduler.id).update_one(set__time__updated=dt.datetime.utcnow())
        return (
            misc.response_json(cls.Entry(request.user, request.uri, cls._document(scheduler.id)))
            if cls.request_needs_updated_data(request) else
            sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)
        )

    @classmethod
    def get_custom_fields(cls, _, sch_id):
        return misc.response_json(
            api_task.Task.custom_fields_views(cls._document(sch_id), input=True)
        )

    @classmethod
    def validate_custom_fields(cls, request, sch_id):
        doc = cls._document(sch_id)
        return misc.response_json(map(
            dict,
            api_task.Task.update_and_validate_custom_fields(
                request, controller.TaskWrapper(doc), input_data=misc.request_data(request, list), validate_only=True
            )[1]
        ))

    @classmethod
    def update_custom_fields(cls, request, sch_id):
        scheduler = cls._document(sch_id)
        template = controller.TaskWrapper(scheduler)
        _, results = api_task.Task.update_and_validate_custom_fields(
            request, template, input_data=misc.request_data(request, list)
        )
        template.update_model()
        controller.Scheduler.save(scheduler)
        mapping.Scheduler.objects(id=scheduler.id).update_one(set__time__updated=dt.datetime.utcnow())
        return misc.response_json(map(dict, results))

    @staticmethod
    def _update_scheduler_schedule(sch, schedule):
        if "repetition" not in schedule:
            return misc.json_error(httplib.BAD_REQUEST, "schedule.repetition field is required")
        repetition = schedule.get("repetition")
        interval, weekly = map(repetition.get, ("interval", "weekly")) if repetition else (None, None)
        if repetition is None:
            sch.plan.repetition = cts.Repetition.NO
        elif interval is not None and weekly is not None:
            return misc.json_error(
                httplib.BAD_REQUEST, "Either repetition.interval or repetition.weekly should be specified."
            )
        elif weekly is not None:
            value = 0
            for day in itertools.imap(functools.partial(common.utils.force_int, default=42), weekly):
                if day < 0 or day > 6:
                    return misc.json_error(
                        httplib.BAD_REQUEST, "Incorrect day number. The value must be a digit from 0 to 6."
                    )
                value |= (1 << day)
            if value == 0:
                return misc.json_error(httplib.BAD_REQUEST, "At least one day must be selected.")
            sch.plan.repetition = cts.Repetition.WEEKLY
            sch.plan.days_of_week = value
        elif interval is not None:
            sch.plan.repetition = cts.Repetition.INTERVAL
            sch.plan.interval = common.utils.force_int(interval)
            if sch.plan.interval <= 0:
                return misc.json_error(httplib.BAD_REQUEST, "Repetition interval must be positive.")
        else:
            return misc.json_error(
                httplib.BAD_REQUEST, "Expect repetition.interval or repetition.weekly, but both are missing."
            )

        start_time = schedule.get("start_time")
        if start_time:
            sch.plan.start_mode = cts.StartMode.SET
            sch.plan.start_time = aniso8601.parse_datetime(start_time)
            if sch.plan.start_time.tzinfo:
                sch.plan.start_time = sch.plan.start_time.astimezone(pytz.utc).replace(tzinfo=None)
        else:
            sch.plan.start_mode = cts.StartMode.IMMEDIATELY

        sequential_run = schedule.get("sequential_run")
        if sequential_run is not None:
            sch.plan.sequential_run = sequential_run

        if schedule.get("fail_on_error"):
            sch.plan.retry = cts.Retry.FAILURE
            return
        retry = schedule.get("retry")
        if not retry:
            return
        ignore, interval = map(retry.get, ("ignore", "interval",))
        if (not ignore and not interval) or (ignore and interval):
            return misc.json_error(httplib.BAD_REQUEST, "Either retry.ignore or retry.interval should be specified.")
        if interval:
            sch.plan.retry = cts.Retry.INTERVAL
            sch.plan.retry_interval = common.utils.force_int(interval)
            if sch.plan.retry_interval <= 0:
                return misc.json_error(httplib.BAD_REQUEST, "Retry interval must be positive.")
        elif ignore:
            sch.plan.retry = cts.Retry.NO

    @classmethod
    def _update_scheduler(cls, scheduler, data):  # type: (mapping.Scheduler, dict) -> None
        author, owner, schedule, task, scheduler_notifications = map(
            data.get, ("author", "owner", "schedule", "task", "scheduler_notifications")
        )

        if owner or author:
            new_author = author or scheduler.author
            new_owner = owner or scheduler.owner
            if not user_controller.user_has_permission(new_author, [new_owner]):
                return misc.json_error(
                    httplib.FORBIDDEN,
                    "User {} is not permitted to own scheduler with group {}.".format(new_author, new_owner),
                )
            scheduler.author = new_author
            scheduler.owner = new_owner

        try:
            if scheduler_notifications is not None:
                controller.Scheduler.set_notifications(scheduler, scheduler_notifications)
            if schedule:
                cls._update_scheduler_schedule(scheduler, schedule)
                scheduler.time.next_run = controller.Scheduler.get_next_task_creation_time(scheduler, manual=True)
            if task:
                controller.Task.update_tasks_resource(task, scheduler)
                template = controller.TaskWrapper(scheduler)
                template.update_common_fields(task)
                if "notifications" in task:
                    cls.Model.objects.filter(id=scheduler.id).update(
                        set__notifications=controller.Task.notifications(task["notifications"]),
                    )
                custom_fields = task.get("custom_fields")
                requirements = task.get("requirements")
                if custom_fields or requirements:
                    api_task.Task.update_and_validate_custom_fields(
                        None, template, input_data=custom_fields, requirements_data=requirements
                    )
                template.update_model()
        except ValueError as ex:
            misc.json_error(httplib.BAD_REQUEST, str(ex))
        controller.Scheduler.save(scheduler)


registry.registered_json("scheduler")(Scheduler.list)
registry.registered_json(
    "scheduler", ctm.RequestMethod.POST, restriction=ctu.Restriction.AUTHENTICATED
)(Scheduler.create)
registry.registered_json("scheduler/(\d+)")(Scheduler.get)
registry.registered_json(
    "scheduler/(\d+)", ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED
)(Scheduler.update)
registry.registered_json("scheduler/(\d+)/custom/fields")(Scheduler.get_custom_fields)
registry.registered_json(
    "scheduler/(\d+)/custom/fields", ctm.RequestMethod.PUT, restriction=ctu.Restriction.AUTHENTICATED
)(Scheduler.update_custom_fields)
registry.registered_json("scheduler/(\d+)/custom/fields", ctm.RequestMethod.POST)(Scheduler.validate_custom_fields)
