import re
import json
import urllib
import httplib
import datetime as dt

import distutils.util

from sandbox import common
import sandbox.common.types.user as ctu
import sandbox.common.types.misc as ctm

import yasandbox.controller
import yasandbox.controller.user

import yasandbox.database.mapping

from sandbox.web import api
from sandbox.yasandbox.api.json import Base
from sandbox.yasandbox.api.json import PathBase
from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import registry

import sandbox.web.helpers
import sandbox.web.response


###########################################################
# API Version 1.0
###########################################################
class Semaphore(Base):
    """
    The class encapsulates all the logic related to REST API representation of any entities related to semaphore object
    """

    # Shortcuts for database models
    Model = yasandbox.database.mapping.Semaphore

    # A list of list operation query parameters mapping
    LIST_QUERY_MAP = (
        Base.QueryMapping("limit", "limit", None, int),
        Base.QueryMapping("offset", "offset", None, int),
        Base.QueryMapping("order", "order_by", None, str),
        Base.QueryMapping("name", "name", None, str),
        Base.QueryMapping("group", "group", None, str),
        Base.QueryMapping("owner", "owner", None, str),
        Base.QueryMapping("shared", "shared", None, str),
        Base.QueryMapping("auto", "auto", None, lambda _: bool(distutils.util.strtobool(_))),
    )

    class BaseEntry(dict):
        def __init__(self, user, base_url, doc, value, groups):
            super(Semaphore.BaseEntry, self).__init__(
                id=doc.id,
                rights=yasandbox.controller.Semaphore.rights(user, doc, groups),
                name=doc.name,
                owner=doc.owner,
                url="{}/{}".format(base_url, doc.id),
                shared=doc.shared,
                capacity=doc.capacity,
                value=value,
                auto=doc.auto,
                public=doc.public,
                time=dict(
                    created=sandbox.web.helpers.utcdt2iso(doc.time.created),
                    updated=sandbox.web.helpers.utcdt2iso(doc.time.updated)
                )
            )

    class ListItemEntry(BaseEntry):
        pass

    class Task(dict):
        def __init__(self, task_weight):
            super(Semaphore.Task, self).__init__(task_id=task_weight[0], weight=task_weight[1])

    class Entry(BaseEntry):
        def __init__(self, user, base_url, doc, value, groups):
            super(Semaphore.Entry, self).__init__(user, base_url, doc, value, groups)
            tasks = yasandbox.controller.TaskQueue.qclient.semaphore_tasks(doc.id)
            self["tasks"] = map(Semaphore.Task, tasks)
            if doc.auto:
                return
            settings = common.config.Registry().server.solomon
            solomon_settings = {
                "project": settings.push.project,
                "cluster": settings.push.cluster,
                "service": settings.push.service,
                "type": "semaphore_capacity|semaphore_value|semaphore_blocked",
                "sensor": doc.name
            }
            if all(solomon_settings.itervalues()):
                self["statistics_url"] = "{}?{}".format(settings.url, urllib.urlencode(solomon_settings))

    @staticmethod
    def __check_owner_setup(owner, user, user_groups):
        if not yasandbox.controller.Group.exists(owner):
            return misc.json_error(httplib.BAD_REQUEST, "Owner must be an existing group")
        return (
            None
            if yasandbox.controller.Semaphore.can_modify(user, owner, user_groups) else
            misc.json_error(
                httplib.FORBIDDEN,
                "User '{}' has no permissions to set '{}' as owner".format(user.login, owner)
            )
        )

    @staticmethod
    def __check_shared(data):
        shared = data.get("shared")
        if shared:
            try:
                yasandbox.controller.user.validate_credentials(shared, groups=True)
            except Exception as ex:
                return misc.json_error(httplib.BAD_REQUEST, "Validation error: {}".format(ex))

    @staticmethod
    def __check_user_permissions(doc, user, user_groups):
        if not yasandbox.controller.Semaphore.can_modify(user, doc.owner, user_groups):
            return misc.json_error(
                httplib.FORBIDDEN,
                "User '{}' not permitted to modify semaphore".format(user.login)
            )

    @classmethod
    def list(cls, request):
        # Parse query arguments and form them as keyword arguments to database query builder
        try:
            kwargs, offset, limit = cls._handle_args(request)
        except (TypeError, ValueError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Query parameter validation error: {}".format(ex))
        if limit is None:
            return misc.json_error(httplib.BAD_REQUEST, "Required parameter 'limit' not provided.")

        order = kwargs.pop("order_by", None)
        user_groups = yasandbox.controller.Group.get_user_groups(request.user.login)
        name = kwargs.get("name")
        if name:
            try:
                kwargs["name"] = re.compile(name, re.IGNORECASE)
            except re.error as error:
                misc.json_error(httplib.BAD_REQUEST, "Incorrect regex: {} \n {}".format(name, error))
        group = kwargs.pop("group", None)
        sem_ids, groups = None, []
        if group:
            sem_ids, groups = yasandbox.controller.TaskQueue.qclient.semaphore_group(group)
            if not sem_ids and not groups:
                return misc.json_error(httplib.BAD_REQUEST, "Semaphore group '{}' does not exist".format(group))
            if sem_ids:
                kwargs["id__in"] = sem_ids
        if sem_ids is None or sem_ids:
            query = cls.Model.objects(yasandbox.database.mapping.Q(**kwargs))
            total = query.count()
            if order:
                query = query.order_by(order)
            docs = list((query if not offset else query.skip(offset)).limit(limit))
            values = yasandbox.controller.TaskQueue.qclient.semaphore_values(map(lambda _: _.id, docs))
        else:
            docs, values, total = [], [], 0
        return misc.response_json({
            "limit": limit,
            "offset": offset,
            "total": total,
            "items": [
                cls.ListItemEntry(request.user, request.uri, doc, value, user_groups)
                for doc, value in zip(docs, values)
            ],
            "groups": groups
        })

    @classmethod
    def get(cls, request, obj_id):
        doc = yasandbox.controller.Semaphore.get(obj_id)
        if not doc:
            return misc.json_error(httplib.NOT_FOUND, "Semaphore #{} not found".format(obj_id))
        user_groups = yasandbox.controller.Group.get_user_groups(request.user.login)
        value = yasandbox.controller.TaskQueue.qclient.semaphore_values([doc.id])[0]
        return misc.response_json(cls.Entry(request.user, request.uri.rsplit("/", 1)[0], doc, value, user_groups))

    @classmethod
    def create(cls, request):
        data = misc.request_data(request)
        fields = "name", "owner"
        if any(field not in data for field in fields):
            return misc.json_error(
                httplib.BAD_REQUEST,
                "Field(s): {} - is(are) required".format(", ".join(field for field in fields if field not in data))
            )
        name, owner = (data[key] for key in fields)
        user_groups = yasandbox.controller.Group.get_user_groups(request.user.login)
        for result in (cls.__check_owner_setup(owner, request.user, user_groups), cls.__check_shared(data)):
            if result:
                return result

        fields = {"name": name, "owner": owner}
        if "capacity" in data:
            fields["capacity"] = data["capacity"]
        if "shared" in data:
            fields["shared"] = data["shared"]
        if "public" in data:
            fields["public"] = data["public"]
        try:
            doc = yasandbox.controller.Semaphore.create(fields)
        except yasandbox.controller.Semaphore.Exception as ex:
            return misc.json_error(httplib.BAD_REQUEST, str(ex))
        return sandbox.web.helpers.response_created(
            "{}/{}".format(request.uri, doc.id),
            content_type="application/json",
            content=json.dumps(
                cls.Entry(request.user, request.uri, doc, 0, user_groups),
                ensure_ascii=False, encoding="utf-8"
            )
        )

    @classmethod
    def update(cls, request, obj_id):
        fields = misc.request_data(request)
        doc = yasandbox.controller.Semaphore.get(obj_id)
        if not doc:
            return misc.json_error(httplib.NOT_FOUND, "Semaphore #{} not found".format(obj_id))
        user_groups = yasandbox.controller.Group.get_user_groups(request.user.login)
        checks = (
            cls.__check_user_permissions(doc, request.user, user_groups),
            cls.__check_owner_setup(fields["owner"], request.user, user_groups) if "owner" in fields else None,
            cls.__check_shared(fields)
        )
        for result in checks:
            if result:
                return result
        event = fields.pop("event", None)
        updatable_fields = {"owner", "shared", "capacity", "auto", "public"}
        extra_fields = fields.viewkeys() - updatable_fields
        if extra_fields:
            return misc.json_error(httplib.BAD_REQUEST, "Cannot update fields: {}".format(list(extra_fields)))
        try:
            doc = yasandbox.controller.Semaphore.update(doc.id, fields, event=event)
        except yasandbox.controller.Semaphore.NotExists as ex:
            return misc.json_error(httplib.NOT_FOUND, str(ex))
        except (yasandbox.controller.Semaphore.Exception, TypeError, ValueError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, str(ex))
        value = yasandbox.controller.TaskQueue.qclient.semaphore_values([doc.id])[0]
        return (
            misc.response_json(cls.Entry(request.user, request.uri, doc, value, user_groups))
            if cls.request_needs_updated_data(request) else
            sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)
        )

    @classmethod
    def delete(cls, request, obj_id):
        doc = yasandbox.controller.Semaphore.get(obj_id)
        if not doc:
            return misc.json_error(httplib.NOT_FOUND, "Semaphore #{} not found".format(obj_id))
        user_groups = yasandbox.controller.Group.get_user_groups(request.user.login)
        result = cls.__check_user_permissions(doc, request.user, user_groups)
        if result:
            return result
        if not yasandbox.controller.TaskQueue.qclient.delete_semaphore(doc.id):
            return misc.json_error(httplib.LOCKED, "Semaphore #{} is busy".format(obj_id))
        yasandbox.controller.Semaphore.audit(obj_id, "Deleted")
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)


class SemaphoreAudit(PathBase(api.v1.semaphore.SemaphoreAudit)):
    Model = yasandbox.database.mapping.SemaphoreAudit

    @classmethod
    def get(cls, _, semaphore_id):
        ret = []
        for doc in cls.Model.objects(semaphore_id=semaphore_id).order_by("+time"):
            item = api.v1.schemas.semaphore.SemaphoreAuditItem.create()
            item.description = doc.description
            item.time = doc.time or dt.datetime.utcnow()
            item.target = doc.target
            item.author = doc.author
            item.source = doc.source
            ret.append(item)
        return misc.response_json(ret)


registry.registered_json("semaphore")(Semaphore.list)
registry.registered_json("semaphore", ctm.RequestMethod.POST, ctu.Restriction.AUTHENTICATED)(Semaphore.create)
registry.registered_json("semaphore/(\d+)")(Semaphore.get)
registry.registered_json("semaphore/(\d+)", ctm.RequestMethod.PUT, ctu.Restriction.AUTHENTICATED)(Semaphore.update)
registry.registered_json("semaphore/(\d+)", ctm.RequestMethod.DELETE, ctu.Restriction.AUTHENTICATED)(Semaphore.delete)
