import time
import json
import httplib
import datetime as dt
import collections

from sandbox import common

import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.services import update_sandbox_resources

from sandbox.yasandbox import controller

from sandbox.yasandbox.api.json import Base
from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import registry

import sandbox.web.helpers
import sandbox.web.response


class TasksStatistics(object):
    @classmethod
    def amount_per_status(cls, request, type_):
        stats = controller.Statistics.get_statistics(controller.Statistics.Keys.STATUS).get(type_)
        if not stats:
            return misc.json_error(404, "Path not found.")

        return misc.response_json({
            str(group).lower(): {k: stats[k] for k in map(str.lower, group)}
            for group in ctt.Status.Group
            if group.primary
        })


registry.registered_json("service/statistics/task/status/([^/]+)")(TasksStatistics.amount_per_status)


class StorageStatistics(object):
    _cache = None

    @classmethod
    def external_backup_size(cls, request):
        # TODO: Temporary disabled MDS statistics collector.
        return misc.response_json({"immortal_resource_size": 0, "external_backup_size": 0})
        now = dt.datetime.now()
        if not cls._cache or cls._cache["time"] < now - dt.timedelta(minutes=15):
            immortal_resources_size = sum(mapping.Resource.objects(
                state=ctr.State.READY,
                attributes=mapping.Resource.Attribute(key="ttl", value="inf"),
                hosts_states__host__in=[
                    c.hostname for c in controller.Client.list() if ctc.Tag.STORAGE in c.tags_set
                ]
            ).scalar("size")) << 10

            mds_backup_size = sum(mapping.Resource.objects(
                state=ctr.State.READY,
                attributes__key="mds"
            ).scalar("size")) << 10

            cls._cache = {
                "time": now,
                "data": {
                    "immortal_resource_size": immortal_resources_size,
                    "external_backup_size": mds_backup_size
                }
            }

        return misc.response_json(cls._cache['data'])


registry.registered_json("service/statistics/storage/external_backup")(StorageStatistics.external_backup_size)


class Status(object):
    @classmethod
    def echo(cls, request):
        return misc.response_json({
            "method": request.method,
            "headers": request.headers,
            "arguments": request.params,
            "data": request.raw_data,
        })

    @staticmethod
    def _server_info():
        from sandbox.web import server as web_server
        pool = web_server.workers.Workers()
        return pool, pool.statistics

    @classmethod
    def server(cls, request):
        """
        Shows server status. Uptime counted in seconds, memory - in KiBs.
        """
        pool, st = cls._server_info()
        settings = common.config.Registry()
        processes = {
            name: {
                'cpu': {k: dict(cpu) for k, cpu in p.cpu.avg().iteritems()},
                'memory': dict(p.mem)
            } for name, p in st.processes if name != 'workers'
        }
        processes['workers'] = [{
            'cpu': {k: dict(cpu) for k, cpu in p.cpu.avg().iteritems()},
            'memory': dict(p.mem)
        } for p in st.processes.workers]
        ret = {
            'host': settings.this.id,
            'uptime': round(time.time() - st.started, 2),
            'requests': {
                'count': dict(st.requests),
                'rate': st.rps.avg(),
                'queued': pool.qsize,
                'processing': pool.processing,
            },
            'processes': processes,
        }

        data = request.params
        show_iprogress = 'inprogress' in data
        show_last = 'last' in data

        if show_iprogress:
            data = ret['requests']['inprogress'] = sorted(
                (req.measure_stat_data for _, req in pool.in_progress.items()),
                reverse=True, key=lambda _: _.get("duration", 0)
            )
            summary = collections.Counter()
            for req in data:
                summary[req["remote_method"]] += req["duration"]
            ret["requests"]["summary"] = summary
        if show_last:
            ret['requests']['window_size'] = pool.last_requests.period.total_seconds()
            ret['requests']['last'] = [req.measure_stat_data for req in pool.last_requests]

        return misc.response_json(ret)

    @classmethod
    def solomon_metrics(cls, request):
        """Server metrics in solomon format"""
        pool, st = cls._server_info()
        raw_metrics = []
        solomon_metrics = {"metrics": []}

        raw_metrics.append(("workers_in_progress", pool.processing))
        raw_metrics.append(("worker_processes_total", len(st.processes.workers)))
        raw_metrics.append(("worker_threads_total", pool.threads_pool_size))
        raw_metrics.append(("requests_in_queue", pool.qsize))
        for k, v in st.requests:
            raw_metrics.append(("requests_processed_" + k, v))

        for k, v in raw_metrics:
            solomon_metrics["metrics"].append({"labels": {"name": "legacy_" + k}, "value": v})

        return misc.response_json(solomon_metrics)


registry.registered_json("service/status/server")(Status.server)
registry.registered_json("service/status/server_solomon")(Status.solomon_metrics)

for m in ctm.RequestMethod:
    registry.registered_json("service/echo", method=m)(Status.echo)


class UINotification(Base):
    Model = mapping.UINotification

    class Entry(dict):
        def __init__(self, doc):
            super(UINotification.Entry, self).__init__({
                'id': str(doc.id), 'severity': doc.severity, 'content': doc.content})

    @classmethod
    def _id(cls, obj_id):
        return str(obj_id)

    @classmethod
    def create(cls, request):
        data = misc.request_data(request)
        severity, content = map(data.get, ("severity", "content"))
        if not (severity and content):
            return misc.json_error(httplib.BAD_REQUEST, "severity and content fields are required")
        new_notification = cls.Model(severity=severity, content=content)
        new_notification.save(force_insert=True)
        return sandbox.web.helpers.response_created(
            "{}/{}".format(request.uri, str(new_notification.id)),
            content=json.dumps(cls.Entry(new_notification), ensure_ascii=False, encoding="utf-8"),
            content_type="application/json",
        )

    @classmethod
    def update(cls, request, doc_id):
        doc = cls._document(doc_id)
        data = misc.request_data(request)
        for field in ("severity", "content"):
            if field in data:
                setattr(doc, field, data[field])
        doc.save()
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def delete(cls, _, doc_id):
        cls._document(doc_id).delete()
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)


registry.registered_json("service/ui/notification", ctm.RequestMethod.POST, ctu.Restriction.ADMIN)(
    UINotification.create
)
registry.registered_json("service/ui/notification/([a-zA-Z0-9]+)", ctm.RequestMethod.PUT, ctu.Restriction.ADMIN)(
    UINotification.update
)
registry.registered_json("service/ui/notification/([a-zA-Z0-9]+)", ctm.RequestMethod.DELETE, ctu.Restriction.ADMIN)(
    UINotification.delete
)


class Shortify(object):
    @classmethod
    def shortify(cls, request):
        import library.format

        data = set(misc.request_data(request, list))
        return sandbox.web.response.HttpResponse(
            code=201, content_type="text/plain",
            content=library.format.formatHosts(
                _.fqdn for _ in controller.Client.list() if _.hostname in data
            )
        )


registry.registered_json("service/shortify/client", ctm.RequestMethod.POST)(Shortify.shortify)


class ServiceResources(object):
    """
    This class is used to provide REST API to access commonly requested resources
    """

    @classmethod
    def get(cls, request):
        resource_type = request.get("type")
        platform = request.get("platform") or ""
        version = request.get("version") or ""
        up = update_sandbox_resources.UpdateSandboxResources
        return misc.response_json(up.get_env_resource(resource_type, platform, version))


registry.registered_json("service/resources")(ServiceResources.get)
