import os
import shutil
import httplib
import operator
import collections

from sandbox import common
from sandbox import sdk2

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import task
from sandbox.yasandbox.api.json import registry


class Index(object):
    @common.utils.singleton_classproperty
    def location(cls):
        from sandbox import projects
        return os.path.join(os.path.dirname(projects.__file__), common.config.Registry().common.task.index_subdir)

    def __init__(self):
        self.cleanup()
        self.tidx = common.quicksearch.Index("tasks", self.location)
        self.ridx = common.quicksearch.Index("resources", self.location)
        if not self.tidx.exists:
            self.tidx.create(self._tasks_provider())
            self.ridx.create(self._resources_provider())
        else:
            self.tidx.load()
            self.ridx.load()

    @staticmethod
    def _tasks_provider():
        from sandbox import projects
        for t in projects.TYPES.itervalues():
            title = t.cls.type.replace("_", " ")
            descr = t.cls.description
            yield common.quicksearch.Index.Document(t.cls.type, title, descr)

    @staticmethod
    def _resources_provider():
        for t in sdk2.Resource:
            title = str(t).replace("_", " ")
            descr = t.description
            yield common.quicksearch.Index.Document(str(t), title, descr)

    @classmethod
    def cleanup(cls):
        if hasattr(cls, "_clear"):
            return
        if not common.config.Registry().server.services.packages_updater.enabled:
            # Remove index created on a previous run for local Sandbox installations.
            try:
                shutil.rmtree(cls.location)
            except OSError:
                pass
        setattr(cls, "_clear", True)


class Suggest(object):
    @classmethod
    def _client_tags(cls, typename):
        tw = controller.TaskWrapper(mapping.Task(type=typename))
        return str(tw.type_client_tags)

    @classmethod
    def task(cls, request):
        from sandbox import projects
        typename = request.get("type")
        if typename:
            typename = typename.strip().upper()
            ttype = projects.TYPES.get(typename)
            if not ttype:
                return misc.json_error(httplib.BAD_REQUEST, "No task type '{}' found.".format(typename))
            data = [(ttype.cls.type, ttype)]
        else:
            data = sorted(projects.TYPES.iteritems(), key=operator.itemgetter(0))

        return misc.response_json([
            {
                "type": k,
                "owners": v.cls.owners,
                "description": v.cls.description,
                "color": common.format.suggest_color(k),
                "relative_path": common.projects_handler.task_type_relative_path(k),
                "client_tags": cls._client_tags(k)
            }
            for k, v in data
        ])

    @classmethod
    def task_type_custom_fields(cls, _, typename):
        from sandbox import projects
        typename = typename.upper()
        if typename not in projects.TYPES:
            return misc.json_error(httplib.NOT_FOUND, "No task type '{}' found".format(typename))

        dummy = mapping.Task(
            id=0, type=typename, priority=0,
            requirements=mapping.Task.Requirements(), parameters=mapping.Task.Parameters(),
        )
        data = task.Task.custom_fields_views(dummy, input=True, output=True, with_hidden=True)
        for d in data:
            for exclusion in ("value",):
                d.pop(exclusion)
        return misc.response_json(data)

    @classmethod
    def resource(cls, request):
        typename = request.get("type")
        rst2html = common.config.Registry().server.web.rst2html
        if typename:
            rtype = typename.strip().upper()
            rtype = sdk2.Resource[rtype] if rtype in sdk2.Resource else None
            if not rtype or rtype not in sdk2.Resource:
                return misc.json_error(httplib.BAD_REQUEST, "No resource type '{}' found.".format(typename))
            data = [rtype]
        else:
            data = sorted(sdk2.Resource, key=str)
        return misc.response_json([
            {"type": str(k), "description": k.description if rst2html else (k.__doc__ or "").strip()}
            for k in data
        ])

    @classmethod
    def _host_platforms(cls, host_platforms):
        platforms = map(common.platform.get_platform_alias, host_platforms)
        try:
            platforms.extend(map(common.platform.get_arch_from_platform, platforms))
        except ValueError:
            pass
        return platforms

    @classmethod
    def client(cls, request):
        groups = ("platforms", "hosts", "cpu_models")
        res = collections.namedtuple("SuggestResult", groups)(set(), set(), set())
        filters = ("task_type", "platform", "cpu", "host")
        filters = collections.namedtuple("Filter", filters)(*(request.get(arg) for arg in filters))

        args = {}
        tt_hosts = set()
        if filters.cpu:
            args['model'] = filters.cpu

        if filters.host:
            args['hostname'] = filters.host
        if filters.task_type:
            from sandbox import projects
            tt = projects.TYPES[filters.task_type].cls
            client_tags = tt.Requirements.client_tags.default if issubclass(tt, sdk2.Task) else tt.client_tags
            tt_hosts = set(controller.TaskQueue.HostsCache().client_tags[client_tags])

        lxc_platforms_aliases = None
        platforms_cache = {}
        for c in controller.Client.list_query(**args).order_by("hostname").lite():
            update_platforms = False
            if c.lxc:
                if lxc_platforms_aliases is None:
                    update_platforms = True
                    lxc_platforms_aliases = cls._host_platforms(common.platform.LXC_PLATFORMS)
                platforms = lxc_platforms_aliases
            else:
                platform = c.info["system"].get("platform", "")
                if platform not in platforms_cache:
                    update_platforms = True
                    platforms_cache[platform] = cls._host_platforms([platform])
                platforms = platforms_cache[platform]

            if (
                (filters.platform and filters.platform not in platforms) or
                (filters.task_type and c.hostname not in tt_hosts)
            ):
                continue
            res.hosts.add(c.hostname)
            res.cpu_models.add(c.hardware.cpu.model)
            if update_platforms:
                res.platforms.update(platforms)

        res = {k: list(getattr(res, k)) for k in groups}
        res["client_tags"] = cls._client_tags(filters.task_type) if filters.task_type else ""
        return misc.response_json(res)

    @classmethod
    def quicksearch(cls, request, query):
        return misc.response_json([])


registry.registered_json("suggest/task")(Suggest.task)
registry.registered_json(r"suggest/([\w\d_]+)/custom_fields")(Suggest.task_type_custom_fields)
registry.registered_json("suggest/resource")(Suggest.resource)
registry.registered_json("suggest/client")(Suggest.client)
registry.registered_json(r"suggest/quicksearch/([^/]+)")(Suggest.quicksearch)
