# coding: utf-8

import os
import json
import pymongo
import logging
import aniso8601
import distutils.util
import datetime as dt
import operator as op
import itertools as it
import functools as ft
import collections

import six
from six.moves import cPickle as pickle
from six.moves import http_client as httplib
from six.moves.urllib import parse as urlparse

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.statistics as ctst
from sandbox import sdk2

from sandbox.taskbox import errors as tb_errors
from sandbox.taskbox.model import update as model_update
from sandbox.yasandbox import context as request_context
from sandbox.yasandbox import manager
from sandbox.yasandbox import controller
from sandbox.yasandbox.proxy import task as proxy_task
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.controller import user as user_controller
from sandbox.yasandbox.controller import client as client_controller

from sandbox.web import api
from sandbox.yasandbox.api.json import Base
from sandbox.yasandbox.api.json import PathBase
from sandbox.yasandbox.api.json import registry
from sandbox.yasandbox.api.json import mappers
from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import list_arg_parser
from sandbox.yasandbox.api.json import priority_parser

import sandbox.web.helpers
import sandbox.web.response
import sandbox.serviceq.errors as qerrors


def _status_arg2list(x):
    """ Converts comma-separated string of task statuses including statuses groups to a list of statuses. """
    groups = set(map(str, list(ctt.Status.Group)))
    ret = []
    for s in x.split(","):
        if s in groups:
            ret.extend(list(getattr(ctt.Status.Group, s)))
        else:
            ret.append(s)
    return ret


def _get_short_task_result(task):
    try:
        short_res = task.short_task_result
        json.dumps(short_res)
        return short_res
    except Exception as ex:
        return "Can't get short task result or serialize it: {}".format(ex)


def _get_api_url(url):
    p = urlparse.urlsplit(url)
    base_path = '/'.join(p.path.split('/')[:3])  # make it `/api/v.+`
    return urlparse.urlunsplit((p.scheme, p.netloc, base_path, '', ''))


def _get_release_info(api_url, task, task_resources=None):

    release_info = {}

    doc = task.model
    if getattr(doc, "release", None):  # Schedulers has no "release" sub-document
        release_info = {
            "url": "{}/release/{}".format(api_url, doc.id),
            "type": doc.release.status,
            "id": doc.id,
        }

    release_info["warning_message"] = controller.Group.check_abc(doc)

    if task_resources is None:
        task_resources = task.resources()

    releasable = (
        (
            ctt.Status.can_switch(doc.execution.status, ctt.Status.RELEASING) or
            doc.execution.status == ctt.Status.RELEASING
        ) and any(resource.type.releasable for resource in task_resources)
    )

    if releasable and task.cls is not proxy_task.Task:
        release_info.update({
            "resources": [
                {
                    "resource_id": resource.id,
                    "releasers": resource.type.releasers,
                }
                for resource in task_resources if resource.type.releasable
            ]
        })

        tmpl = task.release_template()

        cc = tmpl.get("cc")
        if cc:
            tmpl["cc"] = list(common.utils.chain(cc))

        release_info["template"] = tmpl

    return release_info


def _task_meta(types):
    result = {}
    for task_type in types:
        if task_type in sdk2.Task:
            param_classes = sdk2.Task[task_type].Parameters
            sdk_type = ctt.SDKType.SDK2
            reports = [
                {"label": r.label, "title": r.title}
                for r in six.viewvalues(sdk2.Task[task_type].__reports__)
            ]
        else:
            task_class = proxy_task.getTaskClass(task_type)
            if task_class is proxy_task.Task:
                param_classes = []
                sdk_type = ctt.SDKType.SDK2
            else:
                param_classes = task_class.input_parameters or []
                sdk_type = ctt.SDKType.SDK1
            reports = [{"title": "Footer", "label": "footer"}]
        parameters = model_update.LocalUpdate.parameters_meta_list(
            param_classes, mappers.TaskParametersMetaInfo, short=True
        )
        result[task_type] = {
            "sdk_type": sdk_type, "parameters": [pm.to_dict() for pm in parameters], "reports": reports
        }
    return result


class Task(Base):
    """
    The class encapsulates all the logic related to REST API representation of any entities related to task object.
    """

    logger = logging.getLogger("RESTAPI_Task")

    LIST_AUDIT_MAX_INTERVAL = dt.timedelta(days=14)

    # Shortcuts for database models.
    Model = mapping.Task
    Audit = mapping.Audit
    Priority = ctt.Priority
    Status = ctt.Status
    Resource = mapping.Resource

    __param_parser = lambda data: misc.json_parser(data, expects=dict, expects_each=six.string_types)
    CHUNK_SIZE = 1000

    # A list of list operation query parameters mapping.
    LIST_QUERY_MAP = (
        Base.QueryMapping("id", "id", "id", list_arg_parser(int)),
        Base.QueryMapping("type", "type", None, list_arg_parser(str)),
        Base.QueryMapping("status", "status", "execution__status", _status_arg2list),
        Base.QueryMapping("parent", "parent_id", "parent_id", list_arg_parser(int)),
        Base.QueryMapping("scheduler", "scheduler", "scheduler", list_arg_parser(int)),
        Base.QueryMapping("template_alias", "template_alias", "template_alias", list_arg_parser(str)),
        Base.QueryMapping("host", "host", "requirements__host", str),
        Base.QueryMapping("arch", "arch", "requirements__platform", str),
        Base.QueryMapping("model", "model", "requirements__cpu_model", str),
        Base.QueryMapping("requires", "requires", "requirements__resources", list_arg_parser(int)),
        Base.QueryMapping("owner", "owner", None, str),
        Base.QueryMapping("author", "author", None, str),
        Base.QueryMapping("desc_re", "descr_mask", "description", lambda _: _.encode("utf-8")),
        Base.QueryMapping("children", "show_childs", None, distutils.util.strtobool),
        Base.QueryMapping("hidden", "hidden", None, distutils.util.strtobool),
        Base.QueryMapping("se_tag", "se_tag", None, str),
        Base.QueryMapping("priority", "priority", None, priority_parser),
        Base.QueryMapping("important", "important_only", "flagged", distutils.util.strtobool),
        Base.QueryMapping("created", "created", "time__created", sandbox.web.helpers.datetime_couple),
        Base.QueryMapping("updated", "updated", "time__updated", sandbox.web.helpers.datetime_couple),
        Base.QueryMapping("limit", "limit", None, int),
        Base.QueryMapping("offset", "offset", None, int),
        Base.QueryMapping("order", "order_by", None, str),
        Base.QueryMapping("fields", "fields", None, list_arg_parser(str)),
        Base.QueryMapping("input_parameters", "input_parameters", None, __param_parser),
        Base.QueryMapping("output_parameters", "output_parameters", None, __param_parser),
        Base.QueryMapping("any_params", "any_params", None, distutils.util.strtobool),
        Base.QueryMapping("tags", "tags", "tags", list_arg_parser(lambda x: str(x).upper())),
        Base.QueryMapping("all_tags", "all_tags", None, distutils.util.strtobool),
        Base.QueryMapping("hints", "hints", "hints", list_arg_parser(str)),
        Base.QueryMapping("all_hints", "all_hints", None, distutils.util.strtobool),
        Base.QueryMapping("release", "release", "release__status", list_arg_parser(lambda x: str(x).lower())),
        Base.QueryMapping("semaphore_acquirers", "semaphore_acquirers", "semaphore_acquirers", int),
        Base.QueryMapping("semaphore_waiters", "semaphore_waiters", "semaphore_waiters", int),
    )

    class Entry(dict):

        def __init__(self, user, base_url, doc, short_task_result=False, waits_mutex=False, request=None):
            mapper_fields = mappers.TaskMapper.get_base_fields()

            if short_task_result:
                mapper_fields.add("short_task_result")

            tm = mappers.TaskMapper(
                mapper_fields, user, base_url.rsplit("/", 1)[0],
                {doc.id} if waits_mutex else None, {},
                task_docs=[doc], types_resolve_func=_task_meta
            )
            self.update(tm.dump(doc))

            self.__doc = doc
            self._task = controller.TaskWrapper(doc)
            self["sdk_version"] = self._task.sdk_version

            # Backward compatible behaviour: remove `scheduler` field when it is not set.
            # TODO: remove after making sure that nobody relies on the absence of this field
            if self.get("scheduler", ctm.NotExists) is None:
                self.pop("scheduler")

            if not request or request.session:
                return

            hosts = 0
            pools = []
            if doc.execution.status == ctt.Status.ENQUEUED:
                try:
                    hosts = len(controller.TaskQueue.qclient.queue_by_task(doc.id, secondary=True)[1])
                    if request.source == request.Source.WEB:
                        pools_indexes = set()
                        client_tags = client_controller.Client.perform_tags_by_platform(
                            doc.effective_client_tags, controller.Task.platform_filter(doc)
                        )
                        for tags in controller.TaskQueue.get_enriched_client_tags(client_tags):
                            pool_index = controller.TaskQueue.quota_pools.match_pool(tags)
                            if pool_index is not None:
                                pools_indexes.add(pool_index)
                        if pools_indexes:
                            pools = [controller.TaskQueue.quota_pools.pools[index] for index in sorted(pools_indexes)]
                except qerrors.QException as ex:
                    Task.logger.warning("Error while requesting hosts and pools for task #%s: %s", doc.id, ex)

            if doc.execution.status in common.utils.chain(ctt.Status.Group.REALEXECUTE, ctt.Status.FINISHING):
                self["execution"].update(
                    ps_url=self.ps_url,
                    actions_url=self.actions_url,
                    shell_url=self.shell_url,
                )

            edir_url = self.__execution_dir_url
            logs = []
            if edir_url:
                logs.append({
                    "name": "execution dir",
                    "node_type": "directory",
                    "url": edir_url,
                    "size": doc.execution.disk_usage.last << 10,
                })

            tails = []
            task_resources = self._task.resources()

            common_logs_resources = []
            custom_logs_resources = []
            for log in task_resources:
                if log.type == str(sdk2.service_resources.TaskLogs):
                    common_logs_resources.append(log)
                elif log.type == str(sdk2.service_resources.TaskCustomLogs):
                    custom_logs_resources.append(log)

            if common_logs_resources:
                last_log = common_logs_resources[0]
                for list_, node_type, suffix in (
                    (logs, "file", ""),
                    (tails, "tail", "?tail=1&force_text_mode=1")
                ):
                    list_.extend([
                        {
                            "name": log.get("name"),
                            "node_type": node_type,
                            "url": log.get("url") + suffix,
                        }
                        for log in filter(None, (self.__log_view(name, suffix, last_log) for name in list(ctt.LogName)))
                    ])
            for log in it.chain(common_logs_resources, custom_logs_resources):
                logs.append({
                    "node_type": "directory",
                    "name": log.file_name,
                    "size": log.size << 10,
                    "url": log.proxy_url,
                })

            self.update({
                "lock_host": doc.lock_host,
                "resources": {
                    "url": base_url + "/resources",
                    "count": Task.Resource.objects(task_id=doc.id).count(),
                },
                "context": {
                    "url": base_url + "/context",
                    "count": len(self._task.ctx),
                },
                "dependant": {
                    "url": base_url + "/dependant",
                    "count": controller.Task.dependent(doc.id).count()
                },
                "children": {
                    "url": base_url + "/children",
                    "count": Task.Model.objects(parent_id=doc.id).count(),
                },
                "audit": {
                    "url": base_url + "/audit",
                    "count": Task.Audit.objects(task_id=doc.id).count(),
                },
                "hosts": {
                    "url": base_url + "/hosts",
                    "count": hosts,
                },
                "logs": logs,
                "tails": tails,
                "pools": pools,
                "effective_client_tags": doc.effective_client_tags,
            })

            release_info = _get_release_info(_get_api_url(base_url), self._task, task_resources=task_resources)
            if release_info:
                # TODO: SANDBOX-6686: this field should eventually be removed in favour of `/task/{id}/release`
                self["release"] = release_info

            if doc.execution.status not in common.utils.chain(ctt.Status.Group.DRAFT, ctt.Status.Group.QUEUE):
                results = {
                    "disk_usage": {
                        "last": doc.execution.disk_usage.last << 10 if doc.execution.disk_usage else 0,
                        "max": doc.execution.disk_usage.max << 10 if doc.execution.disk_usage else 0
                    },
                }
                traceback = self._task.ctx.get("__last_error_trace")
                if traceback:
                    results["traceback"] = common.utils.force_unicode_safe(traceback)
                self["results"].update(results)

            if short_task_result:
                self["short_task_result"] = _get_short_task_result(self._task)

            # TODO: SANDBOX-3689: Backward compatibility code for old versions of `upload.sfx.py` and `ya upload` -
            # TODO: do not return new "ASSIGNED" status ever.
            if doc.type == "HTTP_UPLOAD" and doc.execution.status == ctt.Status.ASSIGNED:
                self["status"] = ctt.Status.ENQUEUED

        @property
        def __host_fqdn(self):
            host = self.__doc.execution.host
            if not host:
                return ""
            return mapping.Client.objects.with_id(host).fqdn or ""

        def __proxy_command(self, *args):
            """
            Make a url which works for both local and production installations,
            switching between fileserver and proxy URL usage
            """

            settings = common.config.Registry().client
            tail = "/".join(map(str, args))
            return (
                "{}://{}/{}".format(settings.fileserver.proxy.scheme.http, settings.fileserver.proxy.host, tail)
                if settings.fileserver.proxy.host else
                "http://{}:{}/{}".format(self.__host_fqdn, settings.fileserver.port, tail)
            )

        @property
        def ps_url(self):
            return self.__proxy_command("ps", self.__doc.id)

        @property
        def actions_url(self):
            return self.__proxy_command("actions", self.__doc.id)

        @property
        def shell_url(self):
            return self.__proxy_command("shell", self.__doc.id)

        @staticmethod
        def remote_http(doc):
            if not doc.execution.host:
                return ""
            http_prefix = mapping.Client.objects.with_id(
                doc.execution.host
            ).info.get("system", {}).get("fileserver", "")
            if not http_prefix:
                return ""
            full_url = urlparse.urljoin(http_prefix, os.path.join(*(ctt.relpath(doc.id) + [""])))
            return full_url

        @property
        def __remote_http(self):
            return self.remote_http(self.__doc)

        @property
        def __execution_dir_url(self):
            if not self.__doc.execution.host:
                return None
            settings = common.config.Registry().client.fileserver
            return (
                "{}://{}/task/{}".format(settings.proxy.scheme.http, settings.proxy.host, self.__doc.id)
                if settings.proxy.host else
                self.__remote_http
            )

        def __log_url(self, log_name, include_incomplete, last_log):
            allowed_states = (ctr.State.READY,)
            if include_incomplete:
                allowed_states += (ctr.State.NOT_READY,)
            try:
                if last_log.state not in allowed_states:
                    return
                return "/".join((last_log.proxy_url, log_name))
            except:
                return

        def __log_view(self, logname, include_incomplete, last_log):
            url = self.__log_url(logname, include_incomplete, last_log)
            return {"url": url, "name": logname} if url else None

    class AuditListItemEntry(dict):
        def __init__(self, doc):
            data = zip(
                (
                    'time',
                    'iface',
                    'target',
                    'status',
                    'author',
                    'source',
                    'message',
                    'task_id',
                    'request_id',
                    'wait_targets',
                ),
                (
                    sandbox.web.helpers.utcdt2iso(doc.date),
                    doc.source,
                    doc.hostname,
                    doc.status,
                    doc.author,
                    doc.remote_ip,
                    doc.content,
                    doc.task_id,
                    doc.request_id,
                    doc.wait_targets,
                )
            )
            super(Task.AuditListItemEntry, self).__init__({k: v for k, v in data if v is not None})

    class QueueListItemEntry(dict):
        def __init__(self, base_url, doc):
            name, info = doc
            filter_gen = lambda fname: (f["allowed"] for f in info["filters"] if f["filter_name"] == fname)
            super(Task.QueueListItemEntry, self).__init__({
                "client": {
                    "name": name,
                    "url": "{}/client/{}".format(base_url.rsplit("/", 1)[0], name)
                },
                "alive": next(filter_gen("alive"), False),
                "platform": next(filter_gen("platform"), False),
                "cpu": next(filter_gen("cpu_model"), False),
                "free_space": next(filter_gen("client_free_space"), False),
            })
            queue = info.get("queue")
            if queue:
                self["position"] = dict(zip(("index", "size"), queue))

    @classmethod
    def _filter_items(cls, items, fields):
        field_getters = {
            field: ft.partial(ft.reduce, lambda a, b: (a or {}).get(b), field.split("."))
            for field in fields or {}
        }
        return (
            {
                field: getter(item)
                for field, getter in field_getters.iteritems()
            }
            for item in items
        )

    @classmethod
    def _semaphore_waiters(cls, request, docs):
        return (
            set(controller.TaskQueue.qclient.semaphore_waiters(
                [_.id for _ in docs if _.execution.status == ctt.Status.ENQUEUED]
            ))
            if request and request.source == request.Source.WEB else
            None
        )

    @classmethod
    def list(cls, request):
        # Parse query arguments and form them as keyword arguments to database query builder
        try:
            kwargs, offset, limit = cls._handle_args(request, multi_order=True)
        except (TypeError, ValueError, KeyError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, "Query parameter validation error: " + str(ex))
        if limit is None:
            return misc.json_error(httplib.BAD_REQUEST, "Required parameter 'limit' not provided.")
        if limit > 3000 and not request.user.super_user:
            return misc.json_error(httplib.BAD_REQUEST, "Too big amount of data requested.")

        # FIXME: Protect against DDoS
        if common.config.Registry().common.installation == ctm.Installation.PRODUCTION:
            if (request.user == controller.User.anonymous and kwargs.get("hidden") and
                    not limit and not request.get("parent")):
                return misc.json_error(httplib.UNAUTHORIZED, "Unauthorized request.")
            if (
                request.user == controller.User.anonymous and
                request.remote_ip.startswith("2a02:6b8:b010:1f::")
            ):
                return misc.json_error(httplib.UNAUTHORIZED, "Unauthorized request.")

        ids_to_intersect = None
        sem_id = kwargs.pop("semaphore_acquirers", None)
        if sem_id is not None:
            semaphore_obj = dict(controller.TaskQueue.qclient.semaphores(sem_id)).get(sem_id)
            ids_to_intersect = semaphore_obj.tasks.keys() if semaphore_obj is not None else []

        sem_id = kwargs.pop("semaphore_waiters", None)
        if sem_id is not None and ids_to_intersect is None:
            # if both "semaphore_acquirers" and "semaphore_waiters" are present, ignoring second
            ids_to_intersect = dict(controller.TaskQueue.qclient.semaphore_wanting(sem_id)).get(sem_id, [])

        query_task_ids = kwargs.get("id", None)
        if query_task_ids is not None and ids_to_intersect is not None:
            if isinstance(query_task_ids, (list, tuple)):
                ids_to_intersect = set(ids_to_intersect)
                kwargs["id"] = [task_id for task_id in query_task_ids if task_id in ids_to_intersect]
            else:
                kwargs["id"] = query_task_ids if query_task_ids in ids_to_intersect else []
        elif ids_to_intersect is not None:
            kwargs["id"] = ids_to_intersect

        ids_order = None
        short_task_result = (request.source == request.Source.WEB)
        fields = set(kwargs.pop("fields", []))
        context_in_fields = any(field == "context" or field.startswith("context.") for field in fields)
        kwargs["load_ctx"] = kwargs.get("load_ctx") or short_task_result or context_in_fields
        if "order_by" not in kwargs and "id" in kwargs:
            kwargs["order_by"] = None
            ids_order = kwargs["id"]

        filter_wait_mutex = False
        if request.source == request.Source.WEB and kwargs.get("status"):
            statuses = set(common.utils.chain(kwargs["status"]))
            if ctt.Status.WAIT_MUTEX in statuses:
                if statuses - {ctt.Status.WAIT_MUTEX, ctt.Status.ENQUEUED}:
                    statuses.remove(ctt.Status.WAIT_MUTEX)
                    kwargs["status"] = statuses
                else:
                    filter_wait_mutex = ctt.Status.ENQUEUED not in statuses
                    kwargs["status"] = ctt.Status.ENQUEUED

        if kwargs.get("descr_mask") and request.source == request.Source.WEB:
            if not any(param in kwargs for param in ("owner", "author", "type", "tags", "hints")):
                return misc.json_error(
                    httplib.BAD_REQUEST,
                    "Search by description without 'owner', 'author', 'type', 'tags' or 'hints' fields is not allowed."
                )

        # empty intersection leads to ignoring kwargs["id"] in db query
        task_ids = kwargs.get("id", None)
        if isinstance(task_ids, (list, tuple)) and len(task_ids) == 0:
            return misc.response_json({
                "limit": limit,
                "offset": offset,
                "total": 0,
                "items": [],
            })

        query = manager.task_manager.list_query(**kwargs)
        total = query.count()

        if filter_wait_mutex:
            real_offset = 0
            skipped = 0
            docs = []

            while real_offset < total:
                tmp_docs = list(query.skip(real_offset).limit(cls.CHUNK_SIZE))
                tmp_semaphore_waiters = cls._semaphore_waiters(request, tmp_docs)

                for doc in tmp_docs:
                    if len(docs) >= limit:
                        break

                    if doc.id in tmp_semaphore_waiters:
                        if skipped >= offset:
                            docs.append(doc)
                        else:
                            skipped += 1

                if len(docs) >= limit or len(tmp_docs) < cls.CHUNK_SIZE:
                    break
                real_offset += cls.CHUNK_SIZE
        else:
            docs = list((query if not offset else query.skip(offset)).limit(limit))

        if ids_order:
            docs = sorted(docs, key=lambda _: ids_order.index(_.id))

        if not fields or "scores" in fields:
            weather_map = {_.type: _ for _ in mapping.Weather.objects(type__in={_.type for _ in docs})}
        else:
            weather_map = {}

        semaphore_waiters = set(_.id for _ in docs) if filter_wait_mutex else cls._semaphore_waiters(request, docs)
        postprocesses = []
        if fields:
            fields.add("id")
            mapper_fields = {_ for _ in fields if not _.startswith("context.")}
            if context_in_fields:
                mapper_fields.add("context")
            postprocesses.append(ft.partial(cls._filter_items, fields=fields))
            if "children" in fields:
                postprocesses.append(ft.partial(cls._add_children, docs=docs))
        else:
            mapper_fields = mappers.TaskMapper.get_base_fields()
            if short_task_result:
                mapper_fields.add("short_task_result")
            postprocesses.append(ft.partial(cls._add_children, docs=docs))

        task_mapper = mappers.TaskMapper(
            mapper_fields,
            request.user, request.uri,
            semaphore_waiters, weather_map,
            task_docs=docs, types_resolve_func=_task_meta
        )
        tasks = (task_mapper.dump(doc) for doc in docs)
        for postprocess in postprocesses:
            tasks = postprocess(tasks)
        return misc.response_json({
            "limit": limit,
            "offset": offset,
            "total": total,
            "items": list(tasks),
        })

    @classmethod
    def _add_children(cls, items, docs):
        # Build children list in one query
        children = {
            pid: [(item[0], item[2]) for item in ids]
            for pid, ids in it.groupby(
                cls.Model.objects(
                    parent_id__in=set(d.id for d in docs)
                ).scalar("id", "parent_id", "execution__status").order_by("+parent_id"),
                key=op.itemgetter(1)
            )
        }
        for item in items:
            item["children"] = [{str(row[0]): row[1]} for row in children.get(item["id"], [])]
            yield item

    @classmethod
    def current(cls, request):
        ret = cls.Entry(request.user, request.uri, cls._document(request.session.task), request=request)
        ret["context"] = ret._task.ctx
        return misc.response_json_ex(httplib.OK, ret)

    @classmethod
    def update_current_context(cls, request):
        data = misc.request_data(request)
        doc = cls._document(request.session.task)
        cls.Model.objects(id=doc.id).update_one(set__context=cls.Model.context.to_mongo(pickle.dumps(data)))
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def update_current_context_value(cls, request):
        data = misc.request_data(request)
        doc = cls._document(request.session.task)
        if not ("key" in data and "value" in data):
            return misc.json_error(httplib.BAD_REQUEST, 'Key and value must be supplied.')
        ctx = pickle.loads(doc.context)
        ctx[data["key"]] = data["value"]
        cls.Model.objects(id=doc.id).update_one(set__context=cls.Model.context.to_mongo(pickle.dumps(ctx)))
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def update_current_execution(cls, request):
        data = misc.request_data(request)
        doc = cls._document(request.session.task)
        update_expr = {}
        if "description" in data:
            update_expr["set__execution__description"] = data["description"]
        du, work_du = map(data.get, ("disk_usage", "work_disk_usage"))
        if du is not None:
            du_max, du_last = 0, 0
            if mapping.Client.objects(
                hostname=request.session.client,
                pure_tags=str(ctc.Tag.MULTISLOT),
                read_preference=mapping.ReadPreference.SECONDARY
            ).first():
                du_max, du_last, du_resources = map((work_du or du).get, ("max", "last", "resources"))
                if du_resources is not None:
                    du_max += du_resources
                    du_last += du_resources

            # these values are also reset on task restart, see yasandbox.api.json.batch.BatchTask._op()
            if (du_max, du_last) == (0, 0):
                du_max, du_last = map(du.get, ("max", "last"))
            if du_max is not None:
                update_expr.update(set__execution__disk_usage__max=max(doc.execution.disk_usage.max, du_max >> 10))
            if du_last is not None:
                update_expr.update(set__execution__disk_usage__last=du_last >> 10)

            utcnow = dt.datetime.utcnow()
            common.statistics.Signaler().push(dict(
                type=ctst.SignalType.TASK_DISK_USAGE,
                date=utcnow,
                timestamp=utcnow,
                max_working_set_usage=work_du.get("max", 0),
                last_working_set_usage=work_du.get("last", 0),
                reserved_working_set=doc.requirements.disk_space << 20,  # stored in MiB
                resources_synced=work_du.get("resources", 0),
                place_delta=du_max,
                task_id=doc.id,
                task_type=doc.type,
                owner=doc.owner,
                task_tags=doc.tags
            ))

        if update_expr:
            cls.Model.objects(id=doc.id).update_one(**update_expr)
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def get(cls, request, obj_id):
        from sandbox import projects
        short_task_result = (request.source == request.Source.WEB)
        obj = cls._document(obj_id, lite=True)
        headers = None if obj.type in getattr(projects, "TYPES", {}) else {"X-Task-Without-Code": "true"}
        waits_mutex = bool(cls._semaphore_waiters(request, [obj]))
        return misc.response_json_ex(
            200, cls.Entry(
                request.user, request.uri, obj,
                short_task_result=short_task_result, waits_mutex=waits_mutex, request=request
            ),
            headers=headers
        )

    @classmethod
    def list_audit(cls, request):
        def get_datetime_field(request, field):
            value = request.get(field)
            if value:
                try:
                    date = aniso8601.parse_datetime(value).replace(tzinfo=None)
                except ValueError as e:
                    return misc.json_error(
                        httplib.BAD_REQUEST,
                        "Can't parse '{}={}', invalid format: {}".format(field, value, e)
                    )
                return date
            return None

        ids = request.get("id")
        if ids is not None:
            ids = list_arg_parser(int)(ids)
        since = get_datetime_field(request, "since")
        to = get_datetime_field(request, "to")
        remote_ip = request.get("remote_ip")
        order = request.get("order", "+date")

        default_to = dt.datetime.utcnow()
        if to is None:
            to = default_to
        else:
            to = min(to, default_to)
        if not since and not ids:
            since = to - dt.timedelta(days=1)

        if since and to - since > cls.LIST_AUDIT_MAX_INTERVAL:
            return misc.json_error(
                httplib.BAD_REQUEST,
                "Intervals larger than {} days are not allowed".format(cls.LIST_AUDIT_MAX_INTERVAL.days)
            )

        return misc.response_json(
            map(cls.AuditListItemEntry, manager.task_manager.task_status_history(ids, since, to, remote_ip, order))
        )

    @classmethod
    def audit(cls, _, task_id):
        return misc.response_json(
            map(cls.AuditListItemEntry, cls.Audit.objects(task_id=cls._id(task_id)).order_by("+date"))
        )

    @classmethod
    def release(cls, request, task_id):
        api_url = _get_api_url(request.uri)
        task = controller.TaskWrapper(cls._document(task_id))
        return misc.response_json(_get_release_info(api_url, task))

    @classmethod
    def create_current_release(cls, request):
        task_id = request.session.task
        task = cls._document(task_id)
        if task.execution.status != ctt.Status.RELEASING:
            return misc.json_error(
                httplib.FORBIDDEN,
                "Task in status {}, but must be in status {}".format(task.execution.status, ctt.Status.RELEASING)
            )
        data = json.loads(request.raw_data)
        manager.release_manager.add_release_to_db(
            task_id, data["author"], data["release_status"], data["message_subject"],
            data.get("message_body"), data.get("changelog"), data.get("creation_time")
        )
        return sandbox.web.response.HttpResponse(code=httplib.CREATED)

    @classmethod
    def delete_current_release(cls, request):
        task_id = request.session.task
        task = cls._document(task_id)
        if task.execution.status != ctt.Status.RELEASING:
            return misc.json_error(
                httplib.FORBIDDEN,
                "Task in status {}, but must be in status {}".format(task.execution.status, ctt.Status.RELEASING)
            )
        mapping.Task.objects(id=task_id).update_one(unset__release=True)
        for resource in manager.resource_manager.list_task_resources(task_id):
            if resource.type.releasable:
                manager.resource_manager.drop_attr(resource.id, 'released')
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def create_release(cls, request, task_id):
        data = json.loads(request.raw_data)
        manager.release_manager.add_release_to_db(
            task_id, data["author"], data["release_status"], data["message_subject"],
            data.get("message_body"), data.get("changelog"), data.get("creation_time")
        )
        return sandbox.web.response.HttpResponse(code=httplib.CREATED)

    @classmethod
    def delete_release(cls, request, task_id):
        mapping.Task.objects(id=task_id).update_one(unset__release=True)
        for resource in manager.resource_manager.list_task_resources(task_id):
            if resource.type.releasable:
                manager.resource_manager.drop_attr(resource.id, 'released')

        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def create_audit(cls, request, task_id=None):
        task = cls._document(request.session.task if request.session else task_id)
        data = json.loads(request.raw_data)
        expected_status = data.get("expected_status")
        status = data.get("status")
        message = data.get("message")
        lock = data.get("lock")
        force = data.get("force")
        wait_targets = data.get("wait_targets")

        if status:
            if not (request.session or request.user.super_user):
                return misc.json_error(httplib.FORBIDDEN, "Task status can only be switched within task session")
            if status not in ctt.Status:
                return misc.json_error(httplib.BAD_REQUEST, "Wrong value of field `status`: {}".format(status))
        if not request.session:
            if not controller.user_has_permission(request.user, (task.author, task.owner)):
                return misc.json_error(
                    httplib.FORBIDDEN,
                    "User '{}' is not permitted to modify task #{}".format(request.user.login, task.id)
                )
        prev_status = task.execution.status
        client = request.session and request.session.client
        try:
            if not status:
                audit = controller.Task.audit(mapping.Audit(task_id=task.id, content=message))
            else:
                audit = controller.TaskWrapper(task, logger=cls.logger).set_status(
                    status,
                    event=message,
                    lock_host=None,
                    keep_lock=None,
                    force=force,
                    expected_status=expected_status,
                    wait_targets=wait_targets
                )
            if audit is None:
                return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)
        except common.errors.IncorrectStatus as ex:
            return misc.json_error(httplib.BAD_REQUEST, str(ex))
        except common.errors.UpdateConflict as ex:
            return misc.json_error(httplib.CONFLICT, str(ex))
        except Exception as ex:
            cls.logger.error(
                "Unexpected error while switching status"
                " from %s to %s for task #%s (lock: %s, client: %s, force: %s): %s",
                prev_status, status, task.id, lock, client, force, ex
            )
            raise
        return misc.response_json_ex(httplib.CREATED, cls.AuditListItemEntry(audit))

    @classmethod
    def queue(cls, request, task_id):
        all_ = request.get('all', '').lower() == 'true'
        return misc.response_json([
            cls.QueueListItemEntry(request.uri.rsplit('/', 2)[0], doc)
            for doc in controller.TaskWrapper(cls._document(task_id)).clients_with_compatibility_info()
            if all_ or doc[1].get('queue')
        ])

    @classmethod
    def context(cls, _, task_id):
        return misc.response_json(pickle.loads(cls._document(task_id).context))

    @classmethod
    def children(cls, request, task_id):
        task_id = cls._id(task_id)
        doc = cls._document(task_id)
        children_count = cls.Model.objects(parent_id=doc.id).count()
        parsed_result = urlparse.urlparse(request.uri)

        redirect_path = "/".join(parsed_result.path.split("/")[:4])
        redirect_query = urlparse.urlencode(urlparse.parse_qsl(request.query) + [
            ("parent", task_id),
            ("children", "true"),
            ("hidden", "true"),
            ("limit", children_count)
        ])
        redirect_uri = urlparse.urlunparse(parsed_result._replace(path=redirect_path, query=redirect_query))

        return sandbox.web.response.HttpRedirect(redirect_uri)

    @classmethod
    def dependant(cls, request, task_id):
        try:
            limit = int(request.get("limit", 3000))
            offset = int(request.get("offset", 0))
        except ValueError as ex:
            misc.json_error(httplib.BAD_REQUEST, "Error parsing limit and offset: " + str(ex))

        query = controller.Task.dependent(task_id)
        docs = list((query.limit(limit) if not offset else query.skip(offset).limit(limit)))
        semaphore_waiters = cls._semaphore_waiters(request, docs)

        tm = mappers.TaskMapper(
            None, request.user, request.uri.rsplit("/", 2)[0], semaphore_waiters, {},
            task_docs=docs, types_resolve_func=_task_meta
        )
        return misc.response_json([tm.dump(doc) for doc in docs])

    @classmethod
    def custom_fields(cls, _, task_id):
        return misc.response_json(cls.custom_fields_views(cls._document(task_id), input=True, output=True))

    @classmethod
    def custom_fields_views(cls, model, input=False, output=False, with_hidden=False):
        """
        :type model: mapping.Template
        :rtype: list[dict]
        """
        task = controller.TaskWrapper(model)
        if not isinstance(task, controller.task.TBWrapper):
            from sandbox import projects
            if model.type not in projects.TYPES:
                return misc.json_error(
                    httplib.NOT_FOUND, "Task type {} does not exist".format(model.type),
                    headers={"X-Task-Without-Code": "true"},
                )

        parameters_meta = task.parameters_meta
        in_values, out_values = task.parameters_dicts(parameters_meta)
        fields_views = []

        for pm in parameters_meta.params:
            if (not pm.output and not input) or (pm.output and not output) or (not with_hidden and not pm.type):
                continue

            value = (out_values if pm.output else in_values).get(pm.name)
            # Multi-select field must be a list of strings (selected items), but it is false for sdk1 tasks.
            if pm.type == ctt.ParameterType.MULTISELECT and isinstance(value, six.string_types):
                value = value.split()

            field = {
                "name": pm.name,
                "required": pm.required,
                "title": pm.title,
                "description": pm.description,
                "output": pm.output,
                "modifiers": pm.modifiers,
                "context": pm.context,
                "value": value,
            }
            if pm.type:
                field["type"] = pm.type
            if pm.type != ctt.ParameterType.BLOCK:
                if pm.sub_fields:
                    field["sub_fields"] = pm.sub_fields

            fields_views.append(field)

        return fields_views

    @classmethod
    def output(cls, request, task_id=None):
        doc = cls._document(request.session.task if task_id is None else task_id)
        return misc.response_json(cls.custom_fields_views(doc, output=True))

    @classmethod
    def add_tags(cls, request, task_id):
        data = misc.request_data(request, expects=list, expects_each=six.string_types)
        task = controller.TaskWrapper(cls._document(task_id))
        tags = [tag.upper() for tag in filter(None, data)]
        old_tags = set(task.tags)
        new_tags = [tag for tag in tags if tag not in old_tags]
        if new_tags:
            task.update_common_fields({"tags": task.tags + new_tags})
            task.on_save()
            task.save()
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def delete_tags(cls, request, task_id):
        data = misc.request_data(request, expects=list, expects_each=six.string_types)
        task = controller.TaskWrapper(cls._document(task_id))
        tags = set(tag.upper() for tag in filter(None, data))
        old_tags = task.tags
        new_tags = [tag for tag in old_tags if tag not in tags]
        if len(new_tags) < len(old_tags):
            task.update_common_fields({"tags": new_tags})
            task.on_save()
            task.save()
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def delete_tag(cls, request, task_id, tag):
        task = controller.TaskWrapper(cls._document(task_id))
        if not tag:
            return misc.json_error(httplib.BAD_REQUEST, "Tag is required")
        tag = tag.upper()
        if tag not in task.tags:
            return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)
        else:
            task.update_common_fields({"tags": [t for t in task.tags if t != tag]})
            task.on_save()
            task.save()
            return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def add_hints(cls, request, task_id):
        task = controller.TaskWrapper(cls._document(task_id))
        hints = misc.request_data(request, expects=list, expects_each=six.string_types)
        task.add_explicit_hints(filter(None, hints))
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)

    @classmethod
    def add_current_hints(cls, request):
        return cls.add_hints(request, request.session.task)

    CreateMethodParams = collections.namedtuple(
        "CreateMethodParams",
        ("type", "source", "children", "context", "scheduler_id", "template_alias", "regular_schedule"),
    )

    @classmethod
    def _validate_create_params(cls, params):
        defined_param_count = sum(map(
            lambda x: x is not None, (params.type, params.source, params.scheduler_id, params.template_alias)
        ))
        if defined_param_count < 1:
            raise ValueError("Neither task type nor source ID nor scheduler ID nor template alias is specified")
        if defined_param_count > 1:
            raise ValueError("Only one of task type, source ID or scheduler ID or template alias should be specified")
        if params.context and not isinstance(params.context, dict):
            raise ValueError("Param context must be dictionary")

    @classmethod
    def _validate_task_type(cls, params):
        from sandbox import projects
        if params.type is not None and params.type not in projects.TYPES:
            raise ValueError("Incorrect task type {!r}".format(params.type))

    @classmethod
    @request_context.timer_decorator()
    def _save_parameters_and_fields(cls, task, data, request, setup_notifications, save_task=True):
        try:
            task.update_common_fields(data)
            cls.update_and_validate_custom_fields(
                request.user, task,
                input_data=data.get("custom_fields"),
                output_data=data.get("output"),
                requirements_data=data.get("requirements"),
            )

            if setup_notifications:
                controller.Task.setup_notifications(
                    task.model, data.get("notifications"), new="notifications" not in data
                )
            if save_task:
                tasks_resource = controller.Task.tasks_archive_resource(task.model)
                task.on_save()
                new_tasks_resource = controller.Task.tasks_archive_resource(task.model)
                if tasks_resource != new_tasks_resource and task.sdk_type != ctt.SDKType.SDK1:
                    task = controller.TaskWrapper(task.model)
                if task.sdk_type == ctt.SDKType.TASKBOX:
                    task.parameters_meta  # update parameters meta in task model in case of tasks resource changed

                task.save()
        except (ValueError, mapping.DocumentTooLarge, mapping.ValidationError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, str(ex))

    @classmethod
    def _deduplicate(cls, request, data):
        from sandbox import sdk2
        temp_data = dict(data)
        temp_data.update(id=0, status=ctt.Status.DRAFT, author=None)
        temp_data["input_parameters"] = {_["name"]: _["value"] for _ in temp_data.get("custom_fields", ())}
        task = sdk2.Task.from_json(temp_data)
        if task.Parameters.uniqueness.key == "":
            data["uniqueness"] = dict(
                key=type(task).Parameters.uniqueness.hash(task),
                excluded_statuses=data.get("uniqueness", task.Parameters.uniqueness).get("excluded_statuses")
            )
        uniqueness = data.get("uniqueness", {})
        unique_key = uniqueness.get("key")
        excluded_statuses = list(ctt.Status.Group.expand(
            uniqueness.get("excluded_statuses", task.Parameters.uniqueness.excluded_statuses)
        ))
        if unique_key:
            task = controller.Task.Model.objects(
                unique_key=unique_key,
                execution__status__nin=excluded_statuses
            ).order_by("-id").first()
            if task:
                p = urlparse.urlparse(request.uri)
                redirect_uri = urlparse.urlunparse(p._replace(
                    path="/".join(common.utils.chain(p.path.split("/")[:4], str(task.id)))
                ))
                raise sandbox.web.response.HttpRedirect(redirect_uri)
        return unique_key

    @classmethod
    def _user_in_group(cls, user, group, is_super):
        # type: (str, str, bool) -> bool

        return (
            not group or
            group == user or
            is_super or
            group in controller.Group.get_user_groups(user)
        )

    @classmethod
    def _validate_user_and_group(cls, request_user, provided_author, provided_owner):
        # type: (mapping.User, str or None, str) -> (str, str)

        target_author = provided_author or request_user.login
        if not user_controller.user_has_right_to_act_on_behalf_of(request_user, target_author):
            return misc.json_error(
                httplib.FORBIDDEN,
                "It's forbidden for user '{}' to run task on behalf of '{}'".format(
                    request_user.login, target_author,
                ),
            )

        # do not allow admin to violate user-in-group rule
        is_super = request_user.super_user and target_author == request_user.login

        if not cls._user_in_group(target_author, provided_owner, is_super):
            # check whether owner exists at all
            existing = controller.Group.exists(provided_owner)
            if not existing:
                return misc.json_error(
                    httplib.BAD_REQUEST,
                    'Group "{}" does not exist'.format(provided_owner)
                )

            return misc.json_error(
                httplib.FORBIDDEN,
                'User "{}" does not belong to the group "{}"'.format(target_author, provided_owner)
            )

        return target_author, provided_owner

    @classmethod
    @request_context.timer_decorator("task_create", reset=True, logger=logger, loglevel=logging.INFO)
    def create(cls, request):
        parent_id = None
        data = misc.request_data(request)

        params = cls.CreateMethodParams(*map(data.get, cls.CreateMethodParams._fields))
        try:
            cls._validate_create_params(params)
        except ValueError as error:
            return misc.json_error(httplib.BAD_REQUEST, str(error))

        if params.children:
            if not request.session or not request.session.task:
                return misc.json_error(
                    httplib.BAD_REQUEST,
                    "Child task can only be created by other executing task"
                )
            parent_id = request.session.task

        target_author, target_owner = cls._validate_user_and_group(request.user, data.get("author"), data.get("owner"))

        setup_notifications = True
        try:
            unique_key = None
            if params.source:
                try:
                    source_task = controller.Task.get(params.source)
                except controller.Task.NotExists:
                    return misc.json_error(httplib.BAD_REQUEST, "No task #{} found".format(params.source))
                data["type"] = source_task.type

                task_model = mapping.Task(
                    type=source_task.type,
                    parent_id=parent_id,
                    author=target_author,
                    owner=target_owner,
                )

                if controller.Task.is_taskboxed_task(source_task):
                    task_model.requirements = mapping.Task.Requirements()
                    if not controller.Task.update_tasks_resource(data, task_model):
                        task_model.requirements.tasks_resource = source_task.requirements.tasks_resource
                else:
                    cls._validate_task_type(params)
                    unique_key = cls._deduplicate(request, data)  # FIXME: SANDBOX-7000

                task = controller.TaskWrapper(task_model).create().copy(source_task)
                controller.Task.update_tasks_resource(data, task_model)

            elif params.scheduler_id:
                cls._validate_task_type(params)
                scheduler = controller.Scheduler.load(params.scheduler_id)
                if not scheduler:
                    return misc.json_error(httplib.BAD_REQUEST, "No scheduler #{} found".format(params.scheduler_id))

                regular = params.regular_schedule and request.user.super_user

                if regular:
                    setup_notifications = False
                else:
                    scheduler.author = target_author
                    if target_owner:
                        scheduler.owner = target_owner
                    elif not cls._user_in_group(target_author, scheduler.owner, False):
                        scheduler.owner = None

                task = controller.TaskWrapper(controller.Scheduler.create_new_task(
                    scheduler, enqueue_task=False, manual=not regular
                ))
                controller.Task.update_tasks_resource(data, task.model)

                controller.Task.audit(cls.Audit(
                    task_id=task.id,
                    status=task.status,
                    content="Created using scheduler #{}".format(scheduler.id),
                ))
            else:
                template_alias = params.template_alias
                if template_alias is not None:
                    template = mapping.TaskTemplate.objects.with_id(params.template_alias)
                    if template is None:
                        return misc.json_error(httplib.NOT_FOUND, "No template #{} found".format(params.template_alias))
                    template_wrapper = controller.TemplateWrapper(template)
                    data = template_wrapper.create_task_request(data)
                    params = cls.CreateMethodParams(*map(data.get, cls.CreateMethodParams._fields))

                # Create base model with only necessary fields
                model = mapping.Task(
                    type=params.type,
                    parent_id=parent_id,
                    author=target_author,
                    owner=target_owner,
                    template_alias=template_alias,
                )

                # Check if custom task binary is specified, it may lead to using Taskbox
                controller.Task.update_tasks_resource(data, model)

                if not controller.Task.is_taskboxed_task(model):
                    cls._validate_task_type(params)
                    unique_key = cls._deduplicate(request, data)  # FIXME: SANDBOX-7000

                task = controller.TaskWrapper(model).create()
                if template_alias is not None:
                    controller.Task.audit(cls.Audit(
                        task_id=task.id,
                        status=task.status,
                        content="Created using template '{}'".format(template_alias),
                    ))

            task.update_context(params.context)
            cls._save_parameters_and_fields(
                task, data, request, setup_notifications=setup_notifications, save_task=False
            )
            tasks_resource = controller.Task.tasks_archive_resource(task.model)
            task.on_create()
            new_tasks_resource = controller.Task.tasks_archive_resource(task.model)
            if tasks_resource != new_tasks_resource and task.sdk_type != ctt.SDKType.SDK1:
                tasks_resource = new_tasks_resource
                task = controller.TaskWrapper(task.model)
            task.on_save()
            new_tasks_resource = controller.Task.tasks_archive_resource(task.model)
            if tasks_resource != new_tasks_resource and task.sdk_type != ctt.SDKType.SDK1:
                task = controller.TaskWrapper(task.model)
            if task.sdk_type == ctt.SDKType.TASKBOX:
                task.parameters_meta  # update parameters meta in task model in case of tasks resource changed
            if unique_key is not None:
                task.model.unique_key = unique_key
            task.update_hints()
            task.save()

        except (KeyError, ValueError, TypeError) as exc:
            return misc.json_error(httplib.BAD_REQUEST, str(exc))

        except common.errors.TaskError as exc:
            return misc.json_error(httplib.BAD_REQUEST, exc.message)

        return sandbox.web.helpers.response_created(
            "{}/{}".format(request.uri, task.id),
            content_type="application/json",
            content=json.dumps(
                cls.Entry(
                    request.user, urlparse.urljoin(request.uri.rstrip("/") + "/", str(task.id)),
                    cls.Model.objects.with_id(task.id), request=request
                ),
                ensure_ascii=False, encoding="utf-8"
            )
        )

    @classmethod
    def update(cls, request, task_id):
        doc = cls._document(task_id)  # type: mapping.Task
        update_data = misc.request_data(request)

        task_owner = doc.owner or doc.author
        if not controller.user_has_permission(request.user, [task_owner]):
            return misc.json_error(
                httplib.FORBIDDEN,
                "User '{}' is not permitted to modify task #{} owned by '{}'".format(
                    request.user.login, task_id, task_owner
                )
            )

        provided_owner = update_data.get("owner")
        if provided_owner and provided_owner != doc.owner:
            cls._validate_user_and_group(request.user, doc.author, provided_owner)

        if doc.execution.status != ctt.Status.DRAFT:
            own_session = request.session and request.session.task == int(task_id)
            if doc.execution.status in ctt.Status.Group.EXECUTE and not own_session:
                return misc.json_error(
                    httplib.BAD_REQUEST,
                    "Unable to change task in status `{}`. Task #{}".format(doc.execution.status, task_id)
                )

            exclusions = {"description", "tags", "priority", "expires", "score"}
            for k in update_data:
                if k not in exclusions:
                    return misc.json_error(
                        httplib.BAD_REQUEST,
                        "Cannot modify task in status different to `DRAFT`. "
                        "Exclusion fields: {}. "
                        "Task #{} in status `{}`".format(
                            ', '.join('`{}`'.format(excl) for excl in exclusions),
                            task_id, doc.execution.status
                        )
                    )

        try:
            # Check if custom task binary is specified, it may lead to using Taskbox
            controller.Task.update_tasks_resource(update_data, doc)
        except ValueError as ex:
            return misc.json_error(httplib.BAD_REQUEST, str(ex))

        task = controller.TaskWrapper(doc)
        new_priority = update_data.get("priority")
        if new_priority and doc.execution.status != ctt.Status.DRAFT:
            try:
                controller.TaskQueue.set_priority(request, task, ctt.Priority.make(new_priority))
            except (TypeError, ValueError) as ex:
                return misc.json_error(httplib.BAD_REQUEST, str(ex))
        else:
            cls._save_parameters_and_fields(
                task, update_data, request, setup_notifications="notifications" in update_data
            )
        return (
            misc.response_json(cls.Entry(
                request.user, request.uri, cls.Model.objects.with_id(task_id), request=request
            ))
            if cls.request_needs_updated_data(request) else
            sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)
        )

    @classmethod
    def resources(cls, request, task_id):
        resources_url = '{}/resource?task_id={}&limit=100'.format(request.uri.rsplit('/', 3)[0], task_id)
        return sandbox.web.helpers.redirect_to(resources_url, local=False)

    @classmethod
    def task_meta(cls, request):
        req_data = misc.request_data(request)
        types = req_data.get("types", [])
        all_types = req_data.get("all_types", False)
        if all_types:
            from sandbox import projects
            types = projects.TYPES.keys()

        result = _task_meta(types)
        return misc.response_json(result)

    @classmethod
    def requirements(cls, request, task_id):
        resources_url = '{}/resource?dependant={}&limit=100'.format(request.uri.rsplit('/', 3)[0], task_id)
        return sandbox.web.helpers.redirect_to(resources_url, local=False)

    @classmethod
    def _custom_fields_to_dict(cls, fields, template_task=False):
        data = {}
        for kv in fields or []:
            if kv:
                name, value = str(kv["name"]), kv["value"]
                if isinstance(value, six.string_types) and not template_task:
                    if not value.strip():
                        value = None
                data[name] = value
        return data

    @classmethod
    @request_context.timer_decorator()
    def update_and_validate_custom_fields(
        cls, user, task, input_data=None, output_data=None, requirements_data=None, validate_only=False
    ):
        if not isinstance(task, controller.TaskWrapper):
            doc = cls._document(task)
            if not validate_only:
                forbidden = None
                if input_data and doc.execution.status != ctt.Status.DRAFT:
                    forbidden = "It's forbidden to modify input parameters of task in status {}".format(
                        doc.execution.status
                    )
                elif not controller.user_has_permission(user, (doc.author, doc.owner)):
                    forbidden = "User `{}` is not permitted to modify task #{}".format(user.login, doc.id)
                if forbidden:
                    return misc.json_error(httplib.FORBIDDEN, forbidden)
            task = controller.TaskWrapper(doc)

        try:
            template_task = getattr(task.model, "template_alias", None) is not None
            update_result = task.update_parameters(
                cls._custom_fields_to_dict(input_data, template_task=template_task),
                cls._custom_fields_to_dict(output_data),
                requirements_data, validate_only
            )
        except tb_errors.OutputParameterReassign as ex:
            return misc.json_error(
                httplib.FORBIDDEN,
                "Output parameters for task #{} are already set: {}".format(task.model.id, ", ".join(ex.args[0]))
            )

        if update_result.required:
            pass  # Temporary disabled till UI-side fix.
            # return misc.json_error(httplib.BAD_REQUEST, "Fields are required: {}".format(sorted(required)))
        checks = []
        _VS = ctt.FieldValidationStatus
        bad_status = _VS.ERROR if validate_only else _VS.WARNING
        for kv in common.utils.chain(input_data, output_data):
            if kv:
                name = str(kv["name"])
                error = update_result.errors.get(name)
                checks.append({
                    "name": name,
                    "status": bad_status if error else _VS.SUCCESS,
                    "message": error or "",
                })
        return task, checks

    @classmethod
    def update_custom_fields(cls, request, tid):
        task, res = cls.update_and_validate_custom_fields(
            request.user, tid, input_data=misc.request_data(request, list)
        )
        task.on_save()
        task.save()
        return misc.response_json(map(dict, res))

    @classmethod
    def update_output(cls, request):
        data = misc.request_data(request, list)
        task, res = cls.update_and_validate_custom_fields(request.user, request.session.task, output_data=data)
        output_params = {p.key: p.value for p in task.model.parameters.output}
        reset_on_restart = {item["name"]: bool(item.get("reset_on_restart")) for item in data}
        for r in res:
            if r["status"] != ctt.FieldValidationStatus.SUCCESS:
                continue
            name = r["name"]
            task.model.update(
                add_to_set__parameters__output=cls.Model.Parameters.Parameter(
                    key=name, value=output_params[name], reset_on_restart=reset_on_restart[name]
                )
            )
        # noinspection PyProtectedMember
        if "h" in task.model._changed_fields:
            task.model.update(set__hints=task.model.hints)
        return misc.response_json(map(dict, res))

    @classmethod
    def validate_custom_fields(cls, request, tid):
        _, res = cls.update_and_validate_custom_fields(
            request.user, tid, input_data=misc.request_data(request, list), validate_only=True
        )
        return misc.response_json(map(dict, res))

    @classmethod
    def create_output_trigger(cls, request):
        data = misc.request_data(request)
        try:
            timeout = data.get("timeout")
            if timeout:
                timeout = int(timeout)

            targets = data.get("targets")

            task_ids, fields = [], []
            for tid, flds in targets.iteritems():
                for fld in common.utils.chain(flds):
                    task_ids.append(int(tid))
                    fields.append(fld)

            if not task_ids:
                raise ValueError("No output parameters to wait for")

            tasks = dict(
                cls.Model.objects(
                    id__in=task_ids,
                ).read_preference(pymongo.ReadPreference.PRIMARY).scalar("id", "parameters__output")
            )
            for tid in tasks.keys():
                tasks[tid] = set(p.key for p in tasks[tid])

            missing = set(task_ids) - set(tasks)
            if missing:
                missing_text = ", ".join(map(str, missing))
                raise ValueError("Cannot wait for non-existing task(s): {}".format(missing_text))

            tf_model = controller.TaskOutputTrigger.Model.TargetField
            targets = [
                tf_model(target=tid, field=field) for tid, field in it.izip(task_ids, fields)
                if field not in tasks[tid]
            ]

            wait_all = data["wait_all"]
            if not targets or (not wait_all and len(task_ids) != len(targets)):
                raise sandbox.web.response.HttpResponse(code=httplib.NOT_ACCEPTABLE)
        except (KeyError, TypeError, ValueError) as ex:
            return misc.json_error(httplib.BAD_REQUEST, str(ex))

        try:
            controller.TaskOutputTrigger.create(controller.TaskOutputTrigger.Model(
                source=request.session.task,
                targets=targets,
                wait_all=wait_all,
                token=request.session.token
            ))
        except controller.TaskOutputTrigger.AlreadyExists:
            return sandbox.web.response.HttpResponse(code=httplib.CONFLICT)
        if timeout:
            try:
                controller.TimeTrigger.create(controller.TimeTrigger.Model(
                    source=request.session.task,
                    time=dt.datetime.utcnow() + dt.timedelta(seconds=timeout)
                ))
            except controller.TimeTrigger.AlreadyExists:
                return sandbox.web.response.HttpResponse(code=httplib.CONFLICT)
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)


class TaskReport(PathBase(api.v1.task.TaskReport)):
    Model = mapping.Task

    @classmethod
    def get(cls, _, task_id, label):
        task = controller.TaskWrapper(cls._document(task_id))
        label = label.lower()
        try:
            data = task.report(label)
        except tb_errors.UnknownReportName:
            return misc.json_error(httplib.NOT_FOUND, "There's no report with label '{}'".format(label))
        if not data:
            data = []
        elif isinstance(data, tuple):
            data = [{"helperName": data[0], "content": data[1]}]
        elif not isinstance(data, list):
            data = [{"content": data}]
        return misc.response_json(data)


class TaskCustomFooter(PathBase(api.v1.task.TaskCustomFooter)):
    # TODO: SANDBOX-4706 drop this class and corresponding API path in web.api.* after UI is switched to task reports
    @classmethod
    def get(cls, _, task_id):
        redirect_url = "".join((common.utils.server_url(), api.v1.Api.basePath, TaskReport.api_path.path))
        return sandbox.web.helpers.redirect_to(redirect_url.format(id=task_id, label="footer"), local=False)


registry.registered_json('task')(Task.list)
registry.registered_json('task', ctm.RequestMethod.POST)(Task.create)
registry.registered_json('task/audit')(Task.list_audit)
registry.registered_json('task/current', restriction=ctu.Restriction.TASK)(Task.current)
registry.registered_json('task/current/context', ctm.RequestMethod.PUT, ctu.Restriction.TASK)(
    Task.update_current_context
)
registry.registered_json('task/current/context/value', ctm.RequestMethod.PUT, ctu.Restriction.TASK)(
    Task.update_current_context_value
)
registry.registered_json('task/current/execution', ctm.RequestMethod.PUT, ctu.Restriction.TASK)(
    Task.update_current_execution
)
registry.registered_json('task/current/audit', ctm.RequestMethod.POST, ctu.Restriction.TASK)(
    Task.create_audit
)
registry.registered_json('task/(\d+)')(Task.get)
registry.registered_json('task/(\d+)', ctm.RequestMethod.PUT)(Task.update)
registry.registered_json('task/(\d+)/audit')(Task.audit)
registry.registered_json('task/(\d+)/audit', ctm.RequestMethod.POST, ctu.Restriction.AUTHENTICATED)(Task.create_audit)
registry.registered_json('task/(\d+)/queue')(Task.queue)
registry.registered_json('task/(\d+)/context')(Task.context)
registry.registered_json('task/(\d+)/children')(Task.children)
registry.registered_json('task/(\d+)/dependant')(Task.dependant)
registry.registered_json('task/(\d+)/custom/fields')(Task.custom_fields)
registry.registered_json('task/(\d+)/custom/fields', ctm.RequestMethod.PUT)(Task.update_custom_fields)
registry.registered_json('task/(\d+)/custom/fields', ctm.RequestMethod.POST)(Task.validate_custom_fields)
registry.registered_json('task/(\d+)/output')(Task.output)
registry.registered_json('task/(\d+)/release')(Task.release)
registry.registered_json('task/(\d+)/release', ctm.RequestMethod.POST, ctu.Restriction.TASK)(Task.create_release)
registry.registered_json('task/(\d+)/release', ctm.RequestMethod.DELETE, ctu.Restriction.TASK)(Task.delete_release)
registry.registered_json('task/current/release', ctm.RequestMethod.POST, ctu.Restriction.TASK)(
    Task.create_current_release
)
registry.registered_json('task/current/release', ctm.RequestMethod.DELETE, ctu.Restriction.TASK)(
    Task.delete_current_release
)
registry.registered_json('task/(\d+)/resources')(Task.resources)
registry.registered_json('task/(\d+)/requirements')(Task.requirements)
registry.registered_json('task/(\d+)/tags', ctm.RequestMethod.POST)(Task.add_tags)
registry.registered_json('task/(\d+)/tags', ctm.RequestMethod.DELETE)(Task.delete_tags)
registry.registered_json('task/(\d+)/tags/(.+)', ctm.RequestMethod.DELETE)(Task.delete_tag)
registry.registered_json('task/(\d+)/hints/', ctm.RequestMethod.POST)(Task.add_hints)
registry.registered_json('task/current/hints/', ctm.RequestMethod.POST, ctu.Restriction.TASK)(Task.add_current_hints)
registry.registered_json('task/current/trigger/output', ctm.RequestMethod.POST, ctu.Restriction.TASK)(
    Task.create_output_trigger
)
registry.registered_json('task/current/output', restriction=ctu.Restriction.TASK)(Task.output)
registry.registered_json('task/current/output', ctm.RequestMethod.PUT, ctu.Restriction.TASK)(Task.update_output)
registry.registered_json('task/meta', ctm.RequestMethod.POST)(Task.task_meta)
