import re
import json
import httplib
import logging
import datetime as dt

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.template as ctte

import sandbox.yasandbox.proxy.task
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

from sandbox.yasandbox.api.json import Base
from sandbox.yasandbox.api.json import registry
from sandbox.yasandbox.api.json import mappers
from sandbox.yasandbox.api.json import misc

import sandbox.web.helpers
import sandbox.web.response


class Template(Base):
    """
    The class encapsulates all the logic related to REST API representation of any entities related to template object.
    """

    logger = logging.getLogger("RESTAPI_Template")
    ALIAS_RE = re.compile(r"\A[A-Za-z][\w]{3,}\Z")
    RESERVED_ALIASES = {"audit", "current"}

    Model = mapping.TaskTemplate

    class TemplateView(dict):
        def __init__(self, user, doc):
            tm = mappers.TemplateMapper(None, user)
            self.update(tm.dump(doc))

    class CreationType(common.enum.Enum):
        NEW = None
        FROM_TASK = None
        FROM_TEMPLATE = None

    @classmethod
    def _document(cls, obj_id):
        doc = cls.Model.objects.with_id(obj_id)
        if not doc:
            return misc.json_error(httplib.NOT_FOUND, "{} {} not found.".format(cls.__name__, obj_id))
        return doc

    @classmethod
    def check_alias(cls, alias):
        if not alias or cls.ALIAS_RE.match(alias) is None:
            return misc.json_error(
                httplib.BAD_REQUEST,
                "Alias is required and must starts with english letter and "
                "contains of 4 and more english letters, digits and '_'"
            )

        if alias in alias in cls.RESERVED_ALIASES or mapping.TaskTemplate.objects(alias=alias).count():
            return misc.json_error(
                httplib.BAD_REQUEST,
                "Alias '{}' already used.".format(alias)
            )

    @classmethod
    def _get_creation_type(cls, data):
        if data.get("task_id") is not None and data.get("template_alias") is not None:
            raise ValueError("Template can not be created from task and another template simultaneously")
        if data.get("task_id") is not None:
            return cls.CreationType.FROM_TASK
        if data.get("template_alias") is not None:
            return cls.CreationType.FROM_TEMPLATE
        return cls.CreationType.NEW

    @staticmethod
    def _ensure_user_permission(user, groups):  # type: (mapping.User, List[str]) -> List[str]
        return groups if controller.user_has_permission(user, groups) else (groups or []) + [user.login]

    @classmethod
    def create(cls, request):
        data = misc.request_data(request)
        template = None
        try:
            creation_type = cls._get_creation_type(data)
        except ValueError as exc:
            return misc.json_error(httplib.BAD_REQUEST, str(exc))

        alias = data.get("alias")
        check = cls.check_alias(alias)
        if check:
            return check

        user = request.user

        if creation_type == cls.CreationType.NEW:
            task = mapping.TaskTemplate.Task(type=data["task_type"])
            try:
                template = controller.TemplateWrapper(
                    mapping.TaskTemplate(
                        author=user.login,
                        alias=alias,
                        description=data.get("description"),
                        shared_with=cls._ensure_user_permission(user, data.get("shared_with")),
                        task=task
                    ),
                    tasks_resource_id=data.get("tasks_resource"),
                    tasks_resource_filter=data.get("tasks_resource_filter")
                ).create()
            except AttributeError as exc:
                misc.json_error(
                    httplib.UNPROCESSABLE_ENTITY,
                    "Can't create the template. Reason: {}".format(exc.message)
                )
            tasks_resource = controller.Template.requirements_dict_from_task(template.model.task)["tasks_resource"]
            tasks_resource.value = data.get("tasks_resource")
            tasks_resource.filter = common.api.filter_query(
                data.get("tasks_resource_filter"), controller.Resource.LIST_QUERY_MAP
            )
            if tasks_resource.value is not None or tasks_resource.filter:
                tasks_resource.default_from_code = False
        elif creation_type == cls.CreationType.FROM_TEMPLATE:
            template_alias = data["template_alias"]
            source_template = mapping.TaskTemplate.objects().with_id(template_alias)
            if source_template is None:
                return misc.json_error(httplib.NOT_FOUND, "Template {} not found.".format(template_alias))
            template = controller.Template.copy(
                source_template,
                alias,
                cls._ensure_user_permission(user, data.get("shared_with")),
            )
        elif creation_type == cls.CreationType.FROM_TASK:
            task_id = data["task_id"]
            task_model = mapping.Task.objects().with_id(task_id)
            if task_model is None:
                return misc.json_error(httplib.NOT_FOUND, "Task with id {} not found.".format(task_id))
            task = controller.TaskWrapper(task_model)

            template_task_model = mapping.TaskTemplate.Task(
                type=task_model.type
            )
            template_model = mapping.TaskTemplate(
                author=user.login,
                alias=alias,
                description=data.get("description"),
                shared_with=cls._ensure_user_permission(user, data.get("shared_with")),
                task=template_task_model,
                status=ctte.Status.READY
            )
            try:
                template = controller.TemplateWrapper.create_wrapper_from_task(
                    template_model, task_model
                ).copy_from_task(task)
            except ValueError as error:
                return misc.json_error(httplib.BAD_REQUEST, error.message)

        template.model.time.created = dt.datetime.utcnow()
        try:
            template.save()
        except mapping.NotUniqueError as ex:
            return misc.json_error(httplib.BAD_REQUEST, str(ex))

        return sandbox.web.helpers.response_created(
            "{}/{}".format(request.uri, template.model.alias),
            content_type="application/json",
            content=json.dumps(
                cls.TemplateView(
                    request.user, template.model
                ),
                ensure_ascii=False, encoding="utf-8"
            )
        )

    @classmethod
    def _update_template_fields(cls, data, template, audit):
        for field in ("description", "shared_with"):
            if data.get(field) is not None:
                if getattr(template, field) != data[field]:
                    audit.properties[field] = audit.ChangedProperty(old=getattr(template, field), new=data[field])
                setattr(template, field, data[field])

    @classmethod
    def update(cls, request, template_alias):
        data = misc.request_data(request)
        template = cls._document(template_alias)
        user_login = request.user.login
        if not controller.user_has_permission(request.user, template.shared_with):
            return misc.json_error(
                httplib.FORBIDDEN,
                "User {} has no permissions to update template {}".format(user_login, template.alias)
            )
        now = dt.datetime.utcnow()
        audit = mapping.TemplateAudit(template_alias=template_alias, date=now, author=user_login)
        cls._update_template_fields(data, template, audit)
        if not controller.user_has_permission(request.user, template.shared_with):
            return misc.json_error(
                httplib.BAD_REQUEST,
                "User {} loses access to template after update".format(user_login),
            )
        task_resource_update = data.get("task", {}).get("requirements", {}).get("tasks_resource", {})
        if "value" in task_resource_update and task_resource_update["value"] is None:
            controller.Template.requirements_dict_from_task(template.task)["tasks_resource"].value = None
        if "filter" in task_resource_update.get("meta", {}) and task_resource_update["meta"]["filter"] is None:
            controller.Template.requirements_dict_from_task(template.task)["tasks_resource"].filter = None
        template_wrapper = controller.TemplateWrapper(
            template,
            tasks_resource_id=task_resource_update.get("value"),
            tasks_resource_filter=task_resource_update.get("meta", {}).get("filter")
        )
        try:
            template_wrapper.update_template(data.get("task", {}), audit)
        except ValueError as error:
            return misc.json_error(httplib.BAD_REQUEST, error.message)
        template.time.updated = now
        template.save()
        if any((audit.input_parameters, audit.common_parameters, audit.requirements, audit.properties)):
            audit.save()
        return sandbox.web.response.HttpResponse(code=httplib.NO_CONTENT)


registry.registered_json("template", ctm.RequestMethod.POST)(Template.create)
registry.registered_json("template/([\w]+)", ctm.RequestMethod.PUT)(Template.update)
