import numbers

import sandbox.common.types.misc as ctm

from sandbox.yasandbox import controller

import sandbox.yasandbox.database.mapping

from sandbox.yasandbox.api.json import registry
from sandbox.yasandbox.api.json import misc

import sandbox.web.helpers


class User(object):
    @classmethod
    def current(cls, request):
        role = 'anonymous'
        if request.user.super_user:
            role = 'administrator'
        elif request.user != controller.User.anonymous:
            role = 'regular'
        return misc.response_json({
            'login': request.user.login,
            'role': role
        })

    @classmethod
    def groups(cls, request):
        ret = []
        base_url = request.uri.rsplit('/', 3)[0]
        for g in sorted(controller.Group.get_user_groups(request.user.login, True), key=lambda x: x.name):
            ret.append({
                "name": g.name,
                "url": "{}/group/{}".format(base_url, g.name),
                "priority_limits": {
                    attr: dict(zip(
                        ("class", "subclass"),
                        controller.Group.allowed_priority(None, g.name, request.user, attr, False).__getstate__()
                    ))
                    for attr in ("ui", "api")
                },
            })
        return misc.response_json(ret)

    @classmethod
    def prefs_get(cls, request, subnode):
        usr = controller.User.get(request.user.login)
        ret = usr.preferences.get(subnode) if usr else None
        if ret is None:
            return misc.json_error(404, "Current user's preferences subnode '{}' not found.".format(subnode))
        return misc.response_json(ret)

    @classmethod
    def prefs_set(cls, request, subnode):
        try:
            user = controller.User.get(request.user.login)
            if not user:
                raise Exception("User '{}' not found in the database.".format(request.user.login))
            user.preferences[subnode] = misc.request_data(request, (dict, list, basestring, numbers.Number))
            user.save()
        except sandbox.yasandbox.database.mapping.InvalidDocumentError as ex:
            return misc.json_error(400, "Unable to encode data: {}".format(ex))
        return sandbox.web.helpers.response_created(request.uri)


registry.registered_json('user/current')(User.current)
registry.registered_json('user/current/groups')(User.groups)
registry.registered_json('user/current/preferences/([^/]+)')(User.prefs_get)
registry.registered_json('user/current/preferences/([^/]+)', method=ctm.RequestMethod.PUT)(User.prefs_set)
