import json
import itertools

import six
from six.moves import http_client

from sandbox.common import itertools as common_it
from sandbox.common.types import user as ctu
from sandbox.common.types import misc as ctm

from sandbox.web import helpers as web_helpers
from sandbox.web import response

from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.controller import user as user_controller
from sandbox.yasandbox.controller import vault as vault_controller

from sandbox.yasandbox.api import json as api_base
from sandbox.yasandbox.api.json import misc
from sandbox.yasandbox.api.json import registry


class Vault(api_base.Base):
    """
    The class encapsulates all the logic related to REST API representation of any entities related to resource object.
    """

    # Shortcuts for database models.
    Model = mapping.Vault

    # A list of list operation query parameters mapping.
    LIST_QUERY_MAP = (
        api_base.Base.QueryMapping("limit", "limit", None, int),
        api_base.Base.QueryMapping("offset", "offset", None, int),
        api_base.Base.QueryMapping("order", "order_by", None, str),
        api_base.Base.QueryMapping("name", "name", None, str),
        api_base.Base.QueryMapping("owner", "owner", None, str),
        api_base.Base.QueryMapping("shared", "allowed_users", None, str),
        api_base.Base.QueryMapping("description", "description", None, str),
    )

    class BaseEntry(dict):
        def __init__(self, user, base_url, doc, user_groups_and_login):
            rights = None
            if (user.super_user and not user.robot) or doc.owner in user_groups_and_login:
                rights = ctu.Rights.WRITE
            elif doc.allowed_users and any(g_or_l in doc.allowed_users for g_or_l in user_groups_and_login):
                rights = ctu.Rights.READ
            super(Vault.BaseEntry, self).__init__({
                "id": doc.id,
                "rights": rights,
                "name": doc.name,
                "owner": doc.owner,
                "url": "{}/{}".format(base_url, doc.id),
                "shared": doc.allowed_users,
                "description": doc.description,
            })

    class ListItemEntry(BaseEntry):
        pass

    class Entry(BaseEntry):
        def __init__(self, user, base_url, doc, user_groups_and_login):
            super(Vault.Entry, self).__init__(user, base_url, doc, user_groups_and_login)
            self["data_length"] = vault_controller.Vault.data_length(doc.data)

    @staticmethod
    def __check_owner_setup(owner, user, user_groups):
        return (
            None
            if user.super_user or owner in user_groups else
            misc.json_error(
                http_client.FORBIDDEN,
                "User {} has no permissions to set '{}' as owner for vault.".format(user.login, owner)
            )
        )

    @staticmethod
    def __check_allowed_users(data):
        allowed_users = data.get("shared")
        if allowed_users:
            try:
                user_controller.validate_credentials(allowed_users)
            except Exception as ex:
                return misc.json_error(http_client.BAD_REQUEST, "User validation error: {}".format(ex))

    @staticmethod
    def __check_user_permissions(doc, user, user_groups):
        if not (user.super_user and not user.robot) and doc.owner not in user_groups:
            return misc.json_error(
                http_client.FORBIDDEN,
                "User {} not permitted to make changes in vault.".format(user.login)
            )

    @classmethod
    def list(cls, request):
        # Parse query arguments and feed them as keyword arguments to database query builder
        try:
            kwargs, offset, limit = cls._handle_args(request)
        except (TypeError, ValueError) as ex:
            return misc.json_error(http_client.BAD_REQUEST, "Query parameter validation error: " + str(ex))
        if limit is None:
            return misc.json_error(http_client.BAD_REQUEST, "Required parameter 'limit' not provided")

        order, task = kwargs.pop("order_by", None), None
        from_task = request.session and request.session.task
        if from_task:
            task = mapping.Task.objects.with_id(request.session.task)
            if not task:
                return misc.json_error(
                    http_client.INTERNAL_SERVER_ERROR,
                    "Unknown task #{}".format(request.session.task)
                )
        if not task and request.user.super_user:
            groups_and_login = None
        else:
            for_uog = task.owner if task else request.user.login
            groups_and_login = set(common_it.chain(
                user_controller.Group.get_user_groups(for_uog),
                for_uog
            ))
            if task:
                groups_and_login.update(common_it.chain(
                    user_controller.Group.get_user_groups(task.author),
                    task.author
                ))
        query = cls.Model.objects(mapping.Q(**kwargs))
        total = query.count()
        if order:
            query = query.order_by(order)
        query = query if not offset else query.skip(offset)

        items = [
            cls.ListItemEntry(request.user, request.uri, doc, groups_and_login)
            for doc in list(query if from_task else query.limit(limit))
        ]
        if not limit:
            total = len(items)
        items = filter((lambda item: item["rights"]) if from_task else None, items)
        return misc.response_json({
            "limit": limit,
            "offset": offset,
            # SANDBOX-4427: pretend it's the only entry if it's the only one accessible
            "total": 1 if len(items) == 1 else total,
            "items": items[:limit]
        })

    @classmethod
    def get(cls, request, obj_id):
        user_groups_and_login = set(itertools.chain(
            user_controller.Group.get_user_groups(request.user.login), (request.user.login,)
        ))
        doc = cls._document(obj_id)
        if not doc:
            return misc.json_error(http_client.NOT_FOUND, "Vault with ID '{}' not found".format(obj_id))
        check_result = cls.__check_user_permissions(doc, request.user, user_groups_and_login)
        if check_result:
            return check_result
        return misc.response_json(
            cls.Entry(request.user, request.uri.rsplit("/", 1)[0], doc, user_groups_and_login)
        )

    @classmethod
    def create(cls, request):
        data = misc.request_data(request)

        fields = "owner", "name", "data"
        if any(field not in data for field in fields):
            return misc.json_error(
                http_client.BAD_REQUEST,
                "Field(s): {} - is(are) required".format(", ".join(field for field in fields if field not in data))
            )
        owner, name, vault_data = (data[key] for key in fields)
        description = data.get("description")

        user_groups = set(itertools.chain(
            user_controller.Group.get_user_groups(request.user.login), (request.user.login,)
        ))

        for result in (cls.__check_owner_setup(owner, request.user, user_groups), cls.__check_allowed_users(data)):
            if result:
                return result

        try:
            vault = vault_controller.Vault.create(
                cls.Model(
                    owner=owner, name=name, allowed_users=data.get("shared"),
                    data=str(vault_data), description=description
                )
            )
        except vault_controller.Vault.AlreadyExists as ex:
            return misc.json_error(http_client.BAD_REQUEST, str(ex))
        return web_helpers.response_created(
            "{}/{}".format(request.uri, vault.id),
            content_type="application/json",
            content=json.dumps(
                cls.Entry(request.user, request.uri, vault, user_groups),
                ensure_ascii=False, encoding="utf-8"
            )
        )

    @classmethod
    def update(cls, request, vault_id):
        data = misc.request_data(request)
        doc = cls._document(vault_id)
        if not doc:
            return misc.json_error(http_client.NOT_FOUND, "Document {} not found".format(vault_id))
        user_groups = set(itertools.chain(
            user_controller.Group.get_user_groups(request.user.login), (request.user.login,)
        ))
        checks = (
            cls.__check_user_permissions(doc, request.user, user_groups),
            cls.__check_owner_setup(data["owner"], request.user, user_groups) if "owner" in data else None,
            cls.__check_allowed_users(data),
        )
        for result in checks:
            if result:
                return result

        updatable_fields = ["shared", "name", "owner", "description"]

        vault_data = data.get("data")
        if vault_data:
            updatable_fields.append("data")
            data["data"] = vault_data.encode("utf-8") if isinstance(vault_data, six.text_type) else vault_data

        for name, alias in itertools.izip_longest(updatable_fields, ("allowed_users",)):
            if name in data:
                setattr(doc, (alias or name), data[name])
        vault_controller.Vault.update(request.user, doc, encrypt=bool(vault_data))

        return (
            misc.response_json(cls.Entry(request.user, request.uri, cls.Model.objects.with_id(doc.id), user_groups))
            if cls.request_needs_updated_data(request) else
            response.HttpResponse(code=http_client.NO_CONTENT)
        )

    @classmethod
    def delete(cls, request, vault_id):
        doc = cls._document(vault_id)
        if not doc:
            return misc.json_error(http_client.NOT_FOUND, "Document {} not found".format(vault_id))
        user_groups = itertools.chain(
            user_controller.Group.get_user_groups(request.user.login), (request.user.login,)
        )
        result = cls.__check_user_permissions(doc, request.user, user_groups)
        if result:
            return result
        vault_controller.Vault.delete(request.user, doc)
        return response.HttpResponse(code=http_client.NO_CONTENT)

    @classmethod
    def data(cls, request, vault_id):
        doc = cls._document(vault_id)
        if not doc:
            return misc.json_error(http_client.NOT_FOUND, "Document {} not found".format(vault_id))
        task = mapping.Task.objects.with_id(request.session.task)
        if not task:
            return misc.json_error(http_client.NOT_FOUND, "Task #{} not found".format(request.session.task))
        try:
            data = vault_controller.Vault.encrypt_data(doc, task, request.session.vault)
        except vault_controller.Vault.NotAllowed as ex:
            return misc.json_error(http_client.FORBIDDEN, str(ex))
        return response.HttpResponse("application/octet-stream", data)


registry.registered_json("vault")(Vault.list)
registry.registered_json("vault", ctm.RequestMethod.POST, ctu.Restriction.AUTHENTICATED)(Vault.create)
registry.registered_json("vault/(\d+)")(Vault.get)
registry.registered_json("vault/(\d+)", ctm.RequestMethod.PUT, ctu.Restriction.AUTHENTICATED)(Vault.update)
registry.registered_json(
    "vault/(\d+)", ctm.RequestMethod.DELETE, ctu.Restriction.AUTHENTICATED
)(Vault.delete)
registry.registered_json("vault/(\d+)/data", restriction=ctu.Restriction.TASK)(Vault.data)
