import time
import httplib
import xmlrpclib
import xml.parsers.expat

import common.log
import common.proxy
import common.errors
import common.crypto
import common.config
import common.profiler
import common.statistics
import common.types.task
import common.types.user
import common.types.database

import yasandbox.manager
import yasandbox.controller.user
import yasandbox.database.mapping
import yasandbox.database.mapping.base
from sandbox.yasandbox import controller

import web.helpers

import serviceq.errors

from yasandbox.api.xmlrpc import task
from yasandbox.api.xmlrpc import resource
from yasandbox.api.xmlrpc import release
from yasandbox.api.xmlrpc import client
from yasandbox.api.xmlrpc import misc
from yasandbox.api.xmlrpc import registry


__all__ = [
    'task',
    'resource',
    'release',
    'client',
    'misc',
]


logger = common.log.LogLazyLoader(common.log.get_server_log, ('xmlrpc', ))


class XMLRPCDispatcher(object):
    def __init__(self, request):
        self.started = time.time()
        # parse data
        try:
            self.args, self.method = xmlrpclib.loads(request.raw_data)
        except xml.parsers.expat.ExpatError as ex:
            logger.exception("Error parsing XMLRPC request")
            web.helpers.response_error(
                httplib.BAD_REQUEST,
                headers={"X-Error-Message": "{}: {}".format(ex.__class__.__name__, ex.message)}
            )

        # Update request parameters
        request.source = request.Source.RPC
        request.remote_method = self.method

    def __call__(self, request):
        settings = common.config.Registry()

        # log all debug info
        logger.debug(
            "Request: '%s', login '%s', method: '%s', arguments: %r, session: %r",
            request.id, request.user.login, self.method, self.args, request.session
        )

        # check read only mode
        read_only = (
            controller.Settings.mode() == controller.Settings.OperationMode.READ_ONLY
        )
        if read_only and self.method not in registry.RO_ALLOWED_METHODS:
            raise common.errors.ViewError(
                "Execution of the method '{}' is denied due to ReadOnly state".format(self.method)
            )
        if (
            not request.session and
            self.method in registry.RO_ALLOWED_METHODS and
            request.read_preference == common.types.database.ReadPreference.PRIMARY
        ):
            request.read_preference = common.types.database.ReadPreference.PRIMARY_PREFERRED
        elif request.user != common.types.user.ANONYMOUS_LOGIN and self.method not in registry.RO_ALLOWED_METHODS:
            request.read_preference = common.types.database.ReadPreference.PRIMARY

        # process request
        try:
            # get method name
            try:
                method = registry.XMLRPC_METHODS[self.method]
            except KeyError:
                raise common.errors.ViewError('unknown RPC method {0}'.format(self.method))

            # check user rights
            if (
                settings.server.auth.enabled and
                getattr(method, 'protected', None) and
                not request.user.super_user and
                self.method not in request.user.allowed_api_methods and
                not request.session
            ):
                raise common.errors.AuthorizationError(
                    'User "{}" not allowed to call method "{}"'.format(request.user.login, self.method))

            # call method
            yasandbox.database.mapping.base.tls.request = request
            if getattr(method, 'request', None):
                response = method(request, *self.args)
            else:
                response = method(*self.args)
            if response is None:
                response = True
            return xmlrpclib.dumps((response,), '', True, allow_none=True)

        except common.proxy.ReliableServerProxy.SessionExpired as ex:
            rc = common.proxy.ReliableServerProxy.ErrorCodes.FATAL_ERROR
            logger.exception('Error calling method %r (args %r) - invalid session (%s).', self.method, self.args, ex)
            return xmlrpclib.dumps(
                xmlrpclib.Fault(rc, '{0}: {1}'.format(ex.__class__.__name__, ex.message)),
                '', True
            )
        except Exception as ex:
            # set retry flag if needed
            if isinstance(ex, (
                yasandbox.database.mapping.OperationFailure,
                yasandbox.database.mapping.OperationError,
                serviceq.errors.QNeedValidation,
                serviceq.errors.QRetry
            )):
                if yasandbox.database.mapping.is_query_error(ex):
                    rc = common.proxy.ReliableServerProxy.ErrorCodes.ERROR
                else:
                    rc = common.proxy.ReliableServerProxy.ErrorCodes.RETRYABLE_ERROR
            else:
                rc = common.proxy.ReliableServerProxy.ErrorCodes.ERROR

            if isinstance(ex, common.errors.SandboxException):
                logger.error('Error calling method %r (args %r): %s', self.method, self.args, ex)
            else:
                logger.exception('Error calling method %r (args %r).', self.method, self.args)
            return xmlrpclib.dumps(
                xmlrpclib.Fault(rc, '{0}: {1}'.format(ex.__class__.__name__, ex.message)),
                '', True
            )
        finally:
            ts = int((time.time() - self.started) * 1000)
            request.measures.processing_time = ts
            request.handler = self.args[0] if self.method == misc.managerCall.__name__ else self.method
            logger.info(
                "Request '%s' login '%s' from client '%s' to method '%s' processed in %sms",
                request.id, request.user.login, request.client_address, self.method, ts
            )
