# coding: utf-8

import time
import xmlrpclib
import cPickle as pickle

from common import log
from common import errors
import common.types.notification

from yasandbox.proxy import task as task_proxy
from yasandbox.database import mapping
from yasandbox import manager
from yasandbox import controller

from yasandbox.api.xmlrpc import registry


logger = log.LogLazyLoader(log.get_server_log, ('xmlrpc', ))


###########################################################
# Manager
###########################################################

@registry.xmlrpc_method(protected=True)
def managerCall(request, command, args, kwargs):
    def _getManager(managerName):
        if managerName == 'task':
            return manager.task_manager
        elif managerName == 'resource':
            return manager.resource_manager
        elif managerName == 'client':
            return manager.client_manager
        elif managerName == 'notification':
            return manager.notification_manager
        return None

    try:
        managerName, methodName = command.split('_', 1)
    except:
        raise errors.ViewError('incorrect executor protocol methodName format: {0}'.format(command))

    fn = getattr(_getManager(managerName), methodName, None)
    if not callable(fn):
        raise errors.ViewError('unknown manager ({0}) method {1}'.format(managerName, methodName))

    # loading data from xmlrpclib.Binary wrapper and unpickling
    try:
        args = pickle.loads(args.data)
        kwargs = pickle.loads(kwargs.data)
        logger.debug(
            'Got request from client %s: executing method %s with args %s and kwargs %s',
            request.client_address, str(command), str(args), str(kwargs)
        )
    except:
        logger.exception('error while processing arguments binary data')
        raise

    try:
        mapping = None
        result = fn(*args, **kwargs)
        # dumps result data
        logger.debug('command _handle_ %s result %s', str(command), str(result))
        # Avoid passing `mongoengine.*` objects via RPC. Hack, huh.
        if isinstance(result, task_proxy.Task):
            mapping, result._mapping = result._mapping, None
        ret = pickle.dumps(result)
        if isinstance(result, task_proxy.Task):
            result._mapping = mapping
    except Exception as ex:
        logger.exception(ex)
        raise

    return xmlrpclib.Binary(ret)


###########################################################
# Notifications
###########################################################

@registry.xmlrpc_method(protected=True)
def send_email(
    request, send_to, send_cc, subject, body,
    content_type="text/plain", charset="utf-8", extra_headers=None, urgent=False, author="sandbox",
    task_id=None, view=common.types.notification.View.DEFAULT,
):
    """
    Send a email notification.

    :param send_to: list or string with address of recipient
    :param send_cc: list or string with addresses of recipients of message copy
    :param subject: notification subject
    :param body: notification body
    :param content_type: content type (text/plain or text/html)
    :param charset: encoding of the notification text
    :param extra_headers: list of auxiliary email headers
    :param task_id: task identifier
    :param view: notification view type
    :param urgent: send from sandbox-urgent@ (sandbox-noreply@ otherwise)
    """

    if not task_id and request.session:
        task_id = request.session.task

    notification = controller.Notification.save(
        controller.Notification.Transport.EMAIL,
        send_to, send_cc, subject, body,
        content_type, charset, extra_headers if extra_headers is not None else [], author,
        task_id, view, urgent=urgent
    )

    return str(notification.id)


###########################################################
# Groups
###########################################################

@registry.xmlrpc_method(protected=True)
def list_groups():
    """
    list all group names

    :return: list of groups names
    """
    return [g.name for g in controller.Group.list()]


@registry.xmlrpc_method(protected=True)
def create_group(group, email, users):
    """
    create new group

    :param group: group name
    :param email: group email
    :param users: list of users logins
    :return: group name
    """
    if not isinstance(users, list) and not isinstance(users, set):
        users = set(users.split())
    g = controller.Group.create(mapping.Group(
        name=group,
        email=email,
        users=users,
    ))
    return g.name


@registry.xmlrpc_method(protected=True)
def delete_group(group):
    """
    delete group

    :param group: group name
    """
    controller.Group.delete(group)


@registry.xmlrpc_method(protected=True)
def get_group_users(group):
    """
    get all group users

    :param group: group name
    :return: list of group users logins
    """
    return list(controller.Group.get(group).users)


@registry.xmlrpc_method(protected=True)
def get_user_groups(user):
    """
    get all user groups

    :param user: user login
    :return: list of user login groups
    """
    return list(controller.Group.get_user_groups(user))


@registry.xmlrpc_method(protected=True)
def add_user_to_group(group, user):
    """
    add user to group

    :param group: group name
    :param user: user login
    """
    controller.Group.add_user(group, user)


@registry.xmlrpc_method(protected=True)
def remove_user_from_group(group, user):
    """
    remove user from group

    :param group: group name
    :param user: user login
    :param group: group name
    """
    controller.Group.remove_user(group, user)


###########################################################
#  Service
###########################################################

@registry.xmlrpc_method(ro_allowed=True)
def ping(data="OK"):
    """
    Test server response

    :param data: data to return
    :return: data
    """
    return data


@registry.xmlrpc_method()
def sleep(amount):
    """
    Yet another XMLRPC server test. Sleeps given amount of seconds. Returns argument.
    """
    time.sleep(amount)
    return amount
