# coding: utf-8

import yasandbox.manager
import yasandbox.database.mapping

from yasandbox.api.xmlrpc import registry


###########################################################
# Public API
###########################################################

@registry.xmlrpc_method(alias='listReleases', ro_allowed=True)
def list_releases(fltr=None):
    """
        Returns dictionaries list with releases information in accordance with the filter parameters passed.

        Filter parameters:
            - **task_id**:              released task identifier
            - **limit**:                maximum amount of releases to be returned
            - **offset**:               amount of records to be skipped
            - **include_broken**:       :class:`bool` flag whether include releases
                                        with `BROKEN` resources or not (defaults to `False`)
            - **resource_type**:        filter released resource by type
            - **release_status**:       filter releases by type: `cancelled`, `stable`,
                                        `prestable`, `testing`, `unstable`.
            - **creation_ts_bound**:    :class:`tuple` `(timestamp_min, timestamp_max)` with creation time boundaries
            - **order_by**:             a string with sorting direction and field name (defaults to `-id`)

        :param fltr:    :class:`dict` with filtering parameters.
        :return:        dictionaries list with found releases information.
        :rtype: :class:`list` of :class:`dict`
    """
    if fltr is None:
        fltr = {}
    return yasandbox.manager.release_manager.list_releases(resource_as_dict=True, **fltr)


@registry.xmlrpc_method(alias='createRelease')
def create_release(
    request,
    task_id, release_status='stable',
    release_subject=None, release_comments=None, release_changelog=None,
    addresses_to=None, addresses_cc=None,
    additional_parameters=None,
):
    """
        Создать релиз.

        Для создания релиза через  xmlrpc необходимо, чтобы пользователь sandbox имел право на релиз задачи.
        Релиз будет осуществлён от его имени

        :param task_id: идентификатор задачи, ресурсы которой нужно релизить
        :type task_id: int
        :param release_status: статус релиза - stable, testing или unstable
        :type release_status: str
        :param release_subject: заголовок релиза
        :type release_subject: str
        :param release_comments: описание релиза
        :type release_comments: str
        :param release_changelog: список изменений
        :type release_changelog: list
        :param addresses_to: список адресатов
        :type addresses_to: list
        :param addresses_cc: список из логинов, которые нужно добавить в СС
        :type addresses_cc: list
        :param additional_parameters: словарь параметров для передачи в фукнцию on_release
        :type additional_parameters: dict
        :return: идентификатор созданного релиза
        :rtype: int

        Если релиз создать не удалось, вызывается исключение.
    """
    author = request.user.login
    if request.user.super_user and additional_parameters:
        author = additional_parameters.pop('author', request.user.login)

    release_subject = release_subject or ''

    return yasandbox.manager.release_manager.release_task(
        task_id=task_id, author=author,
        status=release_status,
        message_subject=release_subject, message_body=release_comments,
        changelog=release_changelog,
        addresses_to=addresses_to, addresses_cc=addresses_cc,
        additional_parameters=additional_parameters,
        request=request
    )


@registry.xmlrpc_method(protected=True)
def add_release_to_db(
    task_id, author, status,
    message_subject, message_body=None, changelog=None,
    creation_time=None
):
    yasandbox.manager.release_manager.add_release_to_db(
        task_id, author, status,
        message_subject, message_body, changelog,
        creation_time
    )


@registry.xmlrpc_method(protected=True)
def remove_release_from_db(task_id):
    yasandbox.database.mapping.Task.objects(id=task_id).update_one(unset__release=True)
    for resource in yasandbox.manager.resource_manager.list_task_resources(task_id):
        if resource.type.releasable:
            yasandbox.manager.resource_manager.drop_attr(resource.id, 'released')
