# coding: utf-8

from sandbox.common import log
from sandbox.common import errors
import sandbox.common.types.resource as ctr

from sandbox import sdk2

from sandbox.yasandbox import manager
from sandbox.yasandbox import controller
from sandbox.yasandbox.api.xmlrpc import registry


logger = log.LogLazyLoader(log.get_server_log, ('xmlrpc', ))


###########################################################
# Public API
###########################################################

@registry.xmlrpc_method(alias='listResourceTypes', ro_allowed=True)
def list_resource_types():
    return map(str, sdk2.Resource)


@registry.xmlrpc_method(alias='listResources', ro_allowed=True)
def list_resources(fltr=None):
    """
    List resources with specified filter

    *Example*:
    ``server.list_resources({'resource_type': 'TASK_LOGS', 'state': 'READY', 'any_attrs': {'do_not_remove': True}})``

    **Keys**:
        * **resource_type** *(str)* - тип ресурса
        * **task_id** *(int)* - id-таска владельца ресурса
        * **skynet_id** *(str)* - rbtorrent-id ресурса
        * **hidden** *(bool)* - показывать скрытые ресуры или нет
        * **state** *(str)* - статус ресурса
            - *NOT_READY*
            - *READY*
            - *BROKEN*
            - *DELETED*
        * **omit_failed** *(bool)* - показывать только **READY** ресурсы
        * **arch** *(str)* - архитектура ресурса
            - *any*
            - *freebsd*
            - *linux*
        * **id** *(str)* - id ресурса
        * **name_mask** *(str)* - подстрока в имени ресурса
        * **any_attrs** *(dict)* - search for any selected attributes
        * **all_attrs** *(dict)* - search for all selected attributes
        * **owner** *(str)* - владелец ресурса
        * **order_by** *(str)* - sorting condition. example: '+id', '-size'. default '-id'
        * **limit** *(str)* - кол-во эл-тов на странице
        * **offset** *(str)* - количество первых ресурсов, которые будут пропущены

        :param fltr: dict с фильтрами возвращаемого списка
        :return: кортеж с ресурсами в виде dict-ов
        :rtype: tuple
    """
    if fltr is None:
        fltr = {}
    res_type = fltr.get('type_name')
    if res_type:
        fltr['resource_type'] = res_type
        del fltr['type_name']

    # don't fetch all resources
    if not (fltr.get('task_id') or
            fltr.get('resource_type') or
            fltr.get('limit') or
            fltr.get('skynet_id')):
            raise errors.ViewError('Bad arguments %s'.format(str(fltr)))

    if 'attr_name' in fltr or 'attr_value' in fltr:
        fltr['any_attrs'] = {fltr.get('attr_name', ''): fltr.get('attr_value', '')}

    res = []
    for r in manager.resource_manager.list_task_resources(**fltr):
        if hasattr(r, 'to_dict'):
            res.append(r.to_dict())
        else:
            res.append(r)
    return res


@registry.xmlrpc_method(alias='listResources2', ro_allowed=True)
def list_resources2(fltr=None):
    if fltr is None:
        fltr = {}
    res_type = fltr.get('type_name')
    if res_type:
        fltr['resource_type'] = res_type
        del fltr['type_name']

    # don't fetch all resources
    if not (fltr.get('task_id') or
            fltr.get('resource_type') or
            fltr.get('limit') or
            fltr.get('skynet_id')):
            raise errors.ViewError('Bad arguments %s'.format(str(fltr)))

    if 'attr_name' in fltr or 'attr_value' in fltr:
        fltr['any_attrs'] = {fltr.get('attr_name', ''): fltr.get('attr_value', '')}

    res = []
    for r in manager.resource_manager.list_task_resources(**fltr):
        if hasattr(r, 'to_dict'):
            r = r.to_dict()
        r['id'] = str(r['id'])
        res.append(r)
    return res


@registry.xmlrpc_method(ro_allowed=True)
def last_resources(resource_type, state=ctr.State.READY):
    """
    Returns a list of latest resources grouped by resource type filtered by type and state.

    :param resource_type:   Resource type to filter by.
    :param state:           Resource state to filter by.
    :return:                A list of resources serialized to dicts.
    :rtype:                 list
    """

    rm = manager.resource_manager
    return [r.to_dict() for r in rm.last_resources(resource_type, state)]


@registry.xmlrpc_method(alias='getResource', ro_allowed=True)
def get_resource(resource_id):
    """
        Получить данные ресурса по его идентификатору.

        :param resource_id: идентификатор ресурса
        :return: описание ресурса в виде dict-а; если ресурс не найден, возвращется None
        :rtype: dict or None

    """
    resource = manager.resource_manager.load(resource_id)
    return resource is not None and resource.to_dict()


@registry.xmlrpc_method(alias='getResource2', ro_allowed=True)
def get_resource2(resource_id):
    resource = manager.resource_manager.load(int(resource_id))
    res = resource is not None and resource.to_dict()
    if res:
        res['id'] = str(res['id'])
    return res


@registry.xmlrpc_method(ro_allowed=True)
def bulk_resource_fields(ids, fields):
    """
    Get fields for resources group

    :param ids: resources ids
    :param fields: resource fields
    :return dict: {id: [field_value, ], }
    """
    return manager.resource_manager.bulk_fields(ids, fields, safe_xmlrpc=True)


@registry.xmlrpc_method(ro_allowed=True)
def get_resource_links(resource_id, links_type=None, include_offline=False):
    """
    Returns list of links of requested type to given resource's data sources.
    Firstly listed links to `H@SANDBOX_STORAGE` hosts.

    :param resource_id:     Resource identifier.
    :param links_type:      Type of links to return. Returns all if not provided. Can be a list of types.
    :param include_offline: Return links to hosts, which are currently marked as offline too.
    :return:                List of URLs to given resource's data location.
    """
    resource_id = int(resource_id)
    lt_t = ctr.LinkType
    links_type = links_type or [lt_t.HTTP, lt_t.RSYNC]
    links_type = links_type if isinstance(links_type, (list, tuple)) else (links_type,)
    resource = manager.resource_manager.load(resource_id)
    http = resource.get_http_links(include_offline) if lt_t.HTTP in links_type and resource else []
    rsync = resource.get_rsync_links(include_offline) if lt_t.RSYNC in links_type and resource else []
    return http + rsync


@registry.xmlrpc_method(alias='getResourceHttpLinks', ro_allowed=True)
def get_resource_http_links(resource_id, include_offline=False):
    """ Shortcut to :meth:`get_resource_links` """
    return get_resource_links(resource_id, ctr.LinkType.HTTP, include_offline)


@registry.xmlrpc_method(alias='getResourceRsyncLinks', ro_allowed=True)
def get_resource_rsync_links(resource_id, include_offline=False):
    """ Shortcut to :meth:`get_resource_links` """
    return get_resource_links(resource_id, ctr.LinkType.RSYNC, include_offline)


@registry.xmlrpc_method(alias='updateResourceLastUsage')
def touch_resource(resource_id):
    """
        | Обновить время последнего использования ресурса.
        | Для каждого хоста, где есть копия ресурса, обновлется время его использования.

        :param resource_id: resource id
        :type resource_id: integer
    """
    resource_id = int(resource_id)
    resource = manager.resource_manager.load(resource_id)
    if not resource:
        raise errors.ViewError('Invalid resource_id: {!r}'.format(resource_id))

    if not resource.is_ready():
        raise errors.ViewError(
            'Resource {} must be READY, not {}'.format(resource_id, resource.state))

    manager.resource_manager.touch(resource_id)


@registry.xmlrpc_method(alias='setResourceAttr')
def set_resource_attr(request, resource_id, name, value):
    """
        Установить ресурсу атрибут. Если атрибут уже был установлен, он перезапишется.

        :param resource_id: идентификатор ресурса
        :param name: название атрибута
        :param value: значение атрибута
        :return: True, если атрибут был установлен; False в противном случае
        :rtype: bool
    """
    resource_id = int(resource_id)
    resource = manager.resource_manager.load(resource_id)
    if not resource:
        raise errors.ViewError('Invalid resource_id: {!r}'.format(resource_id))

    # check user rights
    if not resource.user_has_permission(request.user):
        raise errors.ViewError(
            'User "{}" not allowed to set attribute for resource "{}"'.format(request.user.login, resource))

    return manager.resource_manager.set_attr(resource_id, name, value)


@registry.xmlrpc_method(alias='dropResourceAttr')
def drop_resource_attr(request, resource_id, name):
    """
        Удалить атрибут у ресурса.

        :param resource_id: идентификатор ресурса
        :param name: название атрибута
        :return: True, если атрибут был удалён или не существовал;
            False в противном случае (например, ресурса не существует)
    """
    resource_id = int(resource_id)
    resource = manager.resource_manager.load(resource_id)
    if not resource:
        raise errors.ViewError('Invalid resource_id: {!r}'.format(resource_id))

    # check user rights
    if not resource.user_has_permission(request.user):
        raise errors.ViewError(
            'User "{}" not allowed to drop attribute for resource "{}"'.format(request.user.login, resource))

    return manager.resource_manager.drop_attr(resource_id, name)


###########################################################
# Service
###########################################################

@registry.xmlrpc_method(protected=True)
def create_resource(request, parameters):
    """
    Protected service method.

    Create new resource for task with specified parameters.

    :param parameters: resource parameters dict

    :return: new resource ID
    """
    task_id = parameters['task_id']

    task = manager.task_manager.load(task_id)
    if not task:
        raise errors.ViewError('Invalid task_id: {!r}'.format(task_id))

    # check user rights
    if not task.user_has_permission(request.user):
        raise errors.ViewError(
            'User "{}" not allowed to create resource for task "{}"'.format(request.user.login, task))

    resource = task._create_resource(
        resource_desc=parameters['descr'],
        resource_filename=parameters['name'],
        resource_type=parameters['type'],
        arch=parameters.get('arch'),
        attrs=parameters.get('attrs'),
        owner=parameters.get('owner')
    )
    return resource.id


@registry.xmlrpc_method(protected=True, ro_allowed=True)
def get_resource_hosts(resource_id, get_all_resources=False):
    """
        Получить список хостов Sandbox, на которых лежат копии ресурсов

        :param resource_id: идентификатор ресурса
        :param get_all_resources: получить все ресурсы, в том числе в состоянии DELETED
        :return: список ссылок в виде строк
    """
    resource_id = int(resource_id)
    return manager.resource_manager.get_hosts(resource_id, get_all_resources)


@registry.xmlrpc_method(protected=True)
def delete_resource(resource_id, ignore_last_usage_time=False):
    """
        Удалить ресурс. Ресурс помечается как DELETED, его данные ставятся в очередь на удаление.

        :param resource_id: идентификатор ресурса
        :param ignore_last_usage_time: игнорировать последнее время использования ресурса
        :return: True, если ресурс успешно удалён, False в противном случае
    """
    resource_id = int(resource_id)
    return not manager.resource_manager.delete_resource(resource_id, ignore_last_usage_time=ignore_last_usage_time)


@registry.xmlrpc_method(protected=True)
def drop_host_resources(host, resources):
    """
    Drop host specified as data source for resource identifiers list specified.

    :param host:        Host to be dropped.
    :param resources:   Resource identifier(s) to be processed.
    :return:            Nothing.
    """
    return manager.resource_manager.drop_host_resources(host, resources)


@registry.xmlrpc_method(protected=True)
def lock_resources_for_backup(host, limit=10):
    """
    Service method, which is used to lock a chunk of immortal resources, which are located at the host
    provided, to be backed up to MDS.

    :param host:    Host, which wants to perform a backup action.
    :param limit:   Amount of resources to be locked.
    :return:        List of locked resources IDs.
    """
    return controller.Resource.lock_resources_for_backup(host, limit)
