# coding: utf-8

import datetime as dt

from sandbox.common import errors
from sandbox.common import log
import sandbox.common.types.task as ctt
import sandbox.common.types.database as ctd

from sandbox.yasandbox.proxy import task as task_proxy
from sandbox.yasandbox import manager
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping
import sandbox.yasandbox.controller.trigger as trigger_controller
import sandbox.taskbox.model.update as model_update

from sandbox.yasandbox.api.xmlrpc import registry

logger = log.LogLazyLoader(log.get_server_log, ('xmlrpc', ))


###########################################################
# Public API
###########################################################

@registry.xmlrpc_method(ro_allowed=True)
def list_task_types():
    return task_proxy.listTaskTypes()


# TODO: Remove after complete transition to new statuses [SANDBOX-2654]
@registry.xmlrpc_method(ro_allowed=True)
def list_tasks_legacy(request, fltr=None):
    status = fltr.get('status')
    if status is not None:
        fltr['status'] = ctt.Status.new_statuses(status)
    return list_tasks_impl(request, fltr)


@registry.xmlrpc_method(alias='listTasks', ro_allowed=True)
def list_tasks(request, fltr=None):
    return list_tasks_impl(request, fltr)


def list_tasks_impl(request, fltr=None):
    """
    List tasks with specified filter

    **Keys**:
        * **id** *(int)* - task id
        * **parent_id** *(int)* - task parent id

        * **task_type** *(str)* - task type string
        * **owner** *(str)* - task owner. *staff_user* or *SANDBOX_GROUP*
        * **host** *(str)* - tasks on host
        * **descr_mask** *(str)* - regexp description
        * **model** *(str)* - CPU model
        * **arch** *(str)* - os platform or system arch

        * **status** *(str)* - task execution status
            - *NOT_READY*
            - *ENQUEUED*
            - *ENQUEUING*
            - *EXECUTING*
            - *FINISHED*
            - *STOPPED*
            - *FAILURE*
            - *WAIT_CHILD*
            - *UNKNOWN*
            - *DELETED*

        * **important_only** *(bool)* - show only flagged tasks
        * **hidden** *(bool)* - show robot tasks
        * **completed_only** *(bool)* - show only completed tasks (*FINISHED* or *FAILURE*)
        * **show_childs** *(bool)* - show subtasks

        * **load** *(bool)* - return loaded dicts or ids list
        * **load_ctx** *(bool)* - load task ctx for each task or "ctx: None"

        * **order_by** *(str)* - mongodb sorting identifier. "-id" for example
        * **limit** *(int)* - limit result list
        * **offset** *(int)* - offset result list

    :return: loaded tasks or task ids
    :rtype: list of dicts or ints
    """

    if fltr is None:
        fltr = {}
    tasks = []
    if "descr_mask" in fltr:
        source_task = request.session.task if request.session else None
        logger.info(
            "Search by description. User: %s, task: %s, mask: %s",
            request.user.login, source_task, fltr["descr_mask"]
        )
    for t in manager.task_manager.list(**fltr):
        if hasattr(t, 'to_dict'):
            tasks.append(t.to_dict(safe_xmlrpc=True))
        else:
            tasks.append(t)
    return tasks


@registry.xmlrpc_method(alias='getTask', ro_allowed=True)
def get_task(task_id):
    """
    Get task

    *Example*:
    ``server.get_task(1)``

    :param task_id: идентификатор задачи
    :type task_id: integer
    :return: задача в виде dict-а
        Если таск найден, возвращается dict с описанием ресурса
        Если таск не найден, возвращается пустой dict
        Если передан параметр неправильного типа, происходит xmlrpclib.Fault
    :rtype: dict
    """
    task = manager.task_manager.load(
        int(task_id) if isinstance(task_id, basestring) and task_id.isdigit() else task_id)
    if not task:
        return {}
    return task.to_dict(safe_xmlrpc=True)


@registry.xmlrpc_method(alias='getTaskStatus', ro_allowed=True)
def get_task_status(task_id):
    task = manager.task_manager.load(
        (int(task_id) if isinstance(task_id, basestring) and task_id.isdigit() else task_id),
        load_ctx=False
    )
    return task is not None and task.status


@registry.xmlrpc_method(alias='bulkTaskFields', ro_allowed=True)
def bulk_task_fields(ids, fields, strict_mode=False):
    """
        Получить информацию сразу о большом количестве задач, указав нужные поля.

        :param ids: список идентификаторов задач
        :param fields: список с именами полей. Доступные значения:

            - timestamp – время создания таска

            - updated – время последнего изменения состояния таска

            - timestamp_start – время запуска

            - timestamp_finish – время завершения

            - owner – владелец

            - type – тип

            - enqueue – находится ли таск в очереди

            - status – состояние

            - descr – описание

            - info – информация о выполнении

            - ctx – контекст

            - host – хост где работал таск

            - parent_id – если это сабтаск, то его создатель

            - hidden – скрытый таск

            - model – модель проца

            - priority – приоритет задачи (`tuple` of priority class and subclass names)

            - arch – архитектура

            - dir_size – размер директории таска

            - important – важный таск

            - execution_space – необходимое для выполнения место

        :param strict_mode: если True, то в результате должны быть все id из ids, иначе исключение.
        :return: словарь у которого ключ - идентификатор задачи, значение - словарь с полями и их значениями
        :rtype: dict
    """
    if not (ids and fields):
        return {}
    return manager.task_manager.get_bulk_fields(ids, fields, safe_xmlrpc=True, strict_mode=strict_mode)


@registry.xmlrpc_method(alias='createTask')
def create_task(request, params):
    """
        Create new task

        *Example*:
        ``server.create_task({'type_name': 'TEST_TASK', 'owner': 'SANDBOX', 'priority': ('BACKGROUND', 'NORMAL')})``

        **Keys**:

            * **type_name *** *(str)* - тип задачи *** обязательный параметр**

            * **owner *** *(str)* - от имени кого запускать задачу *** обязательный параметр**

            * **arch** *(str)* - архитектура (например, *linux*)

            * **priority** *(str, str)* - кортеж со строками приоритета (class, subclass)
                - class - *BACKGROUND*, *SERVICE*
                - subclass - *LOW*, *NORMAL*, *HIGH*
                .. warning:: for anonimous user (*BACKGROUND*, *LOW*)

            * **ctx** *(dict)* - входные параметры задачи

            * **descr** *(str)* - описание задачи

            * **enqueue** *(bool)* - флаг, определяющий ставить ли задачу в очередь немедленно, или нет.
                - In case of `True` passed (default) - the call will block till the task actually enqueued.
                - In case of `False` passed - the task enqueuing will not be initiated at all.
                - In case of `None` the task enqueuing will be started asynchronously.

            * **ram** *(int)* - требуемое количество оперативной памяти на хосте в MiB

            * **execution_space** - требуемое место на жёстком диске на хосте в MiB

        :return: ID созданной задачи
        :rtype: integer
    """
    request.read_preference = ctd.ReadPreference.PRIMARY
    enqueue = params.pop('enqueue', True)
    if not task_proxy.isValidTaskType(params['type_name']):
        raise errors.ViewError(
            'Incorrect type_name value: unknown task type {}'.format(params['type_name']))

    priority = None
    owner = params.get('owner') or controller.Group.anonymous.name
    if not request.user.super_user and request.user.login == controller.User.anonymous.login:
        logger.warn('ANONYMOUS: %s', params['type_name'])
        owner = controller.Group.anonymous.name

    if 'priority' in params:
        allowed = controller.Group.allowed_priority(request, owner)
        priority = ctt.Priority().__setstate__(params.pop('priority'))
        if priority > allowed:
            logger.warn('Lower new task %s priority from %s to %s', params['type_name'], priority, allowed)
            priority = allowed

    author = request.user.login
    pid = params.get('parent_id')
    if request.session:
        task = manager.task_manager.load(request.session.task)
        author = task.author
        if pid:
            pid = task.id

    owner = owner or request.user.login
    tags = params.get('tags', [])
    task = manager.task_manager.create(
        task_type=params['type_name'],
        owner=owner,
        author=author,
        arch=params.get('arch', 'any'),
        model=params.get('model'),
        host=params.get('host'),
        parent_id=pid,
        priority=priority,
        parameters=params,
        request=request,
        ram=params.get('ram'),
        suspend_on_status=params.get("suspend_on_status"),
        score=params.get("score")
    )

    if params.pop('inherit_notifications', None):
        mapping.Task.objects(
            id=task.id,
        ).update_one(
            set__notifications=mapping.Task.objects.scalar("notifications").with_id(int(pid))
        )
        task.drop_notifications_from_context()
    else:
        task.create_notifications_from_context()
    if tags:
        model_update.ServerSideUpdate.update_tags(tags, task)
    manager.task_manager.update(task)

    if task.status != task.Status.EXCEPTION:
        if enqueue or enqueue is None:
            logger.debug('Wait for task #%s enqueuing.', task.id)
            manager.task_manager.enqueue_task(task, request=request)
    return task.id


@registry.xmlrpc_method()
def enqueue_task(request, task_id):
    """
    Checks the given task is still enqueuing or not. Schedules the task for enqueuing if its not yet.

    :param task_id:  Task ID to be checked.
    :return: `True` if the task was actually enqueued or `False` otherwise.
    """
    task = manager.task_manager.load(task_id)
    try:
        return manager.task_manager.enqueue_task(task, request=request, sync=True)
    except errors.IncorrectStatus:
        return task.status != ctt.Status.ENQUEUING


@registry.xmlrpc_method(alias='restartTask')
def restart_task(request, task_id):
    """
        Restart task. Ignore ctx['do_not_restart'] flag

        :param task_id: идентификатор задачи
        :return: если задача была перезапущена, возвращается True; False в противном случае
        :rtype: bool
    """
    task = manager.task_manager.load(task_id)
    if not task:
        raise errors.ViewError('Invalid task_id: {!r}'.format(task_id))

    # check user rights for
    if not task.user_has_permission(request.user):
        raise errors.ViewError(
            'User "{}" not allowed to restart task "{}"'.format(request.user.login, task))

    return task.restart(request=request)


@registry.xmlrpc_method(alias='cancelTask')
def cancel_task(request, task_id):
    """
        Остановить задачу, перевести её в состояние STOPPED

        :param task_id: идентификатор задачи
        :return:
    """
    task = manager.task_manager.load(task_id)
    if not task:
        raise errors.ViewError('Invalid task_id: {!r}'.format(task_id))

    # check user rights for
    if not task.user_has_permission(request.user):
        raise errors.ViewError(
            'User "{}" not allowed to cancel task "{}"'.format(request.user.login, task))

    task.stop(request=request)


@registry.xmlrpc_method()
def delete_task(request, task_id):
    """
    Delete task

    :param task_id: task identifier
    """
    task = manager.task_manager.load(task_id)
    if not task:
        raise errors.ViewError('Invalid task_id: {!r}'.format(task_id))

    # check user rights for
    if not task.user_has_permission(request.user):
        raise errors.ViewError(
            'User "{}" not allowed to delete task "{}"'.format(request.user.login, task))

    task.delete(request=request)


@registry.xmlrpc_method()
def set_task_priority(request, task_id, priority):
    """
    Set new task priority

    :param task_id:     task identifier to be operated
    :param priority:    task priority to be set. A list of two values priority class and subclass values.
    - class - *BACKGROUND*, *SERVICE*
    - subclass - *LOW*, *NORMAL*, *HIGH*
    :return:            `True` in case of successful priority switch and `False` otherwise.
    :rtype: bool
    """
    task = manager.task_manager.load(task_id)
    if not task:
        raise errors.ViewError('Invalid task_id: {!r}'.format(task_id))

    try:
        return manager.task_manager.set_priority(request, task, ctt.Priority().__setstate__(priority))
    except ValueError as ex:
        raise errors.ViewError(str(ex))


@registry.xmlrpc_method(protected=True)
def set_task_status(request, task_id, status, lock_host=""):
    """
    Set new task status

    :param task_id: task identifier
    :param status: new status
    """
    task = manager.task_manager.load(task_id)
    if not task:
        raise errors.ViewError('Invalid task_id: {!r}'.format(task_id))

    task.set_status(status, request=request, lock_host=lock_host)
    manager.task_manager.update(task)


###########################################################
# Service
###########################################################

@registry.xmlrpc_method(protected=True)
def set_task_context_value(task_id, key, value):
    """
        Задать значение в контексте задачи

        :param task_id: идентификатор задачи
        :param key: ключ в контексте
        :param value: задаваемое значение
        :return: True, если удалось установить значение; False в противном случае
    """
    key = str(key)
    task = manager.task_manager.load(task_id)
    if not task:
        return False
    task.ctx[key] = value
    if key.startswith('__'):
        return False
    manager.task_manager.update(task)
    return True


@registry.xmlrpc_method(protected=True)
def increase_task_priority(request, login, task_id):
    """
    Увеличить значение приоритета задачи с указанным id

    :param task_id: идентификатор задачи
    :param login: логин пользователя, от которого делается изменение (legacy, not used)
    :raises `errors.ViewError`: if not trusted user is trying to set SERVICE:HIGH or USER:* priority
    :return: True if task priority was successfully increased, False - otherwise
    :rtype: bool
    """
    task = manager.task.TaskManager.load(task_id)
    priority = task.priority.next
    try:
        return manager.task_manager.set_priority(request, task, priority, controller.User.get(login))
    except ValueError as ex:
        raise errors.ViewError(str(ex))


@registry.xmlrpc_method(protected=True)
def wait_time(request, task_id, timeout):
    """
    Wait for time is out.

    :param task_id: id of task to switch to wait state
    :param timeout: time in seconds to stay in waiting state
    """
    trigger_controller.TimeTrigger.create(mapping.TimeTrigger(
        source=task_id,
        time=dt.datetime.utcnow() + dt.timedelta(seconds=timeout),
        token=request.session and request.session.token
    ))


@registry.xmlrpc_method(protected=True)
def wait_tasks(request, task_id, tids, statuses, wait_all=False):
    """
    Wait for tasks switched to one of the states.

    :param task_id: id of task to switch to wait state
    :param tids: list of task id to wait for
    :param statuses: list of task statuses to wait for
    :param wait_all: if True then wait for all tasks else wait for any task
    """
    statuses = set(statuses) - set(ctt.Status.Group.NONWAITABLE)
    if not statuses:
        raise ValueError("Empty statuses list to wait for")

    targets = trigger_controller.TaskStatusTrigger.get_not_ready_targets(
        targets=tids, statuses=statuses
    )
    if not targets or (not wait_all and len(tids) != len(targets)):
        raise errors.NothingToWait

    trigger_controller.TaskStatusTrigger.create(mapping.TaskStatusTrigger(
        source=task_id,
        targets=targets,
        statuses=statuses,
        wait_all=wait_all,
        token=request.session and request.session.token,
    ))
