import time
import logging
import functools as ft
import threading
import contextlib
import collections

import werkzeug.local

from sandbox.common import context as common_context

__all__ = ("Context", "current", "set_current")


tls = threading.local()


class Span(object):
    def __init__(self, name, duration, data=None):
        self.name = name
        self.duration = duration
        self.data = data

    def __repr__(self):
        return "<{}({}s)={}>".format(self.name, self.duration, self.data)


class Spans(object):
    """
    Container for sequence of spans.
    """

    def __init__(self):
        self._spans = []
        self._total_duration = collections.defaultdict(int)  # name -> duration

    def add_span(self, span):
        self._spans.append(span)
        self._total_duration[span.name] += span.duration

    def total_duration_for(self, span_name):
        return self._total_duration.get(span_name, 0)

    def as_short_string(self, group_by_name=False):
        if group_by_name:
            spans = self._total_duration.items()
        else:
            spans = ((span.name, span.duration) for span in self._spans)

        return "|".join("{}={:.5f}".format(name, duration) for name, duration in spans)


def timer_decorator(timer_name=None, reset=False, logger=None, loglevel=logging.DEBUG):
    def decor(func):
        @ft.wraps(func)
        def wrapper(*args, **kws):
            context = _get_current()
            if reset:
                context.timer.reset()
            timer = context.timer[timer_name or func.__name__].start()
            try:
                return func(*args, **kws)
            finally:
                (logger or context.logger or logging).log(
                    loglevel, "Timer<%s>: %s", timer_name or func.__name__, timer
                )
                timer.stop()
        return wrapper
    return decor


class Context(object):
    def __init__(self, logger, user):
        self.logger = logger
        self.user = user

        self._spans = Spans()
        self._timer = None

    @property
    def spans(self):
        return self._spans

    @property
    def timer(self):
        if self._timer is None:
            self._timer = common_context.Timer(auto_start_parent=True)
        return self._timer

    def add_span(self, name, duration, data=None):
        self._spans.add_span(Span(name, duration, data))

    @contextlib.contextmanager
    def span(self, name, data=None):
        start = time.time()
        s = Span(name, None, data)
        yield s
        s.duration = time.time() - start
        self._spans.add_span(s)


def set_current(value):
    assert isinstance(value, Context) or value is None
    setattr(tls, "value", value)
    return value


def _get_current():
    ctx = getattr(tls, "value", None)
    if ctx is None:
        ctx = set_current(Context(None, None))
    return ctx


current = werkzeug.local.LocalProxy(_get_current)
