from __future__ import absolute_import

import time
import httplib
import urlparse
import functools

from sandbox import common
import sandbox.common.types.misc as ctm

import sandbox.agentr.client
import sandbox.web.response


class AgentR(sandbox.agentr.client.Session):
    """ Replaces some AgentR methods for server-side calls """

    # noinspection PyMissingConstructor
    def __init__(self, task):
        self.task = task

    def __call__(self, method, *args, **kwargs):
        raise NotImplementedError("Method '{}' is not allowed in this context".format(method))

    def resource_register(
        self, path, resource_type, description, arch, attrs,
        share=True, service=False, for_parent=False, depth=None, resource_meta=None, system_attributes=None
    ):
        from sandbox.yasandbox import controller
        import yasandbox.api.json.resource as api

        from sandbox import sdk2

        if for_parent:
            assert self.task.parent, "Task #{} does not have a parent".format(self.task.id)
            task_id = self.task.parent.id
            attrs["from_task"] = self.task.id
        else:
            task_id = self.task.id

        if not resource_meta:
            resource_meta = sdk2.Resource[resource_type].__getstate__()

        resource = controller.Resource.create(
            description, path, None, resource_type, task_id, resource_meta, self.task,
            arch=arch or ctm.OSFamily.ANY, attrs=attrs, system_attributes=system_attributes
        )
        entry = api.Resource.Entry("", "", resource)
        return entry


class RestClient(common.rest.Client):
    """
    Replaces REST client for server-side requests.

    All requests, except POST ones, are dispatched locally instead of being piped over HTTP,
    which means that REST API methods are called like regular functions.
    This may be handy when one wants to patch a certain server-side routine or behaviour
    and test it afterwards.

    Usage:

    .. code-block:: python

        import sandbox.yasandbox.controller.dispatch as serverside

        api = serverside.RestClient(None, "sandbox-user")()
        ctime = (api >> api.PLAINTEXT).service.time.current[:]  # saved you a GET
    """

    # tuple of HTTP response codes to retry requests for
    RETRYABLE_CODES = (
        httplib.SERVICE_UNAVAILABLE,
    )

    # noinspection PyPep8Naming
    class __metaclass__(type):
        def __call__(cls, task_id, author=None, jailed=True, return_error=False):
            def http_error(result):
                if return_error:
                    return result
                else:
                    raise sandbox.web.response.HTTPError(result)

            # noinspection PyUnusedLocal
            def _request(self, method, path, params=None, headers=None):
                import sandbox.web.controller
                import sandbox.web.server.request
                method_name = (method if isinstance(method, basestring) else method.__name__).upper()
                self.logger.debug(
                    "local REST request: %s %s, task_id=%s",
                    method_name, path, task_id,
                )

                request = sandbox.web.server.request.LocalSandboxRequest(
                    method_name, path, params, headers, task_id, author, jailed=jailed
                )

                spent = 0
                started = time.time()
                result = None

                # this cycle only runs for 60 seconds, but that should be enough for a successful retry
                # (the only possible reason for 503 here is serviceq.errors.QRetry, and response to the next request
                # is a redirect)
                while spent < self.DEFAULT_TIMEOUT:
                    try:
                        dispatched_method = sandbox.web.controller.dispatch(request.path, request)
                        if dispatched_method is None:
                            return http_error(sandbox.web.response.HttpErrorResponse(httplib.NOT_FOUND))
                        result = dispatched_method(request)
                        if isinstance(result, sandbox.web.response.HttpRedirect):
                            parsed_url = urlparse.urlparse(result.redirect_url)
                            params = dict(urlparse.parse_qsl(parsed_url.query)) if parsed_url.query else {}
                            return self._request(method, parsed_url.path, {"params": params}, headers)
                    finally:
                        request.restore()
                    result.status_code = result.code
                    if isinstance(result, sandbox.web.response.HttpErrorResponse):
                        if result.status_code not in self.RETRYABLE_CODES:
                            return http_error(result)

                        time.sleep(max(0, min(self.DEFAULT_INTERVAL, self.DEFAULT_TIMEOUT - spent)))
                        spent = time.time() - started
                    else:
                        break
                else:
                    if isinstance(result, sandbox.web.response.HttpErrorResponse):
                        return http_error(result)

                return result

            pcls = cls.__base__
            return type(pcls)(
                pcls.__name__, (pcls,), dict(_request=_request, HTTPError=sandbox.web.response.HTTPError)
            )


def local(fn):
    """ Decorator for task wrapper methods to force the use of the local client """
    @functools.wraps(fn)
    def wrapper(self, *args, **kwargs):
        with common.rest.DispatchedClient as dispatch:
            if self._rest is None:
                self._rest = RestClient(self._model.id, self._model.author)
            dispatch(self._rest)
            return fn(self, *args, **kwargs)

    return wrapper


ALLOWED_SERVER_SIDE_REST_REQUESTS = {
    "get": [],  # allow all get requests
    "post": ["/task/current/hints"],
}


def invalid_rest_call_client(rest_client, message):
    class InvalidRESTCallClient(rest_client):
        def __init__(self, *args, **kwargs):
            super(InvalidRESTCallClient, self).__init__(*args, **kwargs)
            self.message = message

        def _request(self, method, path, params=None, headers=None):
            allowed_paths = ALLOWED_SERVER_SIDE_REST_REQUESTS.get(method.__name__)
            if allowed_paths is None or allowed_paths and path not in allowed_paths:
                raise common.errors.InvalidRESTCall(self.message)
            return super(InvalidRESTCallClient, self)._request(method, path, params=params, headers=headers)

    return InvalidRESTCallClient
