import os
import cPickle
import urlparse

import six
import copy
import calendar
import datetime as dt
import operator as op

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
import sandbox.common.types.scheduler as cts
import sandbox.common.types.notification as ctn

from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.controller import user as user_controller


class Notification(object):

    Model = mapping.Notification

    Type = ctn.Type
    Transport = ctn.Transport
    Charset = ctn.Charset

    class NotificationType(common.enum.Enum):
        TASK = None
        SCHEDULER = None

    TRIGGER_STATUSES = {
        NotificationType.TASK: set(ctt.Status),
        NotificationType.SCHEDULER: set(cts.Status)
    }

    NOTIFICATION_MODELS = {
        NotificationType.TASK: mapping.Task.Notification,
        NotificationType.SCHEDULER: mapping.Scheduler.SchedulerNotifications.Notification
    }

    @classmethod
    def list_query(cls, author=None, recipient=None, transport=None, sent=None, task_id=None, created=None):
        """
        Build query with specified parameters for notifications

        :param author: notification author
        :param recipient: notification recipient
        :param transport: notification transport
        :param sent: notification was sent
        :param task_id: task_id of notification
        :param created: filter notifications by created time after created[0] and before created[1]
        :return: dict with database query
        """

        query = {}
        if author is not None:
            query["author"] = author
        if transport is not None:
            query["transport"] = transport
        if sent is not None:
            query["sent"] = sent
        if task_id is not None:
            query["task_id"] = task_id
        if recipient is not None:
            query["send_to"] = recipient

        if created:
            query["date__gte"] = created[0]
            query["date__lte"] = created[1]
        return query

    @classmethod
    def initialize(cls):
        cls.Model.ensure_indexes()

    @classmethod
    def recipients(cls, *args, **kwargs):
        """
        Returns a list of notification recipients given.
        Arguments can be lists or strings of user names and/or group names
        with or without domain, which will be cut-off.
        Also, this function will filter out non-real users (robots).

        :param resolve_groups:  Perform resolving of group name to its email (or list of members).
        :param exclude_robots:  Exclude known robots from the result.

        :return: A list of recipients in order of arguments provided.
        """
        ret = {}
        resolve_groups, exclude_robots = map(kwargs.get, ("resolve_groups", "exclude_robots"))
        for addr in common.utils.chain(*args):
            addr = addr.strip() if addr else addr
            if not addr:
                continue
            try:
                gr = user_controller.Group.get(addr)
                ret[gr.email if resolve_groups and gr.email else gr.name] = len(ret)
            except ValueError:
                u = addr.split("@", 1)[0].lower()
                uobj = user_controller.User.valid(u) if exclude_robots else None
                if uobj:
                    if not uobj.robot:
                        ret[uobj.login] = len(ret)
                else:
                    ret[u] = len(ret)
        return filter(None, map(op.itemgetter(0), sorted(ret.iteritems(), key=op.itemgetter(1))))

    @classmethod
    def telegram_recipients(cls, *recipients):
        """
        Returns a list of telegram logins.
        Arguments can be lists or strings if groups, usernames or telegram logins
        (values will be resolved in this priority)
        :param recipients: lists or strings if groups, usernames or telegram logins
        :return: list of telegram logins
        """

        ret = set()
        for name in common.utils.chain(*recipients):
            name = name.strip() if name else name
            if not name:
                continue

            try:
                group = user_controller.Group.get(name)
                if group.telegram_chat_id:
                    ret.add(group.telegram_chat_id)
                else:
                    users = mapping.User.objects(login__in=group.users, robot=False)
                    for user in users:
                        if user.telegram_login:
                            ret.add(user.telegram_login)
            except ValueError:
                user = user_controller.User.get(name)
                if user:
                    if user.telegram_login:
                        ret.add(user.telegram_login)
                elif user_controller.User.check_telegram_username(name):
                    ret.add(name)
        return list(ret)

    @classmethod
    def q_recipients(cls, *recipients):
        """
        Returns a list of q chat ids.
        lists or strings of user names and/or group names
        (values will be resolved in this priority)
        :param recipients: lists or strings if groups, usernames or telegram logins
        :return: list of q chat ids
        """

        ret = set()
        for name in filter(None, map(lambda s: s.strip(), common.utils.chain(*recipients))):
            name = name.strip() if name else name
            if not name:
                continue

            try:
                group = user_controller.Group.get(name)
                if group.messenger_chat_id is not None:
                    ret.add(group.messenger_chat_id)
            except ValueError:
                user = user_controller.User.get(name)
                if user and user is not None and user.messenger_chat_id is not None:
                    ret.add(user.messenger_chat_id)
        return list(ret)

    @classmethod
    def perform_juggler_query(cls, query_string):
        query = {k: v[0] for k, v in six.iteritems(urlparse.parse_qs(query_string))}
        if len(query) != 2:
            raise ValueError("Juggler query must be like host=namespace.host&service=service")
        host, service = query["host"], query["service"]

        if not host or not service:
            raise ValueError(
                "Juggler query must be like host=namespace.host&service=service, service or host not found"
            )

        return host, service

    @classmethod
    @common.utils.ttl_cache(300)
    def juggler_recipient(cls, name, juggler_key):
        name = name.strip() if name else name
        if not name:
            return None
        try:
            group = user_controller.Group.get(name)
            if not group.juggler_settings:
                return None
            if not juggler_key or juggler_key not in group.juggler_settings.checks:
                host = group.juggler_settings.default_host
                service = group.juggler_settings.default_service
            else:
                host = group.juggler_settings.checks[juggler_key].host or group.juggler_settings.default_host
                service = (
                    group.juggler_settings.checks[juggler_key].service or group.juggler_settings.default_service
                )

            if not host or not service:
                return None
            return "host={}&service={}".format(host, service)
        except ValueError:
            return name

    @classmethod
    def juggler_expanded_recipients(cls, recipients, juggler_key=None):
        ret = set()
        for name in recipients:
            item = cls.juggler_recipient(name, juggler_key)
            if item:
                ret.add(item)
        return list(ret)

    @classmethod
    def juggler_recipients(cls, *recipients):
        ret = set()
        for name in filter(None, map(lambda s: s.strip(), common.utils.chain(*recipients))):
            name = name.strip() if name else name
            if not name:
                continue

            try:
                group = user_controller.Group.get(name)
                if group.juggler_settings:
                    ret.add(name)
            except ValueError:
                cls.perform_juggler_query(name)
                ret.add(name)
        return list(ret)

    @classmethod
    def notification(
        cls, transport, statuses, recipients, notification_type=NotificationType.TASK, check_status=None,
        juggler_tags=None
    ):
        """
        Creates an instance of task notification trigger object for given transport, status(es) and recipients.
        In case of some required parameters provided are empty, returns nothing.

        :return: :class:`yasandbox.database.mapping.Task.Notification` class instance or `None`.
        """
        statuses = list(common.utils.chain(statuses))
        for status in statuses:
            if status not in cls.TRIGGER_STATUSES[notification_type]:
                raise ValueError("Unknown task status {!r}".format(status))
        recipients = cls.recipients(*recipients)
        if transport not in ctn.Transport:
            raise ValueError("Unknown transport {}".format(transport))
        return (
            cls.NOTIFICATION_MODELS[notification_type](
                transport=transport, statuses=statuses, recipients=recipients, check_status=check_status,
                juggler_tags=juggler_tags or []
            )
            if transport and statuses and recipients else
            None
        )

    @staticmethod
    def _generate_query(inbox, user=None, transport=None, query=None):
        if not query:
            request = {}
        else:
            request = copy.deepcopy(query)
        user_handler = {True: "send_to", False: "author"}
        if user:
            request[user_handler[inbox]] = user
        if "date" in request:
            min_date = request["date"]
            max_date = min_date + dt.timedelta(days=1) - dt.timedelta(seconds=1)
            request["date__gt"], request["date__lt"] = min_date, max_date
            del request["date"]
        if transport:
            request["transport"] = transport
        return request

    @staticmethod
    def expand_groups_emails(addresses):
        """
        :param addresses: list of user logins and/or group names
        :return: set of user logins, and, if groups were present in addresses, e-mails of corresponding groups
        """
        result = set()
        for address in common.utils.chain(addresses):
            if user_controller.Group.exists(address):
                gr = user_controller.Group.get(address)
                if gr.email:
                    result.update(gr.email.split())
                else:
                    result.update(gr.users)
            else:
                result.add(address)
        return result

    @staticmethod
    def _prepare_email_addressees(addrs):
        if not addrs:
            return
        if isinstance(addrs, basestring):
            it = addrs.split(',')
        else:
            it = iter(addrs)
        result = set()
        domain = '@yandex-team.ru'
        for a in it:
            a = a.replace(domain, '')
            a = a.strip()
            if a:
                result.add(a)
        return ', '.join(result)

    @staticmethod
    def release_info(task, user, comments):
        """
        Construct plain text release form

        :param task: task to release object
        :param user: releaser login
        :param comments: comments to release
        :return: info about release (author, comments, resources etc)
        :rtype: str
        """
        if not comments:
            comments = u" "
        info_string = u"Author: {}\nSandbox task: {}\n".format(user, common.utils.get_task_link(task.id))
        resource_string = "Resources:\n"
        for resource in task.list_resources():
            if resource.type.releasable:
                b = u" * {} - id:{} [{}]".format(resource.type, resource.id, (resource.arch or "any"))
                binary_info = u"{}:\n     File name:         {}, {}KB (md5: {})".format(
                    b,
                    os.path.basename(resource.file_name),
                    resource.size,
                    resource.file_md5
                )
                padding = u"   "
                sandbox_info = u"{}  Resource page:     {}".format(padding, resource.sandbox_view_url())
                http_info = u"{}  HTTP download URL: {}".format(padding, resource.proxy_url)
                skynet_info = u"{}  Skynet ID:         {}".format(padding, resource.skynet_id or "N/A")

                resource_string += u"{}\n\n".format(
                    "\n".join((binary_info, sandbox_info, http_info, skynet_info))
                )
        sign_string = u"--\nsandbox \n https://sandbox.yandex-team.ru/ \n"
        return u"\n".join([info_string, comments, resource_string, sign_string])

    @classmethod
    def save(
        cls, transport, send_to, send_cc, subject, body, content_type="text/plain", charset="utf-8",
        extra_headers=None, author='sandbox', task_id=None, view=ctn.View.DEFAULT, task_model=None,
        urgent=False, check_status=None, juggler_tags=None
    ):
            """
            Save notification document to DB

            :param transport: string with appropriate transport param (`telegram` or `email`)
            :param send_to: list of recipient logins
            :param send_cc: list of cc-recipient logins
            :param subject: notification subject
            :param body: notification body
            :param content_type: type of notification content (text, html)
            :param charset: notification encoding
            :param extra_headers: user defined headers of notification
            :param author: notification author
            :param task_id: task id to generate execution and release reports
            :param view: view type
            :param task_model: db model of the task
            :param urgent: send from sandbox-urgent@
            :param check_status: check status
            :param juggler_tags: tags for Juggler
            """
            send_list = set()
            if transport == ctn.Transport.EMAIL:
                if send_to:
                    prepared_addresses_to = cls._prepare_email_addressees(send_to)
                    send_list = send_list.union(prepared_addresses_to.split(", "))
                if send_cc:
                    prepared_addresses_cc = cls._prepare_email_addressees(send_cc)
                    send_list = send_list.union(prepared_addresses_cc.split(", "))
            else:
                send_list = set(send_to)
            notification_type = cls.Type.HTML
            if not subject:
                subject = "Default subject"
            if content_type == "text/plain":
                notification_type = cls.Type.TEXT

            doc = cls.Model(
                author=author,
                date=dt.datetime.utcnow(),
                send_to=list(send_list),
                subject=subject,
                body=body,
                transport=transport,
                type=notification_type,
                charset=charset,
                headers=extra_headers if extra_headers is not None else [],
                view=view,
                execution=task_model.execution if task_model else None,
                urgent=urgent,
                check_status=check_status,
                juggler_tags=juggler_tags or []
            )
            if task_id:
                doc.task_id = task_id
            return doc.save()

    @classmethod
    def report_service_error(cls, comment, thread_name, lock_name=None, additional_recipients=None):
        """
        Generate Sandbox service thread error report

        :param comment: error description
        :param thread_name: name of service thread
        :param lock_name: additional info about the current lock
        :param additional_recipients: list of prepared (i.e. without domain) list of logins
        """
        settings = common.config.Registry()
        try:
            recipients = cls.expand_groups_emails(settings.common.service_group)
        except AttributeError:
            recipients = ["sandbox-errors"]
        if not recipients:
            recipients = ["sandbox-errors"]
        if additional_recipients:
            recipients = set(common.utils.chain(recipients, additional_recipients))
        body = (
            "{node} detected following problems during `{thread_name}` service thread execution:\n\n{comment}"
            "\n\nWBR,{lock_sign}\n{node}"
        ).format(
            node=settings.this.fqdn,
            thread_name=thread_name,
            comment=comment,
            lock_sign="\nformer `{}` lock keeper".format(lock_name) if lock_name is not None else ""
        )
        cls.save(
            transport=cls.Transport.EMAIL,
            send_to=recipients,
            send_cc=None,
            subject="[WARNING] Error occurred in `{}` service thread".format(thread_name),
            body=body
        )

    @classmethod
    def count_sent(cls, user=None, transport=None, query=None):
        """
        Get amount of sent notification according to given params. All params are optional.

        :param user: user's login
        :param transport: string with appropriate transport param (`telegram` or `email`)
        :param query: dictionary with query params
        :return: amount of documents that satisfies given params
        :rtype: int
        """
        request = cls._generate_query(inbox=False, user=user, transport=transport, query=query)
        return cls.Model.objects(**request).count()

    @classmethod
    def count_inbox(cls, user=None, transport=None, query=None):
        """
        Get amount of inbox notification according to given params. All params are optional.

        :param user: user's login
        :param transport: string with appropriate transport param (`telegram` or `email`)
        :param query: dictionary with query params
        :return: amount of documents that satisfies given params
        :rtype: int
        """
        request = cls._generate_query(inbox=True, user=user, transport=transport, query=query)
        return cls.Model.objects(**request).count()

    @classmethod
    def list_sent(cls, user=None, limit=20, offset=0, query=None):
        """
        Get notifications generated by `user`. All params are optional

        :param limit: amount of documents to limit result
        :param offset: amount of first documents to skip
        :param user: user login
        :param query: dictionary with query params
        :return: list of notifications generated by `user`
        :rtype: list
        """
        request = cls._generate_query(inbox=False, user=user, query=query)
        return cls.Model.objects(**request).skip(offset).limit(limit).order_by("-date")

    @classmethod
    def list_inbox(cls, user=None, limit=20, offset=0, query=None):
        """
        Get inbox notifications for `user`. All params are optional

        :param limit: amount of documents to limit result
        :param offset: amount of first documents to skip
        :param user: user login
        :param query: dictionary with query params
        :return: list of inbox notifications for `user`
        :rtype: list
        """
        request = cls._generate_query(inbox=True, user=user, query=query)
        return cls.Model.objects(**request).skip(offset).limit(limit).order_by("-date")

    @classmethod
    def load(cls, notification_id):
        """
        Get notification document

        :param notification_id: id of notification
        :return: notification document with given id
        :rtype: Notification
        """
        return cls.Model.objects.with_id(notification_id)

    @staticmethod
    def get_task_log_size(task_id):
        """
        Get logs size of task with specified task_id

        :param task_id: task id
        :return: task's logs size
        :rtype: int
        """
        return int(sum(
            mapping.Resource.objects(task_id=task_id, type=str(ctr.TASK_LOG_RESOURCE_TYPE)).scalar("size")
        ))

    @classmethod
    def task_execution_info(cls, task, notification):
        """
        Get task execution info.

        :param task: task object
        :param notification: notification object
        :return: dictionary with task execution results
        :rtype: dict
        """
        from sandbox.yasandbox import controller
        execution = notification.execution or task.execution

        start = execution.last_execution_start
        finish = execution.last_execution_finish
        timestamp_start = calendar.timegm(start.timetuple()) if start else 0
        timestamp_finish = calendar.timegm(finish.timetuple()) if finish else 0

        return {
            "type": task.type,
            "id": task.id,
            "status": execution.status,
            "description": task.description,
            "info": execution.description,
            "result": "",
            "url": common.utils.get_task_link(task.id),
            "logs_url": controller.Task.remote_http(task),
            "logs_size": cls.get_task_log_size(task.id),
            "delta": timestamp_finish - timestamp_start,
            "created": calendar.timegm(task.time.created.timetuple()),
            "host": execution.host,
            "started": timestamp_start,
            "finished": timestamp_finish
        }

    @staticmethod
    def task_release_info(task, notification):
        """
        Get task release info

        :param task: task object
        :param notification: notification object
        :return: dictionary with task release details
        """
        from sandbox.yasandbox import controller

        return {
            "author": task.author,
            "release_author": task.execution.release_params.get("releaser", ""),
            "id": task.id,
            "view": common.utils.get_task_link(task.id),
            "type": task.type,
            "changelog": (cPickle.loads(task.context) if task.context else {}).get("release_changelog", ""),
            "description": task.description,
            "resources": [
                {
                    "id": resource.id,
                    "filename": os.path.basename(resource.path),
                    "size": common.utils.size2str(resource.size << 10) if resource.size is not None else "",
                    "md5": resource.md5,
                    "view": urlparse.urljoin(common.utils.server_url(), 'resource/{}'.format(resource.id)),
                    "download": controller.Resource.proxy_url(resource),
                    "skyid": resource.skynet_id,
                    "type": resource.type
                }
                for resource in controller.Resource.list_task_resources(task.id)
            ]
        }
