import logging
import collections
import datetime as dt
import itertools as it

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu
import sandbox.common.types.scheduler as cts
import sandbox.common.types.statistics as ctst
import sandbox.common.types.notification as ctn

from sandbox.yasandbox.database import mapping

from sandbox.yasandbox.controller import task as task_controller
from sandbox.yasandbox.controller import user as user_controller
from sandbox.yasandbox.controller import notification as notification_controller
from sandbox.yasandbox.controller import trigger as trigger_controller

MAX_SCHEDULER_PRIORITY = int(ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH))


class Scheduler(object):
    logger = logging.getLogger(__name__)
    Model = mapping.Scheduler

    @common.utils.singleton_classproperty
    def __oauth_token(cls):
        config = common.config.Registry()
        return (
            common.utils.read_settings_value_from_file(config.server.auth.oauth.token)
            if config.server.auth.enabled else
            None
        )

    @classmethod
    def initialize(cls):
        cls.Model.ensure_indexes()
        mapping.ParametersMeta.ensure_indexes()

    @common.utils.classproperty
    def _server(cls):
        return common.rest.ThreadLocalCachableClient(auth=cls.__oauth_token, component=ctm.Component.SERVICE)

    @classmethod
    def _repr(cls, scheduler):
        return "<Scheduler #{}: {}, {}, next_run={}>".format(
            scheduler.id,
            scheduler.type,
            scheduler.status,
            scheduler.time.next_run if scheduler.time.next_run else None,
        )

    @classmethod
    def create(cls, task_type, owner, author, data=None):
        """
            :return: new scheduler object
            :rtype: yasandbox.database.mapping.Scheduler
        """

        scheduler = mapping.Scheduler(type=task_type, owner=owner, author=author)
        if data:
            task_controller.Task.update_tasks_resource(data.get("task", {}), scheduler)
        template = task_controller.TaskWrapper(scheduler)
        template.init_model()
        template.update_context({}, remove_fields=["__GSID"], save=False)
        scheduler.scheduler_notifications = mapping.Scheduler.SchedulerNotifications()
        cls.save(scheduler)
        return scheduler

    @classmethod
    def copy(cls, scheduler_id, author):
        """
            Copy existing scheduler

            :param scheduler_id: scheduler id to copy
            :param author: author of the new scheduler
            :return: new scheduler object, None if it was not copied
            :rtype: yasandbox.database.mapping.Scheduler
        """
        old_scheduler = cls.Model.objects.with_id(scheduler_id)
        if not old_scheduler:
            return None
        cls.logger.info("[Copy] Old scheduler: {0}".format(cls._repr(old_scheduler)))
        new_scheduler = old_scheduler.clone()
        new_scheduler.author = author
        new_scheduler.time.created = dt.datetime.utcnow()
        if new_scheduler.owner not in user_controller.Group.get_user_groups(author):
            new_scheduler.owner = author
        new_scheduler.status = cts.Status.STOPPED
        new_scheduler.cur_tasks_ids = []
        new_scheduler.scheduler_notifications = old_scheduler.scheduler_notifications
        cls.save(new_scheduler)
        cls.logger.info("[Copy] New scheduler: {0}".format(cls._repr(new_scheduler)))
        return new_scheduler

    @classmethod
    def save(cls, scheduler, previous_status=None):
        if scheduler.priority > MAX_SCHEDULER_PRIORITY:
            scheduler.priority = MAX_SCHEDULER_PRIORITY
        try:
            scheduler.save(save_condition={"status": previous_status or scheduler.status})
        except mapping.SaveConditionError as e:
            cls.logger.warning("Conflict during save: %r", e)
        else:
            cls.logger.info("%s saved", cls._repr(scheduler))

    @classmethod
    def load(cls, scheduler_id):
        return cls.Model.objects.with_id(scheduler_id)

    @classmethod
    def fast_load_list(cls, ids):
        objects = cls.Model.objects.in_bulk(map(mapping.ObjectId, ids))
        return map(objects.get, ids)

    @classmethod
    def list_query(
        cls, owner="", status="", task_type="", limit=0, offset=0, last_id="", author="", ready_at=None, scalar=None
    ):
        """
            Compiles query object for schedulers collection.

            :rtype: `mongoengine.QuerySet`
        """
        q_obj = mapping.Q()
        args = {}
        if owner:
            args["owner"] = owner
        if status:
            if isinstance(status, basestring):
                args["status"] = status
            else:
                args["status__in"] = list(status)
        else:
            args["status__ne"] = cls.Model.Status.DELETED
        if task_type:
            args["type"] = task_type
        if last_id:
            args["id__gt"] = last_id
        if author:
            args["author"] = author
        if ready_at:
            q_obj |= mapping.Q(
                time__next_run__lt=ready_at
            ) | mapping.Q(
                time__next_run__exists=False
            )

        query = cls.Model.objects(q_obj, **args)
        if scalar:
            query = query.fast_scalar(*scalar)
        query = query.skip(offset)
        return query.limit(limit) if limit else query

    @classmethod
    def restart(cls, scheduler):
        if cls._enabled(scheduler):
            return False
        previous_status = scheduler.status
        cls.set_status(scheduler, cls.Model.Status.WAITING)
        cls.check(scheduler, previous_status, manual=True)
        return True

    @classmethod
    def stop(cls, scheduler):
        previous_status = scheduler.status
        cls.set_status(scheduler, cls.Model.Status.STOPPED)
        cls.save(scheduler, previous_status)
        return True

    @classmethod
    def delete(cls, scheduler):
        try:
            cls.status_notification(scheduler, cls.Model.Status.DELETED)
        except:
            cls.logger.exception("Can't update notifications for scheduler #%s", scheduler.id)
        cls.Model.objects(id=scheduler.id).update_one(set__status=cls.Model.Status.DELETED)
        return True

    @classmethod
    def check(cls, scheduler, previous_status=None, manual=False):
        """ Check scheduler and schedule task if it's required. """
        if manual:
            cls.logger.info("Scheduler %d is being re-scheduled manually", scheduler.id)

        previous_status = previous_status or scheduler.status
        now = dt.datetime.utcnow()
        scheduler.time.checked = now

        last_task = cls._get_last_task(scheduler)
        next_run = cls.get_next_task_creation_time(scheduler, now, manual, last_task=last_task)
        if not scheduler.time.next_run or (next_run and next_run < scheduler.time.next_run):
            scheduler.time.next_run = next_run
            cls.logger.debug("Scheduler #%d next run: %s", scheduler.id, next_run)

        cls._update_status(scheduler, last_task, manual)
        if cls._enabled(scheduler) and cls.delayed(scheduler, now):
            if not manual:
                delay = max(int((now - scheduler.time.next_run).total_seconds()), 0) if scheduler.time.next_run else 0
                common.statistics.Signaler().push(dict(
                    type=ctst.SignalType.TEMPLATE_RUN_DELAY,
                    date=now,
                    timestamp=now,
                    template_id=scheduler.id,
                    task_type=scheduler.type,
                    delay=delay,
                    owner=scheduler.owner,
                ))
            cls.logger.info(
                "Creating task by scheduler #%s: type=%r, owner=%r", scheduler.id, scheduler.type, scheduler.owner
            )
            task_model = (cls.create_new_task if manual else cls._create_new_task_remotely)(scheduler)
            if task_model:
                scheduler.time.next_run = cls.get_next_task_creation_time(scheduler, last_task=task_model)
                scheduler.time.last_run = task_model.time.created
                cls.Model.objects(id=scheduler.id).update(add_to_set__cur_tasks_ids=task_model.id)
                cls._update_status(scheduler, task_model)
        cls.save(scheduler, previous_status)

    @classmethod
    def check_owner_coherence(cls, schedulers):
        """
        Check if the scheduler author fits the owner group and the owner group is ok for each scheduler

        :type schedulers: a list of schedulers to check

        :returns: scheduler id to tuple: True is author fits the owner, False otherwise, and the error message
        :rtype: dict(scheduler_id: tuple(bool, string))
        """
        coherence = dict()

        group_names, user_logins = set(), set()
        for scheduler in schedulers:
            group_names.add(scheduler.owner)
            user_logins.add(scheduler.author)

        users = {_.login: _ for _ in mapping.User.objects(login__in=user_logins)}

        groups = set()
        groups_cache = collections.defaultdict(list)
        for group in mapping.Group.objects(name__in=group_names).fast_scalar("name"):
            groups.add(group)

        groups_for_check_permissions = list(mapping.Group.objects(users__in=user_logins).fast_scalar("name", "users"))
        for group_name, user_names in groups_for_check_permissions:
            for name in user_names:
                groups_cache[name].append(group_name)

        for scheduler in schedulers:
            user = scheduler.author
            owner = scheduler.owner
            message = ""
            if (
                owner and
                common.config.Registry().server.auth.enabled and
                owner != ctu.OTHERS_GROUP and owner not in groups and
                owner != ctu.ANONYMOUS_LOGIN and owner not in users
            ):
                message = "Owner group {} ({}) validation failed".format(owner, common.urls.get_group_link(owner))
            elif owner and not user_controller.user_has_permission(users.get(user), [owner], groups_cache=groups_cache):
                message = "The author {} ({}) is not in the group {} ({})".format(
                    user, common.urls.get_user_link(user),
                    owner, common.urls.get_group_link(owner)
                )

            coherence[scheduler.id] = (not message, message,)

        return coherence

    @classmethod
    def delayed(cls, scheduler, now):
        """
        :type scheduler: `mapping.Scheduler`
        :type now: `dt.datetime.datetime`
        :rtype: bool
        """
        if scheduler.time.next_run and scheduler.time.next_run <= now:
            if scheduler.status == cts.Status.WAITING:
                return True
            elif scheduler.status == cts.Status.WATCHING:
                return not scheduler.plan.sequential_run

    @classmethod
    def _enabled(cls, scheduler):
        """
            Check if scheduler is in active state, i.e. it should produce tasks.
            :rtype: bool
        """
        return scheduler.status in [cls.Model.Status.WAITING, cls.Model.Status.WATCHING]

    @classmethod
    def _create_new_task_remotely(cls, scheduler):
        response = cls._server.task({"scheduler_id": scheduler.id, "regular_schedule": True})
        task_id = response["id"]
        cls.logger.info("Task #%s was created for %s", task_id, cls._repr(scheduler))

        start_response = cls._server.batch.tasks.start.update(
            {"id": [task_id], "comment": "Scheduled by scheduler #{}".format(scheduler.id)}
        )
        op_status = start_response[0]["status"]
        if op_status != ctm.BatchResultStatus.SUCCESS:
            cls.logger.warning("Problems while starting task #%s: %s", task_id, start_response[0]["message"])
            if op_status == ctm.BatchResultStatus.ERROR:
                raise Exception("Failed to start task #{} by scheduler #{}".format(task_id, scheduler.id))
        return task_controller.Task.get(task_id)

    @classmethod
    def create_new_task(cls, scheduler, enqueue_task=True, manual=False):
        coherence = cls.check_owner_coherence([scheduler])
        author_owner_match, error_desc = coherence[scheduler.id]
        if not author_owner_match:
            raise common.errors.TaskError(error_desc)

        task_model = scheduler.cast_to(mapping.Task)
        task_model.scheduler = -scheduler.id if manual else scheduler.id
        task_model.fail_on_any_error = bool(scheduler.plan.sequential_run) or task_model.fail_on_any_error

        task = task_controller.TaskWrapper(task_model).create()
        if task is None:
            return None
        if enqueue_task:
            task.on_create().on_save()
        task.save()
        cls.logger.info("[run_task] %s was created for %s", task, cls._repr(scheduler))

        mapping.Task.objects.filter(id=task.id).update(set__notifications=scheduler.notifications)
        if enqueue_task:
            task_controller.TaskQueue.enqueue_task(task, event="Scheduled by scheduler #{}".format(scheduler.id))
        return task.model

    @classmethod
    def resolve_notification_recipients(cls, scheduler, force=False):
        if not scheduler.scheduler_notifications:
            return
        now = dt.datetime.utcnow()
        delta = common.config.Registry().server.scheduler.notification_resolving_delta

        if force or now - dt.timedelta(minutes=delta) > scheduler.scheduler_notifications.updated:
            for notification in scheduler.scheduler_notifications.notifications:
                notification.resolved_recipients = trigger_controller.TaskStatusNotifierTrigger.notification_resolver[
                    notification.transport
                ](*notification.recipients)
                if notification.transport == ctn.Transport.JUGGLER:
                    for recipient in notification.recipients or ():
                        if mapping.Group.objects(name=recipient).count() and recipient != scheduler.owner:
                            raise ValueError("Notification owner must be equal to task owner")
                    notification.resolved_recipients = notification_controller.Notification.juggler_expanded_recipients(
                        notification.resolved_recipients, juggler_key=ctn.JugglerCheck.SCHEDULER_STATUS_CHANGED
                    )
            scheduler.scheduler_notifications.updated = now

    @classmethod
    def set_notifications(cls, scheduler, data):
        notifications = filter(None, (
            notification_controller.Notification.notification(
                item.get("transport", ""),
                ctt.Status.Group.expand(item.get("statuses", ())),
                item.get("recipients", ()),
                notification_type=notification_controller.Notification.NotificationType.SCHEDULER,
                check_status=item.get("check_status"),
                juggler_tags=item.get("juggler_tags", [])
            )
            for item in data or ()
        ))
        scheduler_notifications = mapping.Scheduler.SchedulerNotifications(notifications=notifications)
        scheduler.scheduler_notifications = scheduler_notifications
        cls.resolve_notification_recipients(scheduler, force=True)

    @classmethod
    def status_notification(cls, scheduler, new_status):
        if scheduler.status == new_status or not scheduler.scheduler_notifications:
            return
        notifications = [n for n in scheduler.scheduler_notifications.notifications if new_status in n.statuses]
        if not notifications:
            return
        cls.resolve_notification_recipients(scheduler)
        for notification in notifications:
            if notification.transport == ctn.Transport.JUGGLER:
                body = u"\nScheduler {type} #{id} owned by {owner} is {st}.\n{desc}\n{url}\n".format(
                    type=scheduler.type,
                    id=scheduler.id,
                    st=new_status,
                    desc=scheduler.description,
                    url=common.utils.get_scheduler_link(scheduler.id),
                    owner=scheduler.owner
                )
                notification_controller.Notification.save(
                    transport=notification.transport,
                    send_to=notification.resolved_recipients,
                    send_cc=[],
                    subject=None,
                    body=body,
                    author=str(scheduler.owner),
                    content_type="text/html",
                    view=ctn.View.EXECUTION_REPORT,
                    check_status=notification.check_status,
                    juggler_tags=notification.juggler_tags
                )

            elif notification.transport in (ctn.Transport.TELEGRAM, ctn.Transport.Q):
                link = common.utils.get_scheduler_link(scheduler.id)
                if notification.transport == ctn.Transport.TELEGRAM:
                    link = "#<a href='{href_id}'>{id}</a>".format(href_id=link, id=scheduler.id)

                body = u"[Sandbox] Scheduler {type} {link} is {st}".format(
                    type=scheduler.type, link=link, st=new_status
                )
                notification_controller.Notification.save(
                    transport=notification.transport,
                    send_to=notification.resolved_recipients,
                    send_cc=[],
                    subject=None,
                    body=body,
                    author=str(scheduler.owner),
                    content_type="text/html",
                    view=ctn.View.EXECUTION_REPORT,
                )
            else:
                all_recipients = notification_controller.Notification.expand_groups_emails(notification.recipients)
                send_to = sorted(set(_.split("@")[0] for _ in all_recipients))

                subj = u"[Sandbox] Scheduler {type} #{id} is {st}".format(
                    type=scheduler.type, id=scheduler.id, st=new_status
                )
                body = u"\nScheduler {type} #{id} owned by {owner} is {st}.\n{desc}\n{url}\n".format(
                    type=scheduler.type,
                    id=scheduler.id,
                    st=new_status,
                    desc=scheduler.description,
                    url=common.utils.get_scheduler_link(scheduler.id),
                    owner=scheduler.owner
                )

                notification_controller.Notification.save(
                    transport=notification.transport,
                    send_to=send_to,
                    send_cc=[],
                    subject=subj,
                    body=body,
                    author=str(scheduler.owner),
                    content_type="text/html",
                    view=ctn.View.EXECUTION_REPORT,
                )

    @classmethod
    def set_status(cls, scheduler, status, message=None):
        if scheduler.status != status:
            if not message:
                message = "Switching #{} to {} state.".format(scheduler.id, status)
            cls.logger.info(message)
            try:
                cls.status_notification(scheduler, status)
            except:
                cls.logger.exception("Can't update notifications for scheduler #%s", scheduler.id)
            scheduler.status = status

    @classmethod
    def _update_status(cls, scheduler, last_task, manual=False):
        cls.logger.info("Update status for %s", cls._repr(scheduler))
        if not cls._enabled(scheduler):
            return

        task_repr = "#{} ({})".format(last_task.id, last_task.execution.status) if last_task else "None"
        cls.logger.info("Scheduler #%s last task: %s", scheduler.id, task_repr)
        cls.set_status(scheduler, cls.Model.Status.WAITING)
        if last_task:
            if (
                scheduler.plan.retry == cts.Retry.FAILURE and
                last_task.execution.status in ctt.Status.Group.SCHEDULER_FAILURE and
                not manual
            ):
                cls.set_status(scheduler, cls.Model.Status.FAILURE)
            elif last_task.execution.status in common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK):
                if scheduler.plan.repetition == cts.Repetition.NO and not scheduler.time.next_run:
                    cls.set_status(scheduler, cls.Model.Status.STOPPED)
            elif last_task.execution.status in ctt.Status.Group.DRAFT:
                cls.logger.warning("%s last task %s status is draft, retrying", cls._repr(scheduler), task_repr)
            else:
                cls.set_status(
                    scheduler, cls.Model.Status.WATCHING,
                    "{} waits for task {} to be completed.".format(cls._repr(scheduler), task_repr)
                )

    @classmethod
    def get_notification(cls, scheduler):
        """
            Get scheduler status notification params

            :return: tuple (address, subject, text)
            :rtype: tuple
        """
        address = [scheduler.owner]
        subj = u"[Sandbox] Scheduler {task_type} #{id} is {status}".format(
            task_type=scheduler.type, id=scheduler.id, status=scheduler.status
        )
        last_task = cls._get_last_task(scheduler)
        body = u"\nScheduler {task_type} #{id} ({surl}) owned by {owner} is {status}.\nLast created task: {url}".format(
            task_type=scheduler.type,
            id=scheduler.id,
            owner=scheduler.owner,
            status=scheduler.status,
            url=(
                common.utils.get_task_link(last_task.id)
                if last_task else
                "Scheduler didn't run any tasks"
            ),
            surl=common.utils.get_scheduler_link(scheduler.id),
        )
        return address, subj, body

    @classmethod
    def _get_retry_time_if_applicable(cls, scheduler, last_task):
        retriable_statuses = it.chain(ctt.Status.Group.SCHEDULER_FAILURE, ctt.Status.Group.DRAFT)
        if (
            scheduler.plan.retry == cts.Retry.INTERVAL and
            last_task and last_task.execution.status in retriable_statuses
        ):
            retry_interval = scheduler.plan.retry_interval
            finished = last_task.execution.last_execution_finish
            last_task_act_time = finished if finished and finished > dt.datetime(1970, 1, 1) else last_task.time.updated
            cls.logger.info(
                "Use retry_interval=%r for scheduler #%r because of failed task #%r",
                retry_interval, scheduler.id, last_task.id,
            )
            return last_task_act_time + dt.timedelta(seconds=retry_interval)
        return None

    @classmethod
    def _get_repetition_no_next_time(cls, start_time, last_task, manual):
        return start_time if manual or not last_task else None

    @classmethod
    def _get_repetition_weekly_next_time(cls, scheduler, start_time, current_time):
        if not scheduler.plan.days_of_week:
            return None
        start_time = start_time.replace(
            year=current_time.year,
            month=current_time.month,
            day=current_time.day,
        )
        for day_shift in xrange(0, 8):
            candidate = start_time + dt.timedelta(days=day_shift)
            if (scheduler.plan.days_of_week & (1 << candidate.weekday())) and candidate >= current_time:
                return candidate
        return None

    @classmethod
    def _get_repetition_interval_next_time(cls, scheduler, start_time, current_time):
        if start_time >= current_time:
            return start_time
        interval = scheduler.plan.interval
        seconds = max(int((current_time - start_time).total_seconds()) - 1, 0)
        return start_time + dt.timedelta(seconds=((seconds / interval) + 1) * interval)

    @classmethod
    def get_next_task_creation_time(cls, scheduler, current_time=None, manual=False, last_task=None):
        if last_task is None:
            last_task = cls._get_last_task(scheduler)

        if current_time is None:
            current_time = dt.datetime.utcnow()
        start_time = last_task.time.created if last_task else dt.datetime.min
        if scheduler.plan.start_mode == cts.StartMode.SET:
            start_time = scheduler.plan.start_time
        elif scheduler.plan.start_mode == cts.StartMode.IMMEDIATELY and manual:
            start_time = current_time

        repetition_type = scheduler.plan.repetition
        if scheduler.plan.repetition == cts.Repetition.NO:
            return cls._get_repetition_no_next_time(start_time, last_task, manual)
        elif repetition_type == cts.Repetition.WEEKLY:
            next_run = cls._get_repetition_weekly_next_time(scheduler, start_time, current_time)
        elif repetition_type == cts.Repetition.INTERVAL:
            next_run = cls._get_repetition_interval_next_time(scheduler, start_time, current_time)
        else:
            return
        if last_task and not manual:
            # ignore retry interval in case of manual run
            retry_time = cls._get_retry_time_if_applicable(scheduler, last_task)
            if retry_time is not None:
                next_run = retry_time
        return next_run

    @classmethod
    def _get_last_task(cls, scheduler):
        return mapping.Task.objects(
            scheduler=scheduler.id,
            execution__status__ne=ctt.Status.DRAFT,
        ).order_by("-id").first()
