from sandbox import common
import sandbox.common.types.user as ctu

from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.database.mapping import base as mapping_base

from . import task as task_controller


class _Exception(Exception):
    pass


class NotExists(_Exception):
    pass


class AlreadyExists(_Exception):
    pass


class ValidationError(_Exception):
    pass


class Semaphore(object):
    Model = mapping.Semaphore

    Exception = _Exception
    NotExists = NotExists
    AlreadyExists = AlreadyExists
    ValidationError = ValidationError

    @classmethod
    def initialize(cls):
        """
        initialize db for Semaphore model
        """
        cls.Model.ensure_indexes()

    @classmethod
    def can_modify(cls, user, owner, groups):
        return user.super_user or owner in groups

    @classmethod
    def rights(cls, user, doc, groups):
        ret = None
        if cls.can_modify(user, doc.owner, groups):
            ret = ctu.Rights.WRITE
        elif doc.shared and any(g in doc.shared for g in groups):
            ret = ctu.Rights.READ
        return ret

    @common.utils.classproperty
    def __request(self):
        return getattr(mapping_base.tls, "request", None)

    @classmethod
    def audit(cls, semaphore_id, description):
        settings = common.config.Registry()
        audit_item = mapping.SemaphoreAudit()
        audit_item.semaphore_id = semaphore_id
        audit_item.description = description
        audit_item.target = settings.this.id
        request = cls.__request
        if request:
            audit_item.author = request.user.login
            audit_item.source = request.remote_ip
        audit_item.save()

    @classmethod
    def __doc(cls, sem_id, sem):
        return cls.Model(
            id=sem_id,
            name=sem.name,
            owner=sem.owner,
            capacity=sem.capacity,
            auto=sem.auto,
            shared=sem.shared,
            public=sem.public,
            time=cls.Model.Time()
        )

    @classmethod
    def get(cls, sem_id):
        doc = cls.Model.objects.with_id(sem_id)
        if doc is None:
            sem = dict(task_controller.TaskQueue.qclient.semaphores(sem_id)).get(sem_id)
            if sem is not None:
                doc = cls.__doc(sem_id, sem)
        return doc

    @classmethod
    def create(cls, fields):
        """
        Create new semaphore with specified name

        :param fields: fields of Semaphore document
        :return: Semaphore object
        """
        try:
            sem_id, sem = task_controller.TaskQueue.qclient.create_semaphore(fields)
        except mapping.NotUniqueError:
            raise cls.AlreadyExists("Semaphore with name '{}' already exists".format(fields.get("name")))
        except (TypeError, ValueError) as ex:
            raise cls.ValidationError(str(ex))
        cls.audit(sem_id, "Created")
        return cls.__doc(sem_id, sem)

    @classmethod
    def update(cls, sem_id, fields, event=None):
        """
        Update existing semaphore

        :param sem_id: Semaphore id
        :param fields: document fields to update
        :param event: update description, for audit
        :return: Semaphore document
        """
        try:
            old_sem = dict(task_controller.TaskQueue.qclient.semaphores(sem_id)).get(sem_id)
            if old_sem is None:
                raise cls.NotExists("Semaphore #{} does not exist".format(sem_id))
            sem = task_controller.TaskQueue.qclient.update_semaphore(sem_id, fields)
        except (TypeError, ValueError) as ex:
            raise cls.ValidationError(str(ex))

        changes = {}
        for field in ("owner", "shared", "capacity", "auto", "public"):
            old, new = getattr(old_sem, field), getattr(sem, field)
            if old != new:
                changes[field] = (old, new)
        if changes:
            changes_description = ", ".join("{}: {}->{}".format(f, o, n) for f, (o, n) in changes.iteritems())
            if event and isinstance(event, unicode):
                event = event.encode("utf-8")
            cls.audit(sem_id, "{} ({})".format(event, changes_description) if event else changes_description)
        return cls.__doc(sem_id, sem)
