import time
import logging
import datetime as dt
import threading as th
import contextlib

from sandbox import common
import sandbox.common.types.database as ctd

from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.database import upgrade


logger = logging.getLogger(__name__)


class Settings(object):
    Model = mapping.Settings
    OperationMode = Model.OperationMode

    _master_checker_loop_lock = th.Lock()
    _model = None
    _no_master = False
    _actualized = None
    settings = None
    updating_interval = None
    actualizing_interval = None

    class DatabaseState(common.utils.Enum):
        """ Current database state. calculated property. """
        OK = None
        TOO_OLD = None
        TOO_NEW = None

    @classmethod
    def initialize(cls):
        cls.Model.ensure_indexes()
        cls.settings = common.config.Registry()
        cls._actualized = dt.datetime.utcnow()
        cls.updating_interval = dt.timedelta(seconds=cls.settings.server.state_checker.ro_recheck_interval)
        cls.actualizing_interval = dt.timedelta(seconds=cls.settings.server.state_checker.usage_update_interval)

    @common.utils.classproperty
    def model(cls):
        """ Returns current model object fetched from the database. """
        now = dt.datetime.utcnow()
        if cls._model is None or cls._model.time is None or cls._model.time.used + cls.updating_interval < now:
            try:
                cls._model = cls.Model.objects.next()
                cls._model.time.used = now
                if not cls._actualized or cls._actualized + cls.actualizing_interval < now:
                    cls._actualized = now
                    cls._model.save()
            except (mapping.BSONError, mapping.InvalidDocumentError, StopIteration) as ex:
                # Outdated/no any settings in the database yet.
                if not isinstance(ex, StopIteration):
                    logger.exception('Error fetching settings object. Re-initializing it.')
                    cls.Model.drop_collection()
                cls._model = cls.Model()
                cls._model.time = cls.Model.Time()
                cls._model.updates = cls.Model.Updates(applied=cls.Model.Updates.Appiled(
                    pre=upgrade.__pre__, main=upgrade.__all__, post=upgrade.__all__,
                ))
                cls._model.save()
        return cls._model

    @classmethod
    def _master_checker_loop(cls):
        """
        Separated thread loop which will try to establish a master database connection after connectivity problem event
        """
        logging.info("Database master connection checker thread started.")
        while cls._no_master:
            try:
                if mapping.get_connection(
                    ctd.ReadPreference.PRIMARY, reconnect=True
                ).rw.connection.is_primary:
                    cls._no_master = False
                    logging.info("Database master connection established.")
                else:
                    logging.info("No master database.")
            except (mapping.ConnectionFailure, mapping.ConnectionError) as ex:
                logging.warning("Database master connection cannot be established: %s", ex)
            time.sleep(1)
        logging.info("Database master connection checker thread stopped.")

    @classmethod
    def on_master_lost(cls):
        with cls._master_checker_loop_lock:
            if cls._no_master:
                return
            cls._no_master = True
            service = th.Thread(target=cls._master_checker_loop)
            service.daemon = True
            service.start()

    @classmethod
    def mode(cls):
        """ Returns current operation mode of the server (one of `Model.OperationMode`). """
        if cls._no_master:
            return cls.Model.OperationMode.READ_ONLY

        if cls.model.operation_mode != cls.Model.OperationMode.NORMAL:
            return cls.model.operation_mode
        if cls.state != cls.DatabaseState.OK:
            return cls.Model.OperationMode.READ_ONLY
        return cls.Model.OperationMode.NORMAL

    @classmethod
    def set_mode(cls, value):
        """ Current server's operation mode setter. """
        model = cls.model
        model.reload()
        cls._actualized = model.time.used = dt.datetime.utcnow()
        if isinstance(value, tuple):
            model.operation_mode = value[0]
            model.operation_mode_set_by = value[1]
        else:
            model.operation_mode = value
        model.save()

    @common.utils.classproperty
    def state(cls):
        """ Determines and returns current database state (one of `Model.DatabaseState`). """
        model = cls.model
        applied = set(model.updates.applied.main or [])
        known = set(upgrade.__all__)
        if len(applied) < len(known):
            return cls.DatabaseState.TOO_OLD
        if len(applied) > len(known):
            return cls.DatabaseState.TOO_NEW
        if applied ^ known:
            return cls.DatabaseState.TOO_OLD
        return cls.DatabaseState.OK

    @classmethod
    @contextlib.contextmanager
    def upgrade_recorder(cls, node=None):
        """ Context manager which records the database upgrade process. """
        class Recorder(object):
            recorded = cls.model.updates.applied

            @staticmethod
            def record(update, type):
                """
                Records update applied.
                :param update:  update name, which was applied.
                :param type:    update type name - one of `upgrade.Type` elements.
                """
                lst = getattr(cls.model.updates.applied, type)
                if update not in lst:
                    lst.append(update)
                    cls.model.save()

        try:
            cls.model.updates.executor = node if node else cls.settings.this.id
            cls.model.save()
            yield Recorder()

        finally:
            cls.model.time.updated = dt.datetime.now()
            cls.model.updates.executor = None
            cls.model.save()
