import os
import re
import sys
import time
import socket
import pymongo
import cPickle
import logging
import calendar
import urlparse
import datetime as dt
import operator as op
import itertools as it
import functools as ft
import threading as th

import six

from sandbox.common import data as common_data
from sandbox.common import enum as common_enum
from sandbox.common import rest as common_rest
from sandbox.common import errors as common_errors
from sandbox.common import config as common_config
from sandbox.common import format as common_format
from sandbox.common import context as common_context
from sandbox.common import encoding as common_encoding
from sandbox.common import patterns as common_patterns
from sandbox.common import platform as common_platform
from sandbox.common import itertools as common_itertools
from sandbox.common import statistics as common_statistics
import sandbox.common.types.misc as ctm
import sandbox.common.types.user as ctu
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.statistics as cts
import sandbox.common.types.notification as ctn

import sandbox.common.joint.errors as jerrors
import sandbox.common.joint.socket as jsocket

import sandbox.serviceq.types as qtypes
import sandbox.serviceq.client as qclient
import sandbox.serviceq.errors as qerrors
import sandbox.serviceq.config as qconfig

import sandbox.taskbox.errors as tb_errors
import sandbox.taskbox.model.bridge as model_bridge
import sandbox.taskbox.model.update as model_update

from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import context as request_context

from sandbox.yasandbox.controller import user as user_controller
from sandbox.yasandbox.controller import client as client_controller
from sandbox.yasandbox.controller import dispatch
from sandbox.yasandbox.controller import trigger as trigger_controller
from sandbox.yasandbox.controller import settings as settings_controller
from sandbox.yasandbox.controller import notification as notification_controller
from sandbox.yasandbox.controller import task_status_event as tse_controller

# TODO: remove after SANDBOX-4646
ROBOT_OWNERS_WHITE_LIST = """
    robot-arcanum
    robot-aspam-uploader
    robot-awaps-pvl
    robot-axis
    robot-balance-getter
    robot-bass-blender
    robot-bm-admin
    robot-canons-dloader
    robot-capture-rate
    robot-collections
    robot-dssm-conveyor
    robot-elements
    robot-fact-snip
    robot-freshness
    robot-gandalf
    robot-gemini-db
    robot-jams
    robot-lemur-learn
    robot-mailspam
    robot-maps-sandbox
    robot-market-ugc-bot
    robot-mayql
    robot-mlmarines
    robot-modadvert-bcup
    robot-modadvert-yt
    robot-mpfs-stat
    robot-muzsearch
    robot-pssp-profile
    robot-rasp
    robot-rem
    robot-safesearch
    robot-sbs
    robot-similar
    robot-strm
    robot-yabs-antispam
    robot-yabs-google
    robot-yeti-v-nirvane
    teamcity
    zomb-carl-ekb
""".split()


class EnqueueResult(object):
    def __init__(self, new_status, wait_targets=None):
        self.new_status = new_status
        self.wait_targets = wait_targets


class TaskWrapper(object):

    SET_STATUS_TIMEOUT = 600

    class FilterCheckResult(dict):
        """ Task filter result """
        def __init__(self, filter_name, task_parameters, client_values, allowed):
            # noinspection PyTypeChecker
            super(TaskWrapper.FilterCheckResult, self).__init__({
                "filter_name": filter_name,
                "task_parameters": task_parameters,
                "client_values": client_values,
                "allowed": allowed
            })

        def __nonzero__(self):
            return bool(self["allowed"])

        def __str__(self):
            allowed = "allowed" if self["allowed"] else "banned"
            return "Filter {} results: {}, task parameters: {}, client values: {}".format(
                self["filter_name"], allowed, self["task_parameters"], self["client_values"]
            )

        def __repr__(self):
            return str(self)

    def __new__(cls, model, **kws):
        if cls is not TaskWrapper:
            return object.__new__(cls)

        if Task.is_taskboxed_task(model):
            return TBWrapper(model, **kws)

        from sandbox import sdk2
        if model.type in sdk2.Task:
            return NewTaskWrapper(model, **kws)
        return OldTaskWrapper(model, **kws)

    def __init__(self, model, logger=None):
        assert isinstance(
            model, (mapping.Template, mapping.base.LiteDocument)
        ), "`model` must be an instance of {!r} or {!r}".format(mapping.Template, mapping.base.LiteDocument)
        self._model = model
        self._logger = logger or logging
        self._tasks = {}
        self._rest = None  # used to cache server-side REST client class

    @property
    def sdk_version(self):
        return None

    @common_patterns.singleton_property
    def cls(self):
        return None

    def _task(self):
        return None

    def __repr__(self):
        return repr(self._task())

    @property
    def sdk_type(self):
        return None

    @classmethod
    def fast_load_list(cls, ids, load_ctx=True, logger=None):
        query = mapping.Task.objects
        if not load_ctx:
            query.exclude("context")
        objects = query.in_bulk(map(mapping.ObjectId, ids))
        # noinspection PyTypeChecker
        return map(ft.partial(cls, logger=logger), map(objects.get, ids))

    @property
    def model(self):
        return self._model

    @property
    def id(self):
        return self._model.id

    @staticmethod
    def service(task_cls):
        if task_cls is None:
            return False
        from sandbox import sdk2
        import sandbox.yasandbox.proxy.task
        return (
            issubclass(task_cls, sdk2.ServiceTask) or
            issubclass(task_cls, sandbox.yasandbox.proxy.task.Task) and task_cls.SERVICE
        )

    @property
    def kill_timeout(self):
        return None

    @property
    def fail_on_any_error(self):
        return None

    @property
    def enable_yav(self):
        return bool(self._model.enable_yav)

    @property
    def dump_disk_usage(self):
        return bool(self._model.dump_disk_usage)

    @property
    def tcpdump_args(self):
        return self._model.tcpdump_args

    def save_trace(self, exc=None, save=True, update_gsid=True):
        ctx = cPickle.loads(self.model.context)
        ctx["__last_error_trace"] = common_format.format_exception()
        self.model.context = str(cPickle.dumps(ctx))
        if save:
            self.save()

    @common_patterns.singleton_property
    def parent(self):
        parent_id = getattr(self._model, "parent_id", None)
        return mapping.Task.objects.with_id(parent_id) if parent_id else None

    def init_model(self):
        parent = None

        if isinstance(self._model, mapping.Task):
            parent = self.parent
            self._model.execution = self._model.execution or mapping.Task.Execution()
            self._model.execution.auto_restart = (
                self._model.execution.auto_restart or mapping.Task.Execution.AutoRestart()
            )
            self._model.execution.status = self._model.execution.status or ctt.Status.DRAFT
            self._model.execution.time = mapping.Task.Execution.Time()
            self._model.execution.disk_usage = self._model.execution.disk_usage or mapping.Task.Execution.DiskUsage()
            self._model.time = mapping.Task.Time()

        self._model.description = self._model.description or ""
        self._model.hidden = self._model.hidden or (parent.hidden if parent else False)
        self._model.author = parent.author if parent else self._model.author
        self._model.suspend_on_status = parent.suspend_on_status if parent else self._model.suspend_on_status
        self._model.push_tasks_resource = parent.push_tasks_resource if parent else self._model.push_tasks_resource
        self._model.score = parent.score if parent else self._model.score

        if self._model.priority is None:
            if (
                Task.request and Task.request.source == ctt.RequestSource.WEB and
                (Task.request.user.login != ctu.ANONYMOUS_LOGIN or Task.request.user.super_user)
            ):
                self._model.priority = int(ctu.DEFAULT_PRIORITY_LIMITS.api.next)
            else:
                self._model.priority = int(ctt.Priority())

        self._model.requirements = self._model.requirements or mapping.Task.Requirements()
        self._model.requirements.cpu_model = self._model.requirements.cpu_model or ""
        self._model.requirements.platform = self._model.requirements.platform or ctm.OSFamily.ANY
        settings = common_config.Registry()
        if settings.common.installation == ctm.Installation.LOCAL:
            self._model.requirements.host = settings.this.id

    def create(self):
        self.init_model()

        self.save()
        self._logger.info(
            "New task #%s of type %r in status %r created.",
            self._model.id, self._model.type, self._model.execution.status
        )

        audit_message = "Created"
        if Task.request and Task.request.user.login != self._model.author:
            audit_message += " by {}".format(Task.request.user.login)

        audit_item = mapping.Audit(
            task_id=self._model.id,
            content=audit_message,
            status=self._model.execution.status,
            author=self._model.author,
            source=Task.request.source if Task.request else None
        )
        Task.audit(audit_item)
        return self

    def on_create(self):
        return self

    @request_context.timer_decorator("TaskWrapper.save")
    def save(self, save_condition=None):
        Task.save(self._model, save_condition=save_condition, logger=self._logger)
        return self

    def on_save(self):
        return self

    def _update_task_gsid(self, context):
        prev_gsid = context.pop("GSID") if "GSID" in context else context.get("__GSID", "")
        if not prev_gsid and getattr(self._model, "parent_id", None):
            parent = TaskWrapper(self.parent)
            prev_gsid = parent.ctx.get("__GSID", "")
        context["__GSID"] = Task.extend_gsid(prev_gsid, self.id, self._model.type, self._model.author)
        return context

    def update_context(self, context, remove_fields=None, save=True, update_gsid=True):
        pass

    def copy(self, source):
        return self

    def update_common_fields(self, data):
        pass

    @common_patterns.singleton_property
    def ctx(self):
        return cPickle.loads(self._model.context) if self._model.context else {}

    def _update_requirements_in_model(self):
        pass

    def _update_parameters_in_model(self):
        pass

    def _update_context_in_model(self):
        pass

    def update_model(self, parameters=True):
        self._update_requirements_in_model()
        self._update_context_in_model()
        if parameters:
            self._update_parameters_in_model()

    @dispatch.local
    def release_template(self):
        return None

    def resources(self, resource_type=""):
        from sandbox.yasandbox import manager
        return manager.resource_manager.list_task_resources(self._model.id, resource_type=resource_type)

    @common_patterns.singleton_property
    def iteration(self):
        return mapping.Resource.objects(task_id=self.id, type="TASK_LOGS").count()

    def invalidate_resources(self, with_hosts=False):
        """
        Mark all `NOT_READY` resources as broken.

        :param bool with_hosts: Invalidate only resources with hosts
        """
        Task.invalidate_resources(self.model, with_hosts=with_hosts)

    @property
    def short_task_result(self):
        return None

    @property
    def long_task_result(self):
        return ""

    @property
    def input_parameters(self):
        return ()

    @property
    def output_parameters(self):
        return ()

    @common_patterns.singleton_property
    def parameters_meta(self):
        """
        :rtype: mapping.ParametersMeta
        """
        return None

    def parameters_dicts(self, parameters_meta=None, input=True, output=True):
        """
        Returns dicts of task input and output parameters.

        :type parameters_meta: mapping.ParametersMeta
        """
        if parameters_meta is None:
            parameters_meta = self.parameters_meta
        in_params_dict, out_params_dict = {}, {}
        in_values = self._input_parameters_raw_values() if input else {}
        out_values = self._output_parameters_raw_values() if output else {}
        for pm in parameters_meta.params:
            if (pm.output and not output) or (not pm.output and not input):
                continue
            values = out_values if pm.output else in_values
            if pm.output and pm.name not in values:
                continue
            value = model_update.decode_parameter(values.get(pm.name, pm.default), pm.complex)
            (out_params_dict if pm.output else in_params_dict)[pm.name] = value
        return in_params_dict, out_params_dict

    def _input_parameters_raw_values(self):
        """ Returns raw dict of input parameters from DB model. """
        return {}

    def _output_parameters_raw_values(self):
        """ Returns raw dict of output parameters from DB model. """
        return {}

    def _parameters_values_getter(self, mode):
        return {}

    def _parameters_values_setter(self, mode, value):
        pass

    @property
    def input_parameters_values(self):
        return self._parameters_values_getter("input")

    @input_parameters_values.setter
    def input_parameters_values(self, value):
        self._parameters_values_setter("input", value)

    @property
    def output_parameters_values(self):
        return self._parameters_values_getter("output")

    @output_parameters_values.setter
    def output_parameters_values(self, value):
        self._parameters_values_setter("output", value)

    def update_parameters(self, input_updates, output_updates, requirements_updates, validate_only):
        """
        :type input_updates: dict
        :type output_updates: dict
        :type requirements_updates: dict
        :type validate_only: bool
        :rtype: sandbox.taskbox.client.protocol.ParametersUpdateResult
        """

    def update_hints(self):
        """ Update task hints """

    @common_patterns.singleton_property
    def needs_container(self):
        """ Returns specific container id if a task requires it, or True if any container is ok """
        from sandbox.sdk2 import legacy
        if self._model.requirements.container_resource:
            return self._model.requirements.container_resource
        cnt = next((_ for _ in self.input_parameters if issubclass(_, legacy.Container)), None)
        if cnt:
            container_id = self.input_parameters_values.get(cnt.name)
            if container_id:
                self._logger.warning(
                    "Got container resource from parameters for task #%s of type %s",
                    self._model.id, self._model.type
                )
                return container_id
        if self._model.requirements.privileged:
            return True

    def get_container_platform(self):
        cnt = self.needs_container
        return Task.get_container_platform_impl(cnt)

    def has_custom_porto_properties(self):
        return False

    def get_porto_container_properties(self):
        return [], None  # (layers, platform)

    @property
    def type_client_tags(self):
        return ()

    @property
    def client_tags(self):
        return ()

    def add_explicit_hints(self, hints):
        """ Add new hints directly to database """
        hints = list(hints)
        mapping.Task.objects(id=self.id).update_one(
            add_to_set__explicit_hints=hints,
            add_to_set__hints=hints
        )

    def reset_expires(self):
        expires_at = ctm.NotExists
        if self._model.expires_at:
            expires_at = self._model.expires_at = None
        if self._model.expires_delta:
            model_update.LocalUpdate.set_expires(self._model.expires_delta, self._model)
            expires_at = self._model.expires_at
        if expires_at is not ctm.NotExists:
            self._logger.debug("Resetting task #%s expiration time to %s", self.id, expires_at)
            mapping.Task.objects(id=self.id).update_one(set__expires_at=expires_at)

    @property
    @dispatch.local
    def footer(self):
        return self._task().footer

    @property
    def reports(self):
        """
        :rtype: list[mapping.Template.ReportInfo]
        """
        return []

    @dispatch.local
    def report(self, label):
        return None

    @property
    def host(self):
        return self._model.execution.host

    @property
    def status(self):
        return self._model.execution.status

    @property
    def release_to(self):
        return None

    def set_status(
        self, new_status, event=None, lock_host="", keep_lock=None, force=False,
        expected_status=None, reset_resources_on_failure=False, wait_targets=None, timer=None,
        set_last_action_user=False, host=None
    ):
        """
        Set status to new_status

        :param new_status: new task status
        :param event: event that cause status changing
        :param lock_host: new lock value
        :param keep_lock: forces lock keep - do not update a record, if the lock is not the same
        :param force: check the new status is applicable
        :param expected_status: current task status, use to prevent conflict
        :param reset_resources_on_failure: mark resources as BROKEN if task goes to FAILURE due to `fail_on_any_error`
        :param wait_targets: targets for waiting statuses
        :param timer: Timer object to trace of executing time of request
        :param set_last_action_user: if True set last_action_user of task
        :param host: host name for task execution
        """
        return Task.set_status(
            task=self.model,
            new_status=new_status,
            event=event,
            lock_host=lock_host,
            keep_lock=keep_lock,
            force=force,
            expected_status=expected_status,
            reset_resources_on_failure=reset_resources_on_failure,
            wait_targets=wait_targets,
            timer=timer,
            logger=self._logger,
            set_last_action_user=set_last_action_user,
            host=host
        )

    def release_lock_host(self, **filters):
        mapping.retry_temporary_errors(
            lambda: mapping.Task.objects(id=self.id, **filters).update_one(set__lock_host=""),
            lambda ex: self._logger.warning("Failed releasing lock for task #%s, retrying: %s", self.id, ex),
            timeout=self.SET_STATUS_TIMEOUT
        )

    def set_info(self, info, do_escape=True):
        """
        Adds text into field 'info'

        :param info: test to add
        :param do_escape: if True, escape text before add
        :return: added text
        """

    @property
    def tasks_archive_resource(self):
        return None

    @property
    def tags(self):
        return []

    @staticmethod
    @request_context.timer_decorator()
    def validate_custom_tasks_resource(task_id, resource_id):
        if resource_id is None:
            return

        resource = model_update.ServerSideUpdate.validate_tasks_resource(resource_id)
        mapping.Task.objects(id=task_id).update_one(add_to_set__requirements__resources=resource.id)

    def _register_dependencies(self):
        pass

    def _handle_wait_time(self, ex):
        return EnqueueResult(ctt.Status.WAIT_TIME)

    def _handle_wait_task(self, ex):
        return EnqueueResult(ctt.Status.WAIT_TASK)

    def _on_enqueue_impl(self):
        raise NotImplementedError

    @dispatch.local
    def on_enqueue(self):
        """
        It will call the method :meth:`on_enqueue`.
        """
        from sandbox import sdk2
        with request_context.current.timer["TaskWrapper.on_enqueue"] as timer:
            self.validate_custom_tasks_resource(self.id, self.tasks_archive_resource)
            self._register_dependencies()
            enqueue_result = EnqueueResult(ctt.Status.ENQUEUING)
            retries = 5
            for retry in six.moves.range(1, retries + 1):
                try:
                    self._on_enqueue_impl()

                except (common_errors.WaitTime, sdk2.WaitTime) as ex:
                    enqueue_result = self._handle_wait_time(ex)

                except (common_errors.WaitTask, sdk2.WaitTask, sdk2.WaitOutput) as ex:
                    try:
                        enqueue_result = self._handle_wait_task(ex)
                    except common_errors.NothingToWait:
                        if retry < retries:
                            message = "Wakeup conditions are already met, running on_enqueue() again"
                            self._logger.info("Task #%s: %s", self.id, message)
                            Task.audit(mapping.Audit(task_id=self.id, content=message))
                            continue
                        else:
                            raise

                except common_errors.NothingToWait:
                    # for SDK1 task.on_enqueue() is supposed to run only once
                    self._logger.info("Task #%s: No tasks to wait, ignoring", self.id)

                except common_errors.TaskStop as ex:
                    enqueue_result.new_status = ctt.Status.STOPPING
                    if ex.args:
                        self.set_info(ex.args[0])

                except common_errors.TaskFailure as ex:
                    enqueue_result.new_status = ctt.Status.FAILURE
                    if ex.args:
                        self.set_info(ex.args[0])

                break

            self._logger.info("Task #%s has been enqueued totally in %s", self.id, timer)
        return enqueue_result

    @classmethod
    def before_task_delete(cls, task_id):
        for child in mapping.Task.objects(parent_id=task_id, execution__status__ne=ctt.Status.RELEASED):
            TaskWrapper(child).delete(event="Parent task deleted")

        expires = dt.datetime.utcnow() + dt.timedelta(days=1)
        logging.info("Processing task #%s delete event.", task_id)
        for res in mapping.Resource.objects(task_id=task_id, state=ctr.State.READY):
            attrs = [
                _ for _ in res.attributes
                if _.key not in (ctr.ServiceAttributes.RELEASED, ctr.ServiceAttributes.TTL)
            ]
            if len(attrs) != len(res.attributes):
                logging.info(
                    "Expire resource #%s of deleted task #%s. New attributes: %r",
                    res.id, task_id, {_.key: _.value for _ in attrs}
                )
                res.update(set__attributes=attrs, set__time__expires=expires)

    @common_patterns.singleton_property
    def effective_client_tags(self):
        return Task.effective_client_tags_impl(self._model, self.client_tags)

    def get_client_filters_info(self, client, cache=None):
        """
        Gets info of all filters for the task and given client host

        :param client: client object
        :param cache: cache by clients
        :return: list of filter result objects
        :rtype: list of FilterCheckResult
        """
        return Task.get_client_filters_info(self._model, client.mapping(), cache=cache)

    def clients_with_compatibility_info(self):
        """
        Returns a list of all currently registered clients with detailed information
        about task-to-client compatibility sorted by compatibility.
        """
        import sandbox.yasandbox.proxy.task as task_model

        result = []
        task_queue = TaskQueue.qclient.task_queue(self.id, secondary=True)
        cache = TaskQueue.HostsCache()
        for client in cache.client_mappings.itervalues():
            alive = client.alive
            queued = client.hostname in task_queue
            filter_results = [task_model.FilterCheckResult("alive", True, alive, alive)]
            filter_results += Task.get_client_filters_info(self.model, client)
            filter_results.append(Task.check_available_space_filter(self._model, client))
            result.append((client.hostname, {
                "allowed": queued and all(filter_results),
                "filters": map(dict, filter_results),
                "queue": task_queue.get(client.hostname)
            }))
        return sorted(
            result,
            key=lambda x: (not x[1]["allowed"], x[1]["queue"][0] if x[1]["queue"] else sys.maxint, x[0])
        )

    def _check_permission(self, action):
        if (
            Task.request and
            not user_controller.user_has_permission(Task.request.user, (self._model.author, self._model.owner))
        ):
            raise common_errors.AuthorizationError(
                "User {!r} is not permitted to {} task #{} (author: {!r}, owner: {!r})".format(
                    Task.request.user.login, action, self.id, self._model.author, self._model.owner
                )
            )

    def author_mismatch(self):
        """
        Return True if the request is not from the task author.
        :rtype: bool
        """
        request = Task.request
        if not request:
            return False

        mismatch = (not request.user.super_user and request.user.login != self._model.author)
        if mismatch:
            owned_robots = mapping.RobotOwner.objects.fast_scalar("robots").with_id(request.user.login) or []
            mismatch = self._model.author not in owned_robots
        return mismatch

    def restart(self, event=None):
        """
        Restart task

        :param event: string with description of event that cause restarting
        :return: True, if task was restarted, False otherwise
        """

    def stop(self, event=None):
        """
        Stop task execution

        :param event: string with description of event that cause stop
        :return: True if task was successfully stopped, False otherwise
        """

    def delete(self, event=None):
        """
        Delete task

        :param event: string with description of event that cause deleting
        :return: True if task was successfully deleted, False otherwise
        """

    def suspend(self, event=None):
        """
        Suspend task execution

        :param event: string with description of event that cause suspend
        """

    def resume(self, event=None):
        """
        Resume execution of the suspended task

        :param event: string with description of event that cause resume
        """

    def expire(self, event=None):
        """
        Expire a task

        :param event: string with description of event which caused expiration
        """


class NewTaskWrapper(TaskWrapper):

    def __init__(self, model, logger=None):
        self._tb = self._create_task_bridge(model)
        super(NewTaskWrapper, self).__init__(model, logger)

    def _create_task_bridge(self, model):
        def setup_agentr(task):
            task.agentr = dispatch.AgentR(task)

        request_id = "DEADC0DE" if Task.request is None else Task.request.id[:8]
        return model_bridge.LocalTaskBridge(model, request_id, setup_agentr=setup_agentr)

    @property
    def sdk_version(self):
        return 2

    @property
    def _model(self):
        return self._tb.model

    @_model.setter
    def _model(self, value):
        self._tb.model = value

    @common_patterns.singleton_property
    def cls(self):
        from sandbox import sdk2
        return sdk2.Task[self._model.type]

    @property
    def sdk_type(self):
        resource_id = self.tasks_archive_resource
        if resource_id is None:
            return ctt.SDKType.SDK2
        resource_type = mapping.Resource.objects.fast_scalar("type").with_id(resource_id)
        from sandbox import sdk2
        if resource_type is None or resource_type != str(sdk2.service_resources.SandboxTasksBinary):
            return ctt.SDKType.SDK2
        else:
            return ctt.SDKType.BINARY

    @dispatch.local
    def _task(self, model=None, reset=False):
        from sandbox import sdk2
        if model is None:
            model = self._model
        task = self._tasks.get(model.id, ctm.NotExists)
        if reset or task is ctm.NotExists:
            task = self._tasks[model.id] = sdk2.Task.from_model(model)
            task.agentr = dispatch.AgentR(task)
        return task

    def _handle_wait_time(self, ex):
        task_id = self._model.id
        time_ = dt.datetime.utcnow() + dt.timedelta(seconds=ex.timeout)
        try:
            if trigger_controller.TimeTrigger.create(
                trigger_controller.TimeTrigger.Model(source=task_id, time=time_, activated=True)
            ):
                return EnqueueResult(ctt.Status.WAIT_TIME, {"time": ex.timeout})
        except trigger_controller.TimeTrigger.AlreadyExists:
            self._logger.warning("Time trigger for task %s already exists", task_id)
        return EnqueueResult(ctt.Status.ENQUEUING)

    def _handle_wait_task(self, ex):
        from sandbox import sdk2
        task_id = self._model.id
        try:
            if self._make_wait_task_trigger(task_id, ex):
                wait_targets = {}
                if isinstance(ex, sdk2.WaitTask):
                    wait_targets["tasks"] = ex.tasks
                else:
                    wait_targets["output_parameters"] = {str(k): v for k, v in ex.targets.iteritems()}
                new_status = ctt.Status.WAIT_TASK if isinstance(ex, sdk2.WaitTask) else ctt.Status.WAIT_OUT
                return EnqueueResult(new_status, wait_targets)
        except trigger_controller.TaskStatusTrigger.AlreadyExists:
            self._logger.warning("Status trigger for task %s already exists", task_id)
        except trigger_controller.TaskOutputTrigger.AlreadyExists:
            self._logger.warning("Output trigger for task %s already exists", task_id)
        except trigger_controller.TimeTrigger.AlreadyExists:
            self._logger.warning("Time trigger for task %s already exists", task_id)
        return EnqueueResult(ctt.Status.ENQUEUING)

    def _make_wait_task_trigger(self, task_id, ex):
        from sandbox import sdk2
        if isinstance(ex, sdk2.WaitTask):
            targets = trigger_controller.TaskStatusTrigger.get_not_ready_targets(
                targets=ex.tasks, statuses=ex.statuses
            )
            if not targets or (not ex.wait_all and len(ex.tasks) != len(targets)):
                raise common_errors.NothingToWait

            triggers = [
                trigger_controller.TaskStatusTrigger.create(
                    trigger_controller.TaskStatusTrigger.Model(
                        source=task_id,
                        targets=targets,
                        statuses=ex.statuses,
                        wait_all=ex.wait_all,
                        activated=True,
                    )
                )
            ]
        else:
            tf_model = trigger_controller.TaskOutputTrigger.Model.TargetField

            task_ids, fields = [], []
            for tid, flds in ex.targets.iteritems():
                for fld in common_itertools.chain(flds):
                    task_ids.append(int(tid))
                    fields.append(fld)

            tasks = {
                tid: set(p.key for p in output_parameters)
                for tid, output_parameters in (
                    mapping.Task.objects(id__in=ex.targets.iterkeys()).scalar("id", "parameters__output")
                )
            }

            missing = set(task_ids) - set(tasks)
            if missing:
                missing_text = ", ".join(map(str, missing))
                raise common_errors.TaskNotFound("Cannot wait for non-existing task(s): {}".format(missing_text))

            targets = [
                tf_model(target=tid, field=field) for tid, field in it.izip(task_ids, fields)
                if field not in tasks[tid]
            ]

            if not targets or (not ex.wait_all and len(task_ids) != len(targets)):
                raise common_errors.NothingToWait

            triggers = [
                trigger_controller.TaskOutputTrigger.create(
                    trigger_controller.TaskOutputTrigger.Model(
                        source=task_id,
                        targets=targets,
                        wait_all=ex.wait_all
                    )
                )
            ]
        if ex.timeout:
            time_ = dt.datetime.utcnow() + dt.timedelta(seconds=ex.timeout)
            # noinspection PyTypeChecker
            triggers.append(
                trigger_controller.TimeTrigger.create(
                    trigger_controller.TimeTrigger.Model(source=task_id, time=time_)
                )
            )
        return filter(None, triggers)

    @property
    def kill_timeout(self):
        return Task.kill_timeout(self.model)

    @property
    def fail_on_any_error(self):
        return bool(self._model.fail_on_any_error)

    @property
    def tags(self):
        return self._model.tags or []

    @tags.setter
    def tags(self, value):
        model_update.ServerSideUpdate.update_tags(value, self._model)

    @dispatch.local
    def release_template(self):
        return self._tb.release_template()

    @property
    def reports(self):
        return self._tb.reports()

    @dispatch.local
    def report(self, label):
        return self._tb.report(label)

    def init_model(self):
        self._tb.init_model()
        self._tb.parameters_meta()
        context = cPickle.loads(self._model.context)
        self._update_task_gsid(context)
        self._model.context = cPickle.dumps(context)

        super(NewTaskWrapper, self).init_model()

    @dispatch.local
    def create(self):
        return super(NewTaskWrapper, self).create()

    def save_trace(self, exc=None, save=True, update_gsid=True):
        self.update_context(
            {"__last_error_trace": common_format.format_exception()}, save=save, update_gsid=update_gsid
        )

    @dispatch.local
    def on_create(self):
        super(NewTaskWrapper, self).on_create()
        try:
            self._tb.on_create()
            self._update_notifications()
            self.update_common_fields({})
        except Exception as ex:
            self._logger.exception("Error occurred while calling method `on_create` for #%d", self._model.id)
            Task.audit(mapping.Audit(
                task_id=self._model.id,
                author=self._model.author,
                content="Error occurred while calling method `on_create`: {}".format(ex),
            ))
            # Save all changes to model because we are about to re-raise exception
            # and we don't know whether caller is going to handle it
            self.save_trace(ex)
            raise
        return self

    @dispatch.local
    def _update_requirements_in_model(self):
        model_update.LocalUpdate.update_requirements(self._task(), self._model)

    @dispatch.local
    def _update_parameters_in_model(self):
        model_update.LocalUpdate.update_parameters(self._task(), self._model)

    @dispatch.local
    def _update_context_in_model(self):
        model_update.LocalUpdate.update_context(self._task(), self._model)

    @request_context.timer_decorator()
    def update_context(self, context, remove_fields=None, save=True, update_gsid=True):
        if not context and not remove_fields:
            return
        if "GSID" in context and update_gsid:
            self._update_task_gsid(context)
        ctx = cPickle.loads(self._model.context) if self._model.context else {}
        for field in (remove_fields or []):
            ctx.pop(field, None)
        ctx.update(context)
        self._model.context = str(cPickle.dumps(ctx))
        if save:
            self.save()

    def _update_notifications(self):
        notifications = [
            {
                "statuses": n.statuses,
                "recipients": n.recipients,
                "transport": n.transport,
                "check_status": n.check_status,
                "juggler_tags": n.juggler_tags
            }
            for n in self._model.notifications or ()
        ]
        Task.setup_notifications(self._model, notifications)

    def _on_save_impl(self):
        self._tb.on_save(self._task(reset=True))

    @dispatch.local
    def on_save(self):
        if self._model.execution.status != ctt.Status.DRAFT:
            return
        try:
            with common_rest.DispatchedClient as _dispatch:
                _dispatch(dispatch.invalid_rest_call_client(self._rest, "Local REST calls are forbidden in on_save"))
                self._on_save_impl()

            self._update_notifications()
            self.update_common_fields({})
        except Exception as ex:
            self._logger.exception("Error occurred while calling method `on_save` for task #%s", self._model.id)
            Task.audit(mapping.Audit(
                task_id=self._model.id,
                author=self._model.author,
                content="Error occurred while calling method `on_save`: {}".format(ex),
            ))
            # Save all changes to model because we are about to re-raise exception
            # and we don't know whether caller is going to handle it
            self.save_trace(ex)
            raise

        return self

    def _on_enqueue_impl(self):
        self._tb.on_enqueue(self._task())

    @property
    def input_parameters(self):
        return filter(lambda _: not _.__output__, self.cls.Parameters)

    @property
    def output_parameters(self):
        return filter(lambda _: _.__output__, self.cls.Parameters)

    @common_patterns.singleton_property
    def parameters_meta(self):
        return self._tb.parameters_meta()

    @staticmethod
    def __parameters_raw_values(parameters, mode):
        parameters = getattr(parameters, mode) if parameters else []
        return {p.key: p.value for p in (parameters or []) if p.value is not None}

    def _input_parameters_raw_values(self):
        return self.__parameters_raw_values(self._tb.model.parameters, "input")

    def _output_parameters_raw_values(self):
        return self.__parameters_raw_values(self._tb.model.parameters, "output")

    def _parameters_values_getter(self, mode):
        parameters = {
            p.name: p
            for p in (self.input_parameters if mode == "input" else self.output_parameters)
        }
        return {
            i.key: model_update.decode_parameter(i.value, parameters[i.key].__complex_type__)
            for i in (getattr(self._model.parameters, mode) if self._model.parameters else ())
            if i.value is not None and i.key in parameters
        }

    def _parameters_values_setter(self, mode, value):
        parameters = self.input_parameters if mode == "input" else self.output_parameters
        if self._model.parameters is None:  # ensure parameters existence for SDK1->SDK2 migrated tasks
            self._model.parameters = mapping.Task.Parameters()
        setattr(
            self._model.parameters, mode,
            [
                mapping.Task.Parameters.Parameter(
                    key=p.name,
                    value=model_update.encode_parameter(value[p.name], p.__complex_type__)
                )
                for p in parameters
                if p.name in value
            ]
        )
        model_update.LocalUpdate.update_hints(self.cls.Parameters, self._model)

    @dispatch.local
    def update_parameters(self, input_updates, output_updates, requirements_updates, validate_only):
        return (self._tb.validate_parameters if validate_only else self._tb.update_parameters)(
            (input_updates, output_updates, requirements_updates)
        )

    def update_hints(self):
        return self._tb.update_hints()

    def has_custom_porto_properties(self):
        return bool(self._model.requirements.porto_layers)

    def get_porto_container_properties(self):
        # First layer is base layer and should have `platform` attribute
        return Task.get_porto_container_properties(self.model)

    def update_parameters_from_source(self, source_task, source_task_parameters):
        return (
            (p.name, model_update.encode_parameter(source_task_parameters[p.name], p.__complex_type__))
            for p in source_task.input_parameters if not p.do_not_copy and p.name in source_task_parameters
        )

    @dispatch.local
    def copy(self, source):
        """
        :type source: mapping.Template
        """
        source_task = TaskWrapper(source)
        parameters = {p.key: p.value for p in self._model.parameters.input}

        # in case the task was rewritten from SDK1 to SDK2, copy parameters from old model's context
        source_task_parameters = (
            source_task.ctx
            if source.parameters is None else
            source_task.input_parameters_values
        )
        parameters.update(self.update_parameters_from_source(source_task, source_task_parameters))
        self._model.parameters.input = [
            mapping.Task.Parameters.Parameter(key=k, value=v)
            for k, v in parameters.iteritems()
        ]

        new_requirements = source.requirements
        new_requirements.privileged = self._model.requirements.privileged
        new_requirements.ramdrive = self._model.requirements.ramdrive
        new_requirements.resources = ()
        new_requirements.semaphores = self._model.requirements.semaphores
        new_requirements.caches = self._model.requirements and self._model.requirements.caches
        self._model.requirements = new_requirements
        self._model.parameters_meta = source.parameters_meta

        self._model.description = source.description
        self._model.priority = source.priority
        self._model.max_restarts = source.max_restarts
        self._model.hidden = source.hidden
        self._model.fail_on_any_error = source.fail_on_any_error
        self._model.enable_yav = source.enable_yav
        self._model.dump_disk_usage = source.dump_disk_usage
        self._model.tcpdump_args = source.tcpdump_args
        self._model.tags = source.tags
        self._model.source_id = source.id
        self._model.tasks_archive_resource = source.tasks_archive_resource
        self._model.kill_timeout = source.kill_timeout
        self._model.suspend_on_status = source.suspend_on_status
        self._model.push_tasks_resource = source.push_tasks_resource
        self._model.score = source.score

        self._model.notifications = source.notifications
        for _ in self._model.notifications:
            _.recipients = map(
                lambda _: (
                    self._model.author
                    if source.author == _ else
                    _
                ),
                _.recipients
            )

        ctx = cPickle.loads(self._model.context)
        ctx["copy_of"] = source.id
        self._model.context = str(cPickle.dumps(ctx))

        self.save()
        self._tasks.pop(self._model.id, None)
        self.on_save()
        self.save()

        Task.audit(
            mapping.Audit(task_id=source.id, content="Copied to #{}".format(self.id))
        )
        Task.audit(
            mapping.Audit(task_id=self.id, content="Copied from #{}".format(source.id))
        )
        return self

    def update_common_fields(self, data):
        old_tags = self._model.tags
        model_update.LocalUpdate.update_common_fields(data, self._model)
        model_update.ServerSideUpdate.postprocess_common_fields(self._model, old_tags=old_tags)

    @property
    def type_client_tags(self):
        return self.cls.Requirements.client_tags.default

    @common_patterns.singleton_property
    def client_tags(self):
        tags = self._model.requirements.client_tags or self.cls.Requirements.client_tags.default
        return ctc.Tag.Query(tags)

    @property
    def release_to(self):
        return self._task().Parameters.release_to

    def set_info(self, info, do_escape=True):
        return Task.set_info(self.model, info, do_escape=do_escape, logger=self._logger)

    @property
    def tasks_archive_resource(self):
        return self._model.requirements.tasks_resource and self._model.requirements.tasks_resource.id

    def dependent_resources(self):
        return self._tb.dependent_resources_from_task(self._task())

    @dispatch.local
    @request_context.timer_decorator()
    def _register_dependencies(self):
        self._logger.info("Registering dependencies for task #%s", self.id)
        if self._model.execution.status != ctt.Status.DRAFT:
            return

        dependent_resources = set(self.dependent_resources())

        self._logger.info("Registering resources %r dependency for task #%s", sorted(dependent_resources), self.id)
        rids_tids = mapping.Resource.objects(id__in=dependent_resources).fast_scalar("id", "task_id")
        missed_rids = dependent_resources - set(_[0] for _ in rids_tids)
        if missed_rids:
            raise common_errors.TaskError("Resources {} don't exist".format(sorted(missed_rids)))
        self_rids = [rid for rid, tid in rids_tids if tid == self.id]
        if self_rids:
            raise common_errors.TaskError(
                "Task #{} cannot depend on own resources {}".format(self.id, sorted(self_rids))
            )
        mapping.Task.objects(id=self.id).update_one(add_to_set__requirements__resources=dependent_resources)

    def _reset_output_parameters(self):
        need_update = False
        for i, param in reversed(list(enumerate(self._model.parameters.output))):
            if param.reset_on_restart:
                need_update = True
                del self._model.parameters.output[i]
        if need_update:
            mapping.Task.objects(id=self.id).update_one(
                set__parameters__output=self._model.parameters.output
            )

    def restart(self, event=None):
        import sandbox.yasandbox.manager

        status = self._model.execution.status

        if not all((ctt.Status.can_switch(status, ctt.Status.ENQUEUING), status not in ctt.Status.Group.WAIT)):
            self._logger.warning("Task #%s cannot be restarted.", self.id)
            # FIXME: Golden crutch for arikon@ SANDBOX-3619
            # return self._model.execution.status in ctt.Status.Group.WAIT
            return False

        left = None

        if status == ctt.Status.TEMPORARY:
            left = self._model.execution.auto_restart.left
            if left is None:
                left = common_config.Registry().common.task.execution.max_restarts
            left -= 1

            if left < 0:
                self._logger.warning("No restarts left for task #%s", self.id)
                self.set_status(ctt.Status.EXCEPTION, "No auto restarts left", force=True)
                return False

        for resource in self.resources():
            if resource.is_deleted():
                sandbox.yasandbox.manager.resource_manager.reset_resource_hosts(resource.id)
                continue
            if resource.type.restart_policy == resource.type.OnRestart.IGNORE:
                continue

            reset = resource.type.restart_policy == resource.type.OnRestart.RESET
            sandbox.yasandbox.manager.resource_manager.reset_resource(
                resource.id,
                state=ctr.State.NOT_READY if reset else ctr.State.DELETED
            )

        self._reset_output_parameters()

        self.reset_expires()

        self.set_info("Task was restarted.")

        if TaskQueue.enqueue_task(self, event=event, logger=self._logger) and left is not None:
            run_interval = common_config.Registry().server.services.auto_restart_tasks.run_interval
            interval = self._model.execution.auto_restart.interval or run_interval
            interval = int(interval * 55 / 34)
            mapping.Task.objects.filter(id=self.id).update_one(
                set__execution__auto_restart__left=left,
                set__execution__auto_restart__interval=interval
            )

        self._logger.info("Task #%s has been restarted.", self.id)

        return True

    def stop(self, event=None):
        if self.status in (ctt.Status.STOPPING, ctt.Status.STOPPED):
            # Task is already stopping/stopped, nothing to do
            return True

        if not ctt.Status.can_switch(self.status, ctt.Status.STOPPING):
            # Can't stop task from current status
            return False

        self._check_permission("stop")

        for doc in mapping.Task.objects(parent_id=self.id):
            subtask = TaskWrapper(doc)
            subtask.stop(event)

        try:
            self.set_status(new_status=ctt.Status.STOPPING, event=event, set_last_action_user=True)
        except common_errors.UpdateConflict:
            # Task already switched to conflict status
            return False

        # Drop all wait triggers, task can't be triggered in STOPPED state.
        trigger_controller.delete_wait_triggers(self.id)

        session = mapping.OAuthCache.objects(task_id=self.id).first()
        if session:
            user_controller.OAuthCache.abort(session, reason=ctt.Status.STOPPING)
            self.invalidate_resources(with_hosts=True)

        return True

    @dispatch.local
    def delete(self, event=None):
        self._check_permission("delete")

        if self.status == ctt.Status.DELETED:
            return True

        if self.status == ctt.Status.DRAFT:
            self.set_status(new_status=ctt.Status.DELETED, event=event)
            return True

        if not ctt.Status.can_switch(self.status, ctt.Status.DELETED):
            return False

        if self.status == ctt.Status.RELEASED and not Task.request:
            raise common_errors.AuthorizationError(
                "Released task #{} can only be deleted within authorized session scope.".format(self.id)
            )

        session = mapping.OAuthCache.objects(task_id=self.id).first()
        if session:
            user_controller.OAuthCache.abort(session, reason=ctt.Status.DELETED)

        # Drop all wait triggers, don't need them in DELETED
        trigger_controller.delete_wait_triggers(self.id)

        self.before_task_delete(self.id)
        self.set_status(new_status=ctt.Status.DELETED, event=event, force=True)

        self._model.lock_host = ""
        self._model.hidden = True
        self._model.execution.disk_usage.last = 0
        self.save()
        return True

    def suspend(self, event=None):
        self._check_permission("suspend")
        self.set_status(
            new_status=ctt.Status.SUSPENDING,
            event="Suspend self" if Task.request.session else event,
        )
        if Task.request.session:
            self._logger.info("Ensure task #%s author will receive appropriate notification.", self.id)
            notifications = mapping.Task.objects.scalar("notifications").with_id(self.id)
            if not any(_ for _ in notifications if ctt.Status.SUSPENDED in _.statuses):
                trigger_controller.TaskStatusNotifierTrigger.append(
                    self.id,
                    notification_controller.Notification.notification(
                        ctn.Transport.EMAIL,
                        ctt.Status.SUSPENDED,
                        [self._model.owner if Task.request.user.robot else self._model.author]
                    )
                )
        return True

    def resume(self, event=None):
        self._check_permission("resume")
        if self._model.execution.status != ctt.Status.SUSPENDED:
            raise common_errors.IncorrectStatus("Cannot resume not suspended task #{}".format(self.id))
        self.set_status(
            new_status=ctt.Status.RESUMING,
            event=event,
        )
        return True

    def expire(self, event=None):
        self._check_permission("expire")
        if self._model.execution.status in it.chain(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH):
            return False

        session = mapping.OAuthCache.objects(task_id=self.id).first()
        if session:
            user_controller.OAuthCache.abort(session, reason=ctt.Status.EXPIRED)

        self.set_status(
            new_status=ctt.Status.EXPIRED,
            event=event,
            force=True
        )

        if session:
            self.invalidate_resources(with_hosts=True)

        return True


class TBWrapper(NewTaskWrapper):
    @common_patterns.singleton_property
    def cls(self):
        return None

    def _create_task_bridge(self, model):
        request_id = None if Task.request is None else Task.request.id[:8]
        return model_bridge.RemoteTaskBridge(model, request_id)

    def update_model(self, parameters=True):
        """
        Taskbox returns already updated model with all internals filled.
        So there is no need to update model using sdk2 task.
        """

    @property
    def sdk_type(self):
        return ctt.SDKType.TASKBOX

    def _update_notifications(self):
        # TODO: update notifications without notification manager
        pass

    def _on_save_impl(self):
        self._tb.on_save()
        self._tasks.pop(self._model.id, None)  # TODO: to be deleted

    def _on_enqueue_impl(self):
        self._tb.on_enqueue()
        self._update_notifications()

    def _parameters_values_getter(self, mode):
        is_input = mode == "input"
        parameters_dicts = self.parameters_dicts(input=is_input, output=not is_input)
        return parameters_dicts[0] if is_input else parameters_dicts[1]

    @property
    def release_to(self):
        return self.parameters_dicts()[0].get("release_to", [])

    @property
    def input_parameters(self):
        parameters_meta = self.parameters_meta
        return list(pm for pm in parameters_meta.params if not pm.output)

    @property
    def output_parameters(self):
        parameters_meta = self.parameters_meta
        return list(pm for pm in parameters_meta.params if pm.output)

    def update_parameters_from_source(self, source_task, source_task_parameters):
        return (
            (p.name, model_update.encode_parameter(source_task_parameters[p.name], p.complex))
            for p in source_task.input_parameters if not p.do_not_copy and p.name in source_task_parameters
        )

    def update_hints(self):
        if self.model.requirements.tasks_resource.age >= 6:
            self._tb.update_hints()

    def dependent_resources(self):
        if self.model.requirements.tasks_resource.age >= 3:
            return self._tb.dependent_resources()
        else:
            return model_bridge.LocalTaskBridge.dependent_resources_from_task(self._task())

    @common_patterns.singleton_property
    def needs_container(self):
        return Task.needs_container(self.model)

    def save_trace(self, exc=None, save=True, update_gsid=True):
        if getattr(exc, "propagate", False):
            self.update_context({"__last_error_trace": "".join(exc.tb)}, save=save, update_gsid=update_gsid)
            exc.args = ("Server error: {}. Trace: {}".format(str(exc), "".join(exc.tb)),)
        else:
            super(TBWrapper, self).save_trace(exc=exc, save=save, update_gsid=update_gsid)


class OldTaskWrapper(TaskWrapper):

    @property
    def sdk_version(self):
        return 1

    @common_patterns.singleton_property
    def cls(self):
        import sandbox.yasandbox.proxy.task
        return sandbox.yasandbox.proxy.task.getTaskClass(self._model.type)

    @dispatch.local
    def create(self):
        return super(OldTaskWrapper, self).create()

    @property
    def sdk_type(self):
        return ctt.SDKType.SDK1

    @property
    def kill_timeout(self):
        try:
            return min(int(self.input_parameters_values.get("kill_timeout")), 604800)
        except (TypeError, ValueError):
            return common_config.Registry().common.task.execution.timeout * 3600

    @property
    def fail_on_any_error(self):
        return bool(self.input_parameters_values.get("fail_on_any_error"))

    @property
    def dump_disk_usage(self):
        return bool(self.input_parameters_values.get("dump_disk_usage", True))

    @property
    def tcpdump_args(self):
        return self._task().tcpdump_args

    @property
    def tags(self):
        return list(common_itertools.chain(self._task().tags))

    @tags.setter
    def tags(self, value):
        model_update.ServerSideUpdate.update_tags(value, self._task())

    def init_model(self):
        cls = self.cls
        if not cls:
            raise common_errors.UnknownTaskType("Unknown task type <{}>".format(self._model.type))

        self._model.owner = self._model.owner or ""
        self._model.max_restarts = self._model.max_restarts or cls.max_restarts
        self._model.suspend_on_status = self._model.suspend_on_status or cls.suspend_on_status
        self._model.score = self._model.score or cls.score

        self._model.requirements = self._model.requirements or mapping.Task.Requirements()
        self._model.requirements.host = self._model.requirements.host or ""
        self._model.requirements.disk_space = self._model.requirements.disk_space or cls.execution_space
        self._model.requirements.ram = self._model.requirements.ram or cls.required_ram
        self._model.requirements.ramdrive = self._model.requirements.ramdrive or (
            mapping.Task.Requirements.RamDrive(
                type=cls.ramdrive.type,
                size=cls.ramdrive.size
            )
            if cls.ramdrive else
            None
        )
        self._model.requirements.client_tags = self._model.requirements.client_tags or str(cls.client_tags)
        self._model.requirements.caches = []
        self._model.requirements.cores = self._model.requirements.cores or cls.cores

        super(OldTaskWrapper, self).init_model()
        self._task().init_context()

    def _task(self, model=None):
        if model is None:
            model = self._model
        task = self._tasks.get(model.id, ctm.NotExists)
        if task is ctm.NotExists:
            import sandbox.yasandbox.proxy.task
            task = self._tasks[model.id] = sandbox.yasandbox.proxy.task.Task.restore(model, cls=self.cls)
        return task

    def _on_enqueue_impl(self):
        task = self._task()
        task.on_enqueue()
        task.check_ctx(task.ctx)

    def save(self, save_condition=None):
        self._task().save(save_condition=save_condition)
        return self

    def update_context(self, context, remove_fields=None, save=True):
        task = self._task()
        for field in (remove_fields or []):
            task.ctx.pop(field, None)
        if context:
            if "GSID" in context:
                self._update_task_gsid(context)
            task.ctx.update(context)
        self._model.context = str(cPickle.dumps(task.ctx))
        if save:
            task.save()

    @dispatch.local
    def copy(self, source):
        task = self._task()
        source_task = self._task(source)
        task.init_as_copy(source_task, request=Task.request)
        task.save()
        # init_as_copy() puts notifications directly to DB without updating the model
        self._model.reload("notifications")
        return self

    def _update_context_in_model(self):
        self.update_context({}, save=False)

    @property
    def short_task_result(self):
        return self._task().get_short_task_result()

    @property
    def long_task_result(self):
        # noinspection PyProtectedMember
        return self._task()._view_long_result()

    @dispatch.local
    def release_template(self):
        return dict(self._task().release_template)

    @property
    def reports(self):
        # stub for SDK1
        return [
            mapping.Template.ReportInfo(title="Footer", label="footer"),
        ]

    @dispatch.local
    def report(self, label):
        if label != "footer":
            raise tb_errors.UnknownReportName("For SDK1 tasks, only 'footer' report is supported")
        return self.footer

    @property
    def input_parameters(self):
        return self.cls.input_parameters

    @common_patterns.singleton_property
    def parameters_meta(self):
        return model_update.LocalUpdate.parameters_meta(self.cls.input_parameters)

    def _input_parameters_raw_values(self):
        return self._task().ctx

    def _parameters_values_getter(self, mode):
        return self._task().ctx if mode == "input" else {}

    def _parameters_values_setter(self, mode, value):
        if mode == "input":
            self._task().ctx = value

    @dispatch.local
    def update_parameters(self, input_updates, output_updates, requirements_updates, validate_only):
        input_dict = self._task().ctx
        update_result = model_bridge.LocalTaskBridge.update_parameters_impl(
            self.cls.input_parameters, input_updates, {}, input_dict, {}, requirements_updates
        )
        if not validate_only and not update_result.required and update_result.input_updated:
            self._task().ctx = input_dict
        return update_result

    def update_common_fields(self, data):
        from sandbox.sdk2 import legacy
        task = self._task()
        requirements = data.get("requirements", {})
        for checker, getter, setter in it.chain(
            (
                (lambda: n in data, lambda: f(data.get(n)), ft.partial(op.setitem, task.ctx, n))  # noqa
                for n, f in (
                    ("kill_timeout", common_data.force_int),
                    ("do_not_restart", bool),
                    ("fail_on_any_error", bool),
                    ("tasks_archive_resource", lambda _: common_data.force_int(_) if _ else None),
                    ("dump_disk_usage", bool),
                )
            ),
            (
                (lambda: dn in data, lambda: f(dn), ft.partial(setattr, task, tn))  # noqa
                for dn, tn, f in (  # noqa
                    ("description", "descr", data.get),
                    ("tcpdump_args", "tcpdump_args", data.get),
                    ("important", "important", lambda _: bool(data.get(_))),
                    ("hidden", "hidden", lambda _: bool(data.get(_))),
                    ("se_tag", "se_tag", data.get),
                    ("owner", "owner", data.get),
                    ("suspend_on_status", "suspend_on_status", data.get),
                    ("max_restarts", "max_restarts", lambda _: common_data.force_int(data.get(_))),
                    ("push_tasks_resource", "push_tasks_resource", lambda _: bool(data.get(_))),
                )
            ),
            (
                (
                    lambda: rn in requirements,  # noqa
                    lambda: f(requirements.get(rn)),  # noqa
                    ft.partial(setattr, task, tn)  # noqa
                )
                for rn, tn, f in (
                    ("disk_space", "execution_space", lambda _: common_data.force_int(_) >> 20),
                    ("ram", "required_ram", lambda _: common_data.force_int(_) >> 20),
                    ("platform", "arch", lambda _: (_ or "").lower()),
                    ("host", "required_host", lambda _: (_ or "").lower()),
                    ("cores", "cores", common_data.force_int),
                    ("dns", "dns", lambda _: ctm.DnsType.val2str(_) and _),
                    ("cpu_model", "model", lambda _: (_ or "").lower()),
                )
            ),
            (
                (
                    lambda: "tags" in data,
                    lambda: list(common_itertools.chain(data.get("tags"))),
                    lambda tags: setattr(self, "tags", tags),
                ),
                (
                    lambda: data.get("score") is not None,
                    lambda: data.get("score"),
                    lambda score: setattr(task, "score", score)
                )
            )
        ):
            if checker():
                setter(getter())

        container_resource = requirements.get("container_resource", ctm.NotExists)
        if container_resource is not ctm.NotExists:
            task.mapping().requirements.container_resource = common_data.force_int(container_resource, default=None)
            # update container in parameters [SANDBOX-7051]
            for param in task.input_parameters:
                if issubclass(param, legacy.Container):
                    task.ctx[param.name] = container_resource
                    break

        priority = data.get("priority")
        if priority:
            allowed = user_controller.Group.allowed_priority(Task.request, task.owner)
            task.priority = ctt.Priority.make(priority)
            if task.priority > allowed:
                self._logger.warning(
                    "Lower new task %s priority from %s to %s", task.type, task.priority, allowed
                )
                task.priority = allowed
        self._model = task.mapping()

        if "hints" in data:
            model_update.LocalUpdate.set_explicit_hints(common_itertools.chain(data.get("hints")), self._model)
            self._model.hints = self._model.explicit_hints  # for sdk1 all hints are explicit

    @property
    def type_client_tags(self):
        return self.cls.client_tags

    @property
    def client_tags(self):
        tags = self._model.requirements.client_tags or self.type_client_tags
        return ctc.Tag.Query(tags)

    @property
    def release_to(self):
        return self._task().release_to

    def set_info(self, info, do_escape=True):
        return self._task().set_info(info, do_escape=do_escape)

    @property
    def tasks_archive_resource(self):
        return self._task().tasks_archive_resource

    def on_enqueue(self):
        """
        The method should be executed only once per particular task's life on first task enqueuing.
        """
        if self._model.execution.status == ctt.Status.DRAFT:
            return super(OldTaskWrapper, self).on_enqueue()
        return EnqueueResult(ctt.Status.ENQUEUING)

    def restart(self, event=None):
        return self._task().restart(event, Task.request)

    def stop(self, event=None):
        return self._task().stop(event, Task.request)

    def delete(self, event=None):
        return self._task().delete(event, Task.request)

    def suspend(self, event=None):
        import sandbox.yasandbox.proxy.task
        return sandbox.yasandbox.proxy.task.Task.suspend(self._task(), event, Task.request)

    def resume(self, event=None):
        return self._task().resume(event, Task.request)


class Task(object):
    Model = mapping.Task
    AUDIT_CONTENT_LIMIT = 1024

    class Exception(Exception):
        pass

    class NotExists(Exception):
        pass

    @classmethod
    def is_taskboxed_task(cls, model):
        return (
            model.requirements and
            model.requirements.tasks_resource and
            model.requirements.tasks_resource.taskbox_enabled
        )

    @classmethod
    def initialize(cls):
        cls.Model.ensure_indexes()
        mapping.Audit.ensure_indexes()
        mapping.TaskTagCache.ensure_indexes()
        mapping.ParametersMeta.ensure_indexes()

    @classmethod
    def get(cls, task_id, lite=False):
        """
        get Task model by id

        :param task_id: task id
        :return model: Task model
        """
        # noinspection PyUnresolvedReferences
        query = cls.Model.objects
        if lite:
            query = query.lite()
        model = query.with_id(task_id)
        if not model:
            raise cls.NotExists("Task #{} does not exist".format(task_id))
        return model

    @classmethod
    def save(cls, model, save_condition=None, logger=None):
        # type: (mapping.Task, dict, logging.Logger) -> mapping.Task
        """ Save task model without status and lock """

        logger = logger or logging.getLogger("task_save")
        oper, kws = "Updating", {"save_condition": save_condition}
        if model.requirements.tasks_resource == 0:
            raise TypeError("Unknown tasks resource #{}".format(model.requirements.tasks_resource))
        if model.is_new:
            oper = "Creating"
            kws = {
                "force_insert": True,
                "write_concern": common_config.Registry().server.mongodb.write_concern
            }
        else:
            # Avoid RC (SANDBOX-5288)

            # Sometimes `model` doesn't have `_changed_fields` attribute.
            # In that case there is nothing to do, so `AttributeError`
            # is ignored (SANDBOX-6348).
            try:
                model._changed_fields.remove("lock")
            except (ValueError, AttributeError):
                pass
            try:
                model.execution._changed_fields.remove("st")
            except ValueError:
                pass
        logger.info(
            "%s task #%s of type %r, status: %r, host: %r/%r, req: %s",
            oper,
            model.id,
            model.type,
            model.execution.status,
            model.requirements.host,
            model.lock_host,
            model.requirements.to_json(),
        )
        model.save(**kws)
        return model

    @classmethod
    def dependent(cls, task_id):
        return Task.Model.objects(
            requirements__resources__in=mapping.Resource.objects(task_id=task_id).fast_scalar("id")
        )

    @classmethod
    def notifications(cls, data):
        for item in data or ():
            if item.get("transport", "") == ctn.Transport.JUGGLER and item.get("check_status") not in ctn.JugglerStatus:
                raise ValueError("check_status is required for juggler notification. Notification: {}".format(item))
        return filter(None, (
            notification_controller.Notification.notification(
                item.get("transport", ""),
                ctt.Status.Group.expand(item.get("statuses", ())),
                item.get("recipients", ()),
                check_status=item.get("check_status"),
                juggler_tags=item.get("juggler_tags", ())
            )
            for item in data or ()
        ))

    @classmethod
    def iteration(cls, task):
        return mapping.Resource.objects(task_id=task.id, type="TASK_LOGS").count()

    @classmethod
    def client_tags(cls, model):
        tags = model.requirements.client_tags or ctc.Tag.GENERIC
        return ctc.Tag.Query(tags)

    @classmethod
    def is_sdk1_type_by_classes(cls, task_meta):
        return task_meta is not None and task_meta.sdk_type == ctt.SDKType.SDK1

    @classmethod
    def is_sdk1_type(cls, task, task_meta):
        return task.parameters_meta is None and cls.is_sdk1_type_by_classes(task_meta)

    @classmethod
    def input_parameters_raw_values(cls, task, task_meta):
        if cls.is_sdk1_type(task, task_meta):
            return task.ctx
        return {p.key: p.value for p in (task.parameters.input or []) if p.value is not None}

    @classmethod
    def output_parameters_raw_values(cls, task, task_meta):
        if cls.is_sdk1_type(task, task_meta):
            return {}
        return {p.key: p.value for p in (task.parameters.output or []) if p.value is not None}

    @classmethod
    def parameters_dicts(cls, task, parameters_meta=None, input=True, output=True, task_meta=None):
        """
        Returns dicts of task input and output parameters.

        :type parameters_meta: mapping.ParametersMeta
        """
        if parameters_meta is None:
            parameters_meta = task.parameters_meta

        if parameters_meta is None and task_meta is None:
            return []

        in_params_dict, out_params_dict = {}, {}
        in_values = cls.input_parameters_raw_values(task, task_meta) if input else {}
        out_values = cls.output_parameters_raw_values(task, task_meta) if output else {}

        params = task_meta.parameters if parameters_meta is None else parameters_meta.params

        for pm in params:
            if (pm.output and not output) or (not pm.output and not input):
                continue
            values = out_values if pm.output else in_values
            if pm.output and pm.name not in values:
                continue
            value = model_update.decode_parameter(values.get(pm.name, pm.default), pm.complex)
            (out_params_dict if pm.output else in_params_dict)[pm.name] = value
        return in_params_dict, out_params_dict

    @classmethod
    def invalidate_resources(cls, task, with_hosts=False):
        """
        Mark all `NOT_READY` resources as broken.

        :param bool with_hosts: Invalidate only resources with hosts
        """
        from . import resource as resource_controller
        resources = resource_controller.Resource.list_task_resources(task.id)
        for resource in resources:
            if resource.state == ctr.State.NOT_READY:
                if with_hosts and not resource_controller.Resource.get_hosts(resource, all=True):
                    # There are no hosts with this resource, don't invalidate it
                    continue
                resource.state = ctr.State.BROKEN
                resource.time.accessed = dt.datetime.utcnow()
                resource.time.updated = dt.datetime.utcnow()
                resource.save()

    @classmethod
    def _set_status_audit(cls, task, old_status, new_status, event, wait_targets=None):
        audit = Task.audit(mapping.Audit(
            task_id=task.id,
            status=new_status,
            content=event,
            client=Task.request and Task.request.session and Task.request.session.client,
            wait_targets=wait_targets
        ))

        if new_status in (ctt.Status.SUSPENDING, ctt.Status.SUSPENDED):
            old_state = ctt.SessionState.ACTIVE
            new_state = ctt.SessionState.SUSPENDED
        else:
            old_state = ctt.SessionState.SUSPENDED
            new_state = ctt.SessionState.ACTIVE

        # Switch from ACTIVE to SUSPENDED or vice versa
        mapping.OAuthCache.objects(task_id=task.id, state=str(old_state)).update(set__state=str(new_state))

        return audit

    @classmethod
    def set_info(cls, task, info, do_escape=True, logger=None):
        # type: (mapping.Task, str, bool, logging.Logger) -> str
        """ Set info for task """
        logger = logger or logging.getLogger("set_info")
        logger.info("Task #%s info added: %s", task.id, info)
        info = common_encoding.force_unicode_safe(info)

        if not task.execution.description:
            task.execution.description = u""
        if do_escape:
            info = common_encoding.escape(info)
        task.execution.description += u"<div class=\"hr\">{}</div>{}\n".format(
            common_format.dt2str(dt.datetime.now()), info
        )
        return task.execution.description

    @classmethod
    def set_status(
        cls, task, new_status, event=None, lock_host="", keep_lock=None, force=False,
        expected_status=None, reset_resources_on_failure=False, wait_targets=None, timer=None, logger=None,
        set_last_action_user=False, host=None
    ):
        """
        Set status to new_status

        :param task: Task model object
        :param new_status: new task status
        :param event: event that cause status changing
        :param lock_host: new lock value
        :param keep_lock: forces lock keep - do not update a record, if the lock is not the same
        :param force: check the new status is applicable
        :param expected_status: current task status, use to prevent conflict
        :param reset_resources_on_failure: mark resources as BROKEN if task goes to FAILURE due to `fail_on_any_error`
        :param wait_targets: targets for waiting statuses
        :param logger: logger for this method
        :param timer: Timer object to trace of executing time of request
        :param set_last_action_user: if True set last_action_user of task
        :param host: host name for task execution
        """
        if logger is None:
            logger = logging.getLogger("task_controller")
        if timer is None:
            timer = common_context.Timer().start()
        status = task.execution.status

        fail_on_any_error = (
            new_status in ctt.Status.Group.FAIL_ON_ANY_ERROR and
            (task.fail_on_any_error or task.ctx.get("fail_on_any_error"))
        )

        final_status = new_status
        if fail_on_any_error:
            event = event + ". " if event else ""
            event += "Switched to {} instead of {}.".format(ctt.Status.FAILURE, new_status)
            final_status = ctt.Status.FAILURE

        if wait_targets is not None:
            logging.info("Wait targets: %r.", wait_targets)

        with timer["finishing_audit"]:
            if status == final_status:
                if (
                    Task.request and Task.request.id and
                    not mapping.Audit.objects(task_id=task.id, status=status, request_id=Task.request.id)
                ):
                    logger.warning("Finishing audit for task #%s status %s", task.id, final_status)
                    return cls._set_status_audit(task, status, final_status, event, wait_targets=wait_targets)
                return None

        if expected_status and status != expected_status:
            raise common_errors.UpdateConflict(
                "Expected task status: {}, but current is {}".format(expected_status, status)
            )

        if not ctt.Status.can_switch(status, new_status) and not force:
            raise common_errors.IncorrectStatus(
                "Cannot switch task #{} status from {} to {}".format(task.id, status, new_status)
            )
        log_suffix = "locking it for {!r}".format(lock_host) if lock_host else "resetting the lock"
        logger.info("Switch task #%s status from %r to %r and %s", task.id, status, new_status, log_suffix)
        now = dt.datetime.utcnow()
        keep_lock = keep_lock or task.lock_host
        kws = dict(
            set__execution__status=new_status,
            set__lock_host=lock_host or "",
            set__time__updated=now,
            write_concern=common_config.Registry().server.mongodb.write_concern,
        )
        if lock_host:
            kws["set__lock_time"] = now

        if new_status in ctt.Status.Group.REALEXECUTE and Task.request.session:
            kws.update(set__execution__host=Task.request.session.client)
        if new_status == ctt.Status.PREPARING:
            kws.update(set__execution__time__started=now, set__execution__time__finished=now)

        if fail_on_any_error:
            new_status = ctt.Status.FAILURE
            kws.update(set__execution__status=new_status)

            if reset_resources_on_failure:
                logger.warning("Invalidating not ready resources for task #%s", task.id)
                with timer["invalidate_resources"]:
                    cls.invalidate_resources(task)

        if new_status in it.chain(
            (ctt.Status.TEMPORARY, ctt.Status.SUCCESS, ctt.Status.FAILURE),
            ctt.Status.Group.WAIT, ctt.Status.Group.BREAK,
        ):
            # TODO: execution.time.started/finished is deprecated, see SANDBOX-3069
            if task.execution.time.started and (
                task.execution.time.started > dt.datetime.utcfromtimestamp(0)
            ):
                kws.update(set__execution__time__finished=now)
            kws.update(Task.close_all_intervals(task))

        if new_status in ctt.Status.Group.WAIT:
            kws.update(Task.open_interval(task, ctt.IntervalType.WAIT))
        elif new_status == ctt.Status.ENQUEUED:
            kws.update(Task.open_interval(task, ctt.IntervalType.QUEUE))
        elif new_status in ctt.Status.ASSIGNED:
            kws.update(Task.open_interval(task, ctt.IntervalType.EXECUTE, host))

        if set_last_action_user:
            if not cls.request or user_controller.user_has_right_to_act_on_behalf_of(cls.request.user, task.author):
                if task.last_action_user is None:
                    kws.update(set__last_action_user=task.author)
            elif cls.request.session:
                kws.update(
                    set__last_action_user=mapping.Task.objects(
                        id=cls.request.session.task
                    ).fast_scalar("last_action_user").first()
                )
            else:
                kws.update(set__last_action_user=cls.request.user.login)

        with timer["task_status_event_create"]:
            task_status_events_ids = tse_controller.TaskStatusEvent.create(task, new_status)
        kws.update(set__status_events=task_status_events_ids)

        with timer["status_update"]:
            updated = mapping.retry_temporary_errors(
                lambda: mapping.Task.objects(
                    id=task.id, execution__status=status, lock_host=keep_lock,
                ).update_one(**kws),
                lambda ex: logger.warning(
                    "Failed to update task #%s to status %s, retrying: %s", task.id, new_status, ex
                ),
                timeout=TaskWrapper.SET_STATUS_TIMEOUT
            )

        if updated:
            if task.acquired_semaphore and task.requirements.semaphores and (
                new_status in ctt.Status.Group.expand(task.requirements.semaphores.release or ())
            ):
                exc = None
                # noinspection PyBroadException
                try:
                    with timer["release_semaphores"]:
                        semaphores_released = TaskQueue.qclient.release_semaphores(task.id, status, new_status)
                except Exception as exc:
                    semaphores_released = None
                if not semaphores_released:
                    err_msg = "Cannot release semaphore(s) for task #{} ({} -> {})".format(task.id, status, new_status)
                    if semaphores_released is None:
                        err_msg = "{}: {}".format(err_msg, exc)
                    logger.error(err_msg)
        else:
            with timer["reload_if_not_updated"]:
                task.reload()
                if task.execution.status == new_status:
                    audit = mapping.Audit.objects(task_id=task.id, status=new_status).order_by("-id").limit(1).first()
                    if audit:
                        return audit
                else:
                    raise common_errors.UpdateConflict(
                        "Cannot switch task #{} status from {} to {}: conflict on update. "
                        "Current task status: {}, lock '{}'/'{}'".format(
                            task.id, status, new_status, task.execution.status, keep_lock, task.lock_host
                        )
                    )

        with timer["set_status_audit"]:
            audit = cls._set_status_audit(task, status, new_status, event, wait_targets=wait_targets)

        task.execution.status = new_status
        task.lock_host = lock_host or ""

        return audit

    @classmethod
    def setup_notifications(cls, model, notifications, new=False):
        if new:
            if model.notifications:
                return
            notifications = cls.notifications([{
                "statuses": [
                    ctt.Status.SUCCESS, ctt.Status.FAILURE,
                    ctt.Status.EXCEPTION, ctt.Status.NO_RES,
                    ctt.Status.TIMEOUT, ctt.Status.EXPIRED,
                ],
                "transport": "email",
                "recipients": [model.author]
            }])
        else:
            for item in notifications or ():
                if item.get("transport", "") == ctn.Transport.JUGGLER:
                    for recipient in item.get("recipients", ()):
                        if mapping.Group.objects(name=recipient).count() and recipient != model.owner:
                            raise ValueError("Notification owner must be equal to task owner")
            notifications = cls.notifications(notifications)
        cls.Model.objects(id=model.id).update_one(set__notifications=notifications)
        model.notifications = notifications

    @common_patterns.classproperty
    def request(self):
        return getattr(mapping.base.tls, "request", None)

    @classmethod
    def set_request(cls, request_obj):
        setattr(mapping.base.tls, "request", request_obj)

    @classmethod
    def audit(cls, audit_item):
        """
        Register action on task

        :param audit_item: a document with at least task_id, event, status fields filled
        :type audit_item: sandbox.yasandbox.database.mapping.Audit
        """

        settings = common_config.Registry()
        audit_item.hostname = settings.this.id

        content = audit_item.content
        if content and len(content) > cls.AUDIT_CONTENT_LIMIT:
            content = common_encoding.force_unicode(content, errors="replace")
            content = content[:cls.AUDIT_CONTENT_LIMIT]
            audit_item.content = "{}... (truncated)".format(content.encode("utf-8"))

        request = cls.request
        if request:
            if not request.user.super_user or not request.user.robot:
                audit_item.author = request.user.login
            audit_item.remote_ip = request.remote_ip
            audit_item.source = request.source
            if request.id:
                audit_item.request_id = request.id
            if (
                request.session and
                (not isinstance(request.session, mapping.OAuthCache or request.session.is_task_session))
            ):
                if request.source == ctt.RequestSource.API:
                    audit_item.source = ctt.RequestSource.TASK

                if request.session.task != audit_item.task_id:
                    suffix = "(by #{})".format(request.session.task)
                    audit_item.content = " ".join((audit_item.content, suffix)) if audit_item.content else suffix

        elif "SSH_CLIENT" in os.environ:
            audit_item.remote_ip = (os.environ.get("SSH_CLIENT") or "").split()[0]
        audit_item.save()

        if audit_item.status:
            common_statistics.Signaler().push(dict(
                type=cts.SignalType.AUDIT,
                timestamp=audit_item.date,  # UTC
                # despite the fact statistics_processor records previous status,
                # send the actual one in case there is no previous audit entry in MongoDB
                status=audit_item.status,
                task_id=audit_item.task_id,
            ))

        return audit_item

    @classmethod
    def reset_semaphores(cls, model=None, task_id=None, logger=None):
        assert model or task_id, "model or task_id must be set"
        logger = logger or logging
        if not model:
            model = cls.get(task_id)
        logger.warning("Release acquired semaphore(s) of task #%s due reseting lock_host", model.id)
        TaskQueue.qclient.release_semaphores(
            model.id, model.execution.status, model.execution.status
        )

    @classmethod
    def create_service_task(cls, task_type, description, parameters, requirements, **kws):
        client = dispatch.RestClient(None, author=user_controller.User.service_user, jailed=False)()
        payload = dict(
            type=task_type,
            author=user_controller.User.service_user,
            owner=user_controller.Group.service_group,
            description=description,
            hidden=True,
            notifications=[],
            requirements=requirements,
            custom_fields=[{"name": key, "value": value} for key, value in parameters.iteritems()],
        )
        payload.update(**kws)
        task_info = client.task(**payload)
        client.batch.tasks.start = [task_info["id"]]
        return task_info["id"]

    @classmethod
    def open_interval(cls, model, interval_type, host=None):
        if not model.execution.intervals:
            model.execution.intervals = mapping.Task.Execution.Intervals()
        now = dt.datetime.utcnow().replace(microsecond=0)
        kws = cls.close_all_intervals(model, interval_type)
        if interval_type == ctt.IntervalType.EXECUTE and host:
            pool_index = TaskQueue.quota_pools.match_pool(mapping.Client.objects.with_id(host).tags)
            pool = TaskQueue.quota_pools.pools[pool_index] if pool_index is not None else None
        else:
            pool = None
        interval = mapping.Task.Execution.Intervals.IntervalData(start=now, pool=pool)
        interval_list = getattr(model.execution.intervals, interval_type) or []
        interval_list.append(interval)
        setattr(model.execution.intervals, interval_type, interval_list)
        if len(interval_list) > 1:
            kw = "set__execution__intervals__{}__{}".format(interval_type, len(interval_list) - 1)
        else:
            kw = "push__execution__intervals__{}".format(interval_type)
        kws[kw] = interval
        return kws

    @classmethod
    def find_open_interval(cls, model, interval_type):
        if not model.execution.intervals:
            model.execution.intervals = mapping.Task.Execution.Intervals()
        interval_list = getattr(model.execution.intervals, interval_type)
        if not interval_list:
            return None
        interval = interval_list[-1]
        if interval.duration is not None:
            return None
        return len(interval_list) - 1, interval

    @classmethod
    def close_all_intervals(
        cls, model, interval_type=ctt.IntervalType.WAIT,
        update=False, consumption=None
    ):
        to_close = []
        for t in ctt.IntervalType.close_order(interval_type):
            found = cls.find_open_interval(model, t)
            if found:
                idx, interval = found
                if to_close and interval.start > to_close[-1][2].start:
                    continue
                to_close.append((t, idx, interval))

        end = dt.datetime.utcnow().replace(microsecond=0)
        kws = {}
        for interval_type, interval_idx, interval in to_close:
            # execute intervals are only closed when the task session is finished
            if interval_type != ctt.IntervalType.EXECUTE or update:
                interval.duration = int((end - interval.start).total_seconds())
                if consumption is not None:
                    interval.consumption = int(consumption)
                kws[
                    "set__execution__intervals__{}__{}".format(interval_type, interval_idx)
                ] = mapping.Task.Execution.Intervals.IntervalData.from_mongo(interval.to_mongo())
            end = interval.start

        if update and kws:
            return cls.Model.objects(id=model.id).update_one(**kws)
        return kws

    @classmethod
    def update_tasks_resource(cls, data, model):
        if not data:
            return False
        tasks_resource = ctm.NotExists
        if "tasks_resource" in data.get("requirements", {}):
            tasks_resource = data["requirements"]["tasks_resource"]
        if "tasks_archive_resource" in data and not tasks_resource:
            tasks_resource = data["tasks_archive_resource"]
        if tasks_resource is not ctm.NotExists:
            model_update.ServerSideUpdate.update_tasks_resource(tasks_resource, model)
            return True
        return False

    @classmethod
    def kill_timeout(cls, task):
        default_timeout = common_config.Registry().common.task.execution.timeout
        return min(task.kill_timeout or default_timeout * 3600, 604800)

    @classmethod
    def tasks_archive_resource(cls, task):
        return (
            task.requirements.tasks_resource and task.requirements.tasks_resource.id or
            task.tasks_archive_resource
        )

    @classmethod
    def get_porto_container_properties(cls, task):
        # First layer is base layer and should have `platform` attribute
        porto_layers = list(task.requirements.porto_layers)
        base_layer_id = porto_layers[0]

        resource = mapping.Resource.objects.lite().with_id(base_layer_id)
        if resource is None:
            raise common_errors.ResourceNotFound

        platform = resource.attributes_dict().get("platform", None)
        if platform:
            platform = common_platform.get_platform_alias(platform)

        return porto_layers, platform

    @classmethod
    def get_container_platform_impl(cls, cnt):
        from . import resource as resource_controller

        res = mapping.Resource.objects.fast_scalar("id", "attributes").with_id(cnt)
        if res is None:
            raise common_errors.ResourceNotFound

        resource_id, attributes = res

        platform = None
        for attr in (attributes or []):
            if attr.get("k") == "platform":
                platform = attr.get("v")
                break

        venv_res = None
        if platform and common_config.Registry().common.installation == ctm.Installation.LOCAL:
            venv_res = resource_controller.Resource.venv_resource(common_platform.get_platform_alias(platform))
        return resource_id, platform, venv_res

    @classmethod
    def get_container_platform(cls, task):
        cnt = cls.needs_container(task)
        return cls.get_container_platform_impl(cnt)

    @classmethod
    def _cores_tags(cls, cores):
        tags = []
        for fixed_cores, tag in (
            (80, ctc.Tag.CORES80),
            (64, ctc.Tag.CORES64),
            (56, ctc.Tag.CORES56),
            (32, ctc.Tag.CORES32),
            (24, ctc.Tag.CORES24),
            (16, ctc.Tag.CORES16),
        ):
            if cores > fixed_cores:
                break
            tags.append(tag)
        tags.reverse()
        return tags

    @classmethod
    def _concatenate_tags(cls, tags, additional_tags):
        if tags is None:
            return additional_tags
        return tags | additional_tags

    @classmethod
    def effective_client_tags_impl(cls, task, client_tags):
        cores = task.requirements.cores
        ram = task.requirements.ram
        bound_to_multislots = False
        if task.requirements.porto_layers:
            new_client_tags = None
            for con in client_tags:
                if ctc.Tag.LXC in con:
                    con = ft.reduce(op.and_, filter(lambda tag: tag != ctc.Tag.LXC, con), ctc.Tag.LXC | ctc.Tag.PORTOD)
                else:
                    con = ft.reduce(op.and_, con, ctc.Tag.PORTOD)
                new_client_tags = cls._concatenate_tags(new_client_tags, con)
            client_tags = new_client_tags
        if task.requirements.privileged:
            new_client_tags = None
            for con in client_tags:
                if ctc.Tag.PORTOD in con or ctc.Tag.LXC in con:
                    con = ft.reduce(op.and_, con, ~ctc.Tag.MULTISLOT)
                else:
                    con = ft.reduce(op.and_, con, ctc.Tag.LXC & ~ctc.Tag.MULTISLOT)
                new_client_tags = cls._concatenate_tags(new_client_tags, con)
            client_tags = new_client_tags
        elif cores and ram and (task.requirements.caches == []) and cores <= 16 and ram <= 65536:
            bound_to_multislots = True
            client_tags &= ctc.Tag.MULTISLOT | ~ctc.Tag.Group.LINUX
        elif (
            cores and ram and (task.requirements.caches == []) and cores <= 32 and ram <= 131072 and
            any(ctc.Tag.BROWSER in con for con in client_tags)
        ):
            new_client_tags = None
            new_client_tags_browser = None
            for con in client_tags:
                if ctc.Tag.BROWSER in con:
                    new_client_tags_browser = cls._concatenate_tags(new_client_tags_browser, ft.reduce(op.and_, con))
                else:
                    new_client_tags = cls._concatenate_tags(new_client_tags, ft.reduce(op.and_, con))
            client_tags = new_client_tags_browser & (ctc.Tag.MULTISLOT | ~ctc.Tag.Group.LINUX)

            if new_client_tags is not None:
                tags = cls._cores_tags(cores)
                new_client_tags &= ft.reduce(op.or_, tags) | ~ctc.Tag.Group.LINUX
                client_tags |= new_client_tags
            bound_to_multislots = True

        if cores and not bound_to_multislots:
            tags = cls._cores_tags(cores)
            if tags:
                client_tags &= ft.reduce(op.or_, tags) | ~ctc.Tag.Group.LINUX
        return client_tags

    @classmethod
    def effective_client_tags(cls, task):
        tags = getattr(task, "__effective_client_tags__", None)
        if tags is None:
            tags = task.__effective_client_tags__ = cls.effective_client_tags_impl(
                task, ctc.Tag.Query(task.requirements.client_tags)
            )
        return tags

    @classmethod
    def needs_container(cls, model):
        """ Returns specific container id if a task requires it, or True if any container is ok """
        if model.requirements.container_resource:
            return model.requirements.container_resource
        if model.requirements.privileged:
            return True

    @classmethod
    def platform_filter(cls, task):
        platform_filter = task.requirements.platform
        if not platform_filter or platform_filter == ctm.OSFamily.ANY:
            platform_filter = None

        pf = platform_filter.lower() if platform_filter else None

        if task.requirements.platforms:
            result = set(task.requirements.platforms)
            if pf:
                result.add(pf)
            return tuple(sorted(result))

        return pf

    @classmethod
    def check_platform_filter(cls, task, client, cache=None):
        """
        Gets info of filter by platform for the task and given client host

        :param task: Task model object
        :param client: client object
        :param cache: cache by clients
        :return: check result
        """
        pf = cls.platform_filter(task)

        exclusive_host = task.requirements.host
        if exclusive_host:
            allowed = (exclusive_host == client.hostname)
        else:
            match_tags = client_controller.Client.match_tags if cache is None else cache.client_tags.match_tags
            allowed = match_tags(client, cls.effective_client_tags(task), pf) is not None
        return TaskWrapper.FilterCheckResult(
            "platform", task.requirements.platform, client.info["system"].get("platform", ""), allowed
        )

    @classmethod
    def check_model_cpu_filter(cls, task, client):
        """
        Gets info of filter by CPU model for the task and given client host

        :param task: Task model object object
        :param client: client object
        :return: check result
        """
        allowed = True
        cpu_model = task.requirements.cpu_model
        if cpu_model and (cpu_model.lower() not in client.hardware.cpu.model.lower()):
            allowed = False

        cpu_cores = task.requirements.cores
        if (
            cpu_cores and client.hardware.cpu.cores < cpu_cores and
            common_config.Registry().common.installation != ctm.Installation.LOCAL
        ):
            allowed = False
        return TaskWrapper.FilterCheckResult("cpu_model", cpu_model, client.hardware.cpu.model, allowed)

    @classmethod
    def check_ram_filter(cls, task, client):
        """
        Gets info of filter by RAM for the task and given client host

        :param task: Task model object
        :param client: client object
        :return: check result
        """
        required_ram = task.requirements.ram
        allowed = not required_ram or required_ram <= client.hardware.ram
        return TaskWrapper.FilterCheckResult("ram", required_ram, client.hardware.ram, allowed)

    @classmethod
    def check_client_hostname_filter(cls, task, client):
        """
        Gets info of filter by host name for the task and given client host

        :param task: Task model object
        :param client: client object
        :return: check result
        """
        allowed = True
        required_host = task.requirements.host
        if required_host and (required_host.lower() not in client.hostname.lower()):
            allowed = False
        return TaskWrapper.FilterCheckResult("hostname", required_host, client.hostname, allowed)

    @classmethod
    def check_available_space_filter(cls, task, client):
        """
        Gets info of filter by required disk space for the task and given client host

        :param task: Task model object
        :param client: client object
        :return: check result
        """
        client_freespace = int(client.hardware.disk_free)
        execution_space = task.requirements.disk_space
        allowed = execution_space <= client_freespace
        return TaskWrapper.FilterCheckResult("client_free_space", execution_space, client_freespace, allowed)

    @classmethod
    def check_available_ramdrive_filter(cls, task, client):
        """
        Checks whether client is able to create a ramdisk required by a task.

        :param task: Task model object
        :param client:  client object to check
        :return: check result
        """
        try:
            arch = common_platform.get_arch_from_platform(client.info["system"].get("platform", ""))
        except ValueError:
            arch = "unknown"
        ramdrive = task.requirements.ramdrive
        return TaskWrapper.FilterCheckResult(
            "client_ramdrive",
            (ctm.OSFamily.LINUX, ramdrive.size),
            (arch, client.hardware.ram),
            (
                arch in (ctm.OSFamily.LINUX, ctm.OSFamily.LINUX_ARM) and
                ramdrive.size + 1024 < client.hardware.ram and
                client.info["system"].get("root")
            )
        )

    @classmethod
    def check_admin(cls, task, client):
        """
        Check that task has admin privileges
        :param task: Task model object
        :param client: client object
        :return: check result
        """
        return TaskWrapper.FilterCheckResult(
            "admin_privileges", client.hostname, client.hostname,
            user_controller.User.is_super(task.author)
        )

    @classmethod
    def get_client_filters_info(cls, task, client, cache=None, check_platform=True):
        """
        Gets info of all filters for the task and given client host

        :param task: Task model object
        :param client: client object
        :param cache: cache by clients
        :return: list of filter result objects
        :rtype: list of FilterCheckResult
        """
        ret = [
            cls.check_client_hostname_filter(task, client),
            cls.check_model_cpu_filter(task, client),
            cls.check_ram_filter(task, client),
        ]
        if check_platform:
            ret.append(cls.check_platform_filter(task, client, cache=cache))
        if ctc.Tag.SERVER in client.tags_set or ctc.Tag.STORAGE in client.tags_set:
            ret.append(cls.check_admin(task, client))
        if task.requirements.ramdrive:
            ret.append(cls.check_available_ramdrive_filter(task, client))
        if cls.needs_container(task):
            result = (
                client.lxc and not (task.requirements.privileged and ctc.Tag.MULTISLOT in client.tags_set) or
                client.porto
            )
            ret.append(TaskWrapper.FilterCheckResult("container", True, result, result))
        return ret

    @classmethod
    def check_user_permission(cls, task, request):
        """
        Check if user has permission to access the task

        :param task: task object
        :param request: SandboxRequest object
        :return: tuple(bool, string); True if author and owner are valid,
            string contains the invalidation reason
        """
        if not task.owner:
            raise common_errors.TaskError("The required parameter 'owner' cannot be empty")

        if (
            task.owner != task.author and
            # Weired permission for admin to launch tasks with any user-group pair. Should be dropped.
            not (request and request.user.super_user) and
            task.owner not in user_controller.Group.get_user_groups(task.author)
        ):
            return False, "User {!r} does not belong to the group {!r}".format(task.author, task.owner)

        if (
            request and
            not user_controller.user_has_right_to_act_on_behalf_of(request.user, task.author) and
            # TODO: forbid to run tasks on behalf of other users SANDBOX-9507
            not user_controller.user_has_permission(request.user, (task.owner,))
        ):
            return False, "User {!r} is not permitted to start task #{} (author: {!r}, owner: {!r})".format(
                request.user.login, task.id, task.author, task.owner
            )

        if task.owner == task.author:  # group here can be a user
            user_owner = user_controller.User.get(task.owner)
            if user_owner and user_owner.robot and task.owner not in ROBOT_OWNERS_WHITE_LIST:
                return False, "Robot {!r} cannot be owner of the task #{}, use a group instead".format(
                    task.owner, task.id
                )

        return True, ""

    @classmethod
    def remote_http(cls, doc):
        if not doc.execution.host:
            return ""
        http_prefix = cls.get_fileserver_url(doc.execution.host)
        if not http_prefix:
            return ""
        full_url = urlparse.urljoin(http_prefix, os.path.join(*(ctt.relpath(doc.id) + [""])))
        return full_url

    @classmethod
    def get_fileserver_url(cls, host):
        client = mapping.Client.objects(hostname=host).lite().first()
        if client:
            return cPickle.loads(client.context).get("system", {}).get("fileserver", "")
        else:
            return ""

    @classmethod
    def extend_gsid(cls, gsid, task_id, task_type, author):
        return "{}{}SB:{}:{} USER:{}".format(
            gsid,
            " " if gsid else "",
            task_type,
            task_id,
            author,
        )

    @staticmethod
    def list_query(
        parent_id=None, type="", completed_only=False, owner="", status="", host="",
        id=None, hidden=False, show_childs=False, descr_mask="", important_only=False, limit=0,
        offset=0, model="", arch="", requires=None, load_ctx=True, order_by="-id",
        created=None, updated=None, author="", scheduler=None, template_alias=None, se_tag=None, priority=None,
        release=None, input_parameters=None, output_parameters=None, any_params=False, tags=None, all_tags=False,
        hints=None, all_hints=None,
    ):
        query = {}

        if parent_id:
            query["parent_id__in" if isinstance(parent_id, (list, tuple)) else "parent_id"] = parent_id
        if type:
            query["type__in" if isinstance(type, (tuple, list)) else "type"] = type
        if owner:
            query["owner"] = owner
        if author:
            query["author"] = author
        if status:
            if isinstance(status, (tuple, list, set)):
                status = set(status)
                if status != set(ctt.Status.Group.ALL):
                    query["execution__status__in"] = status
            elif status != ctt.Status.Group.ALL:
                query["execution__status"] = status

        if id:
            query["id__in" if isinstance(id, (list, tuple)) else "id"] = id
        else:
            # implicit filters
            if not status and not completed_only:
                query["execution__status__ne"] = ctt.Status.DELETED
            if not hidden:
                query["hidden__ne"] = True
            if not show_childs and not parent_id:
                query["parent_id__exists"] = False

        if host:
            query["execution__host"] = host
        if model:
            query["requirements__cpu_model"] = model
        if arch:
            query["requirements__platform"] = arch
        if requires:
            k = "requirements__resources__in" if isinstance(requires, (tuple, list)) else "requirements__resources"
            query[k] = requires
        if priority:
            query["priority"] = int(ctt.Priority.make(priority))
        if completed_only:
            query["execution__status__in"] = ctt.Status.Group.FINISH
        if se_tag:
            query["se_tag"] = se_tag
        if descr_mask:
            try:
                query["description"] = re.compile(descr_mask, re.IGNORECASE)
            except re.error as error:
                logging.error("Incorrect regex: %s \n %s", descr_mask, error)
        if important_only:
            query["flagged"] = True
        if created:
            query["time__created__gte"], query["time__created__lte"] = created
        if updated:
            query["time__updated__gte"], query["time__updated__lte"] = updated

        if scheduler:
            query["scheduler__in" if isinstance(scheduler, (list, tuple)) else "scheduler"] = scheduler
        if template_alias:
            query[
                "template_alias__in" if isinstance(template_alias, (list, tuple)) else "template_alias"
            ] = template_alias
        if release:
            query["execution__status"] = ctt.Status.RELEASED
            query["release__status__in" if isinstance(release, (list, tuple)) else "release__status"] = release

        if tags:
            tag_op = "__all" if all_tags else "__in"
            query["tags" + tag_op] = list(common_itertools.chain(tags))

        if hints:
            hint_op = "__all" if all_hints else "__in"
            query["hints" + hint_op] = list(common_itertools.chain(hints))

        def get_params_query(params):
            return [{"key": key, "value": value} for key, value in params.iteritems()]

        def get_output_params_query(params):
            query = []
            for key, value in params.iteritems():
                for reset_on_restart in (False, True):
                    query.append({"key": key, "value": value, "reset_on_restart": reset_on_restart})
            return query

        if input_parameters:
            param_op = "__in" if any_params else "__all"
            query["parameters__input" + param_op] = get_params_query(input_parameters)

        q_expr = mapping.Q()
        has_q_expr = False
        if output_parameters:
            if any_params:
                query["parameters__output__in"] = get_output_params_query(output_parameters)
            else:
                for key, value in output_parameters.iteritems():
                    has_q_expr = True
                    q_expr &= mapping.Q(parameters__output__match=dict(key=key, value=value))

        args = [q_expr] if has_q_expr else []
        generator = mapping.Task.objects.filter(*args, **query)
        if not load_ctx:
            generator = generator.exclude('context')
        if order_by:
            generator = generator.order_by(*common_itertools.chain(order_by))
        if offset:
            generator = generator.skip(int(offset))
        if limit:
            generator = generator.limit(int(limit))
        return generator


class TaskQueue(object):
    """
    Task queue controller
    """

    HOSTS_CACHE_TTL = 420  # ttl of hosts cache in seconds

    q_hosts = None

    class HostsCache(object):
        __slots__ = ("clients", "client_tags", "client_mappings")

        class ClientTagsCache(object):
            def __init__(self, clients):
                self.__cache = {}
                self.__clients = clients
                self.__match_tags = {}
                self.__lock = th.RLock()
                self.__lock_match = th.RLock()

            def hosts(self, client_tags):
                return [
                    hostname
                    for hostname, client in self.__clients.iteritems()
                    if self.match_tags(client, client_tags)
                ]

            def match_tags(self, client, tags, task_platform=None):
                key = (client.hostname, str(tags), task_platform)
                matching_platform = self.__match_tags.get(key, ctm.NotExists)
                if matching_platform is ctm.NotExists:
                    matching_platform = self.__match_tags[key] = client_controller.Client.match_tags(
                        client, tags, task_platform=task_platform
                    )
                return matching_platform

            def __getitem__(self, client_tags):
                client_tags = str(client_tags)

                if client_tags in self.__cache:
                    return self.__cache[client_tags]

                obj = mapping.ClientTagsToHostsCache.objects.lite().with_id(client_tags)
                if obj is None:
                    hosts = self.hosts(client_tags)
                    obj = mapping.ClientTagsToHostsCache(client_tags=client_tags, hosts=hosts)
                    if hosts:
                        obj.save()
                else:
                    now = dt.datetime.utcnow()
                    if now - obj.accessed > dt.timedelta(seconds=TaskQueue.HOSTS_CACHE_TTL):
                        mapping.ClientTagsToHostsCache.objects(client_tags=obj.client_tags).update(
                            set__accessed=now, write_concern={"w": 0}
                        )
                self.__cache[client_tags] = obj.hosts
                return obj.hosts

        def __init__(self):
            self.client_mappings = {client.hostname: client for client in mapping.Client.objects().lite()}
            self.client_tags = self.ClientTagsCache(self.client_mappings)

    class EnqueueResult(common_enum.Enum):
        SUCCESS = None  # Task enqueued
        NEED_VALIDATION = None  # Need ServiceQ validation
        FAILED = None  # Task not enqueued

    @classmethod
    @common_patterns.ttl_cache(HOSTS_CACHE_TTL)
    def hosts_cache(cls):
        logging.info("Calculate cache.")
        return cls.HostsCache()

    # noinspection PyMethodParameters
    @common_patterns.classproperty
    def qclient(cls, tls=th.local()):
        try:
            return tls.qclient
        except AttributeError:
            tls.qclient = qclient.Client()
            return tls.qclient

    @staticmethod
    def requirements(task):  # type: (mapping.Task) -> qtypes.ComputingResources
        """ Prepare requirements model for push to Q """

        return qtypes.ComputingResources(
            # reserve 10Mb for disk_usage.yaml and 10Mb for ramdrive_usage.yaml if task uses ram disk
            disk_space=task.requirements.disk_space + 10 + (10 if task.requirements.ramdrive else 0),
            resources=(
                dict(mapping.Resource.objects(
                    id__in=task.requirements.resources
                ).read_preference(pymongo.ReadPreference.SECONDARY).fast_scalar("id", "size"))
            ),
            cores=task.requirements.cores,
            ram=task.requirements.ram
        )

    @classmethod
    def task_hosts_list(cls, task, cache=None):
        # type: (mapping.Task, TaskQueue.HostsCache) -> (list[str], list[str], list[str], str)
        """ Calculate list of available hosts with scores for task """

        allowed_hosts = []
        failed_filters = []
        exclusive_host = task.requirements.host
        cache = cache or cls.HostsCache()

        client_tags = str(client_controller.Client.perform_tags_by_platform(
            Task.effective_client_tags(task), Task.platform_filter(task)
        ))
        if exclusive_host:
            exclusive_host_model = cache.client_mappings.get(exclusive_host)
            if exclusive_host_model is None:
                exclusive_host_model = mapping.Client.objects.lite().with_id(exclusive_host)
            hosts = [exclusive_host_model.id] if exclusive_host_model is not None else []
        else:
            hosts = cache.client_tags[client_tags]

        if not len(hosts):
            failed_filters.append(
                "No hosts for client tags '{}' / exclusive host '{}'".format(client_tags, exclusive_host)
            )
            return None, None, failed_filters, client_tags

        for client_hostname in hosts:
            client = cache.client_mappings.get(client_hostname)
            if client is None:
                continue
            filters = Task.get_client_filters_info(
                task, client, cache=cache, check_platform=False
            )
            if all(filters):
                allowed_hosts.append(client)
            else:
                failed_filters.append(
                    "{!r} incompatible: {!r}".format(
                        client_hostname, list(it.ifilterfalse(None, filters))
                    )
                )

        if not allowed_hosts:
            return None, None, failed_filters, client_tags

        hosts = [c.hostname for c in allowed_hosts]
        return allowed_hosts, hosts, None, client_tags

    @classmethod
    def update_task_execution_hosts_list(cls, task, cache=None, logger=None, enqueue_task=True):
        # type: (mapping.Task, TaskQueue.HostsCache, logging.Logger, bool) -> (mapping.Task, list[list[str]], str)
        """ Get allowed hosts list for task, check it and push task to queue if enqueue_task is True """
        if logger is None:
            logger = logging
        allowed_hosts, hosts, failed_filters, client_tags = cls.task_hosts_list(task, cache)
        task.effective_client_tags = str(client_tags)
        if not allowed_hosts:
            faq_url = "https://wiki.yandex-team.ru/sandbox/faq#task-scheduling-failed"
            separator = "<br/>"
            Task.set_info(
                task,
                (
                    "There are no appropriate hosts where this task could be executed "
                    "(see <a href=\"{faq_url}\">FAQ</a> for details). "
                    "Hosts and failed predicates follow: {separator}{hosts_and_reasons}"
                ).format(
                    faq_url=faq_url, separator=separator, hosts_and_reasons=separator.join(failed_filters)
                ),
                do_escape=False,
                logger=logger
            )
            Task.set_status(
                task, ctt.Status.EXCEPTION,
                event="Zero hosts matched by task's filters. Effective client tags: {}".format(client_tags),
                logger=logger
            )
            task.save()
            return None

        result = None

        if not hosts:
            Task.set_status(
                task, ctt.Status.EXCEPTION,
                event="Zero allowed hosts. Effective client tags: {}".format(client_tags),
                logger=logger
            )
            task.save()
        else:
            result = (task, hosts, client_tags)
            if enqueue_task:
                cls.add(task, hosts, client_tags, task.score)

        return result, client_tags

    @classmethod
    def add(cls, task, hosts, client_tags, score):
        semaphores = (
            ctt.Semaphores(**task.requirements.semaphores.to_mongo())
            if task.requirements.semaphores else
            None
        )
        if semaphores and semaphores.acquires:
            existing = list(mapping.Semaphore.objects(
                name__in=[_.name for _ in semaphores.acquires]
            ))
            not_allowed = map(
                lambda _: _.name,
                it.ifilter(
                    lambda _: not _.public and task.owner not in common_itertools.chain(_.owner, _.shared),
                    existing
                )
            )
            if not_allowed:
                Task.set_status(
                    task,
                    ctt.Status.EXCEPTION,
                    event="Task owner has no rights to use semaphore(s): {}".format(not_allowed)
                )
                task.save()
                return
            existing_names = set(_.name for _ in existing)
            not_exist = []
            for acquire in semaphores.acquires:
                if acquire.name not in existing_names and not acquire.capacity:
                    not_exist.append(acquire.name)
            if not_exist:
                Task.set_status(
                    task,
                    ctt.Status.EXCEPTION,
                    event="Semaphores {} do not exist".format(not_exist)
                )
                task.save()
                return
        enqueue_time = (
            calendar.timegm(task.time.updated.timetuple()) or calendar.timegm(dt.datetime.utcnow().timetuple())
        )
        cls.qclient.push(
            task.id, task.priority, cls.compress_hosts(hosts), qtypes.TaskInfo(
                requirements=cls.requirements(task), semaphores=semaphores,
                type=task.type, owner=task.owner,
                enqueue_time=enqueue_time,
                duration=Task.kill_timeout(task) * 34 / 55,
                client_tags=str(client_tags)
            ), score=score or 0
        )

    @classmethod
    def compress_hosts(cls, hosts):
        # type: (list[str]) -> str
        """ Make bit mask from hosts list """
        if cls.q_hosts is None:
            cls.q_hosts = cls.qclient.get_hosts()
        bits, unknown_hosts = cls.q_hosts.make_bits(hosts)
        if unknown_hosts:
            cls.q_hosts = cls.qclient.add_hosts(list(unknown_hosts))
            bits, unknown_hosts = cls.q_hosts.make_bits(hosts)
            assert not unknown_hosts
        return bits

    @classmethod
    def validate(cls, logger=None, stopping=None, add_queue_tids=None, enqueued_tasks_delay=0):
        if logger is None:
            logger = logging
        logger.info("Starting task queue validation")
        with common_context.Timer() as timer:
            queue_tids, executing_jobs, del_queue_tids, jobs_to_complete = [], [], [], []
            with timer["enqueueing_tasks"]:
                enqueueing_tids = list(mapping.Task.objects(
                    execution__status=ctt.Status.ENQUEUING,
                    lock_host=""
                ).order_by("id").fast_scalar("id"))
            if enqueueing_tids:
                logger.debug("Number of tasks in status ENQUEUING %d, adding to prequeue", len(enqueueing_tids))
                with timer["q_prequeue_push"]:
                    cls.qclient.prequeue_push(enqueueing_tids)
            if not add_queue_tids:
                with timer["q_fetch"]:
                    queue_tids, executing_jobs = cls.qclient.validate()
                logger.debug(
                    "Service Q reports %d tasks in queue and %d tasks executing.",
                    len(queue_tids), len(executing_jobs)
                )
                with timer["db_fetch"]:
                    # Some tasks can have sessions (assigned to clients), don't re-enqueue them
                    locked_tids = mapping.OAuthCache.locked_task_ids()
                    add_queue_tids = list(mapping.Task.objects(
                        id__nin=queue_tids + locked_tids,
                        execution__status=ctt.Status.ENQUEUED,
                        time__updated__lte=dt.datetime.utcnow() - dt.timedelta(seconds=enqueued_tasks_delay),
                        lock_host=""
                    ).fast_scalar("id").order_by("-priority").limit(500))
                    del_queue_tids = list(mapping.Task.objects(
                        id__in=queue_tids,
                        execution__status__nin=[ctt.Status.ENQUEUING, ctt.Status.ENQUEUED]
                    ).fast_scalar("id"))
                    jobs_to_complete = list(set(executing_jobs) - set(mapping.OAuthCache.objects(
                        token__in=executing_jobs
                    ).scalar("token")))

            logger.debug(
                "Detected %d tasks to be added to the queue, %d to be deleted and %d jobs to be completed.",
                len(add_queue_tids), len(del_queue_tids), len(jobs_to_complete)
            )

            if add_queue_tids:
                logger.debug("Tasks to add: %r", add_queue_tids)
            if del_queue_tids:
                logger.debug("Tasks to delete: %r", del_queue_tids)
            if jobs_to_complete:
                logger.debug("Jobs to complete: %r", jobs_to_complete)
                with timer["complete_jobs"]:
                    with timer["complete_jobs"]["execution_completed"]:
                        completed = cls.qclient.execution_completed(jobs_to_complete)
                    with timer["complete_jobs"]["close_all_intervals"]:
                        for job in completed:
                            Task.close_all_intervals(
                                Task.get(job.id), update=True, consumption=job.consumption
                            )

            with timer["tasks_load"]:
                tasks = list(mapping.Task.objects(id__in=add_queue_tids).limit(5000))
            cache = cls.HostsCache()
            with timer["add_tasks_to_queue"]:
                for task in tasks:
                    logger.debug("Processing task #%r (%r)", task.id, task.type)
                    if stopping and stopping():
                        logger.warning("Received stopping signal, going to correctly finish work")
                        break
                    with timer["task_hosts_list"]:
                        allowed_hosts, hosts, failed_filters, client_tags = cls.task_hosts_list(task, cache=cache)
                    if hosts:
                        try:
                            with timer["q_push"]:
                                cls.add(task, hosts, client_tags, task.score)
                        except qerrors.QAlreadyExecuting as ex:
                            logger.warning(str(ex))
            with timer["q_sync"]:
                logger.info("Deleting %s task(s) from queue", len(del_queue_tids))
                cls.qclient.sync([[tid, None, None] for tid in del_queue_tids])
            logger.info("Tasks queue validated in %s", timer)

    @classmethod
    def check_enqueued_tasks(cls, logger):
        logger.info("Check enqueued tasks for broken dependencies.")

        # Select all "WAIT_RES" tasks
        enq_tasks = dict(
            mapping.Task.objects(
                execution__status=ctt.Status.WAIT_RES
            ).scalar("id", "requirements__resources")
        )
        # .. which has resource dependencies with state differ from "READY"
        not_ready_resources = mapping.Resource.objects(
            state__ne=ctr.State.READY,
            id__in=set(it.chain.from_iterable(enq_tasks.itervalues()))
        ).fast_scalar("id", "task_id", "state")
        # .. and structure it as a dictionary.
        not_ready_resources = {rid: (tid, state) for rid, tid, state in not_ready_resources}
        # Filter out all READY resources from tasks
        enq_tasks = {tid: filter(not_ready_resources.get, deps) for tid, deps in enq_tasks.iteritems()}
        # .. and also tasks with no broken dependencies
        enq_tasks = {tid: set(deps) for tid, deps in enq_tasks.iteritems() if deps}
        # Prepare a set broken resources' producers
        producers = set(res[0] for res in not_ready_resources.itervalues())
        logger.debug(
            "Checking %d enqueued tasks' %d resource dependencies produced by %d tasks",
            len(enq_tasks), len(not_ready_resources), len(producers)
        )
        fail = []

        def tasks2deps(tasks):
            ret = {}
            tasks = set(tasks)
            producers.difference_update(tasks)
            resources = ((rid, tid) for rid, (tid, state) in not_ready_resources.iteritems() if tid in tasks)
            for rid, tid in resources:
                for enq_tid, deps in enq_tasks.iteritems():
                    if rid in deps:
                        ret[enq_tid] = (rid, tid)
            return ret

        # Firstly fail all the tasks, which has BROKEN dependencies from broken tasks, which was started
        # last time more than 24 hours ago.
        deadline = dt.datetime.utcnow() - dt.timedelta(days=1)
        broken_producers = set(mapping.Task.objects(
            id__in=producers,
            execution__status__in=ctt.Status.Group.BREAK,
            time__updated__lt=deadline
        ).scalar("id").order_by("+id"))
        tasks2break = tasks2deps(broken_producers)
        logger.info(
            "Found %d (%r) producer tasks older than 1 day. Set NO_RES for %d tasks: %r",
            len(broken_producers), sorted(broken_producers), len(tasks2break), sorted(tasks2break.viewkeys())
        )
        for task in TaskWrapper.fast_load_list(tasks2break):
            task.set_info(
                "Task has temporary broken dependencies older than 24 hours.\nSee 'Dependencies' tab."
                "(resource:{}, task:{})".format(*tasks2break[task.id]),
                do_escape=False
            )
            task.set_status(ctt.Status.NO_RES, event="Garbage collector: temporary broken dependencies")
            fail.append(task)

        # Now select the rest of tasks without any age filter.
        completed_tasks = set(mapping.Task.objects(
            id__in=producers,
            execution__status__in=ctt.Status.Group.FINISH
        ).scalar("id"))
        logger.debug("Found %d completed producer tasks: %r", len(completed_tasks), sorted(completed_tasks))
        # But re-check their resources to avoid possible race condition.
        recheck_resources = set(rid for rid, (tid, _) in not_ready_resources.iteritems() if tid in completed_tasks)
        dropped = len(map(
            not_ready_resources.pop,
            mapping.Resource.objects(
                id__in=recheck_resources,
                state=ctr.State.READY
            ).fast_scalar("id")
        ))
        logger.debug(
            "Re-checked %d (%r) resources status. Dropped %d resources from the check.",
            len(recheck_resources), sorted(recheck_resources), dropped
        )
        tasks2delete = tasks2deps(completed_tasks)
        logger.info(
            "Breaking %d tasks as completely dropped dependencies: %r",
            len(tasks2delete), sorted(tasks2delete.viewkeys())
        )
        for task in TaskWrapper.fast_load_list(tasks2delete):
            task.set_info(
                "Some dependencies are completely broken.\nSee 'Dependencies' tab."
                "(resource:{}, task:{})".format(*tasks2delete[task.id]),
                do_escape=False
            )
            task.set_status(ctt.Status.NO_RES, event="Garbage collector: completely broken dependencies")
            fail.append(task)

        for task in fail:
            for resource in task.resources():
                if resource.is_not_ready():
                    resource.mark_broken()
            task.save()

    class NoPlatform(Exception):
        pass

    @classmethod
    def get_task_platform(cls, client, task):
        if task.execution.status in (ctt.Status.RELEASING, ctt.Status.STOPPING):
            return None

        # Checking tag/platform requirements
        task_platform = task.requirements.platform
        if not task_platform or task_platform == ctm.OSFamily.ANY:
            task_platform = None
        else:
            task_platform = task_platform.lower()

        task_platforms = set(task.requirements.platforms)
        if task_platform:
            task_platforms.add(task_platform)
        task_platforms = tuple(sorted(task_platforms))

        matched_task_platform = client_controller.Client.match_tags(
            client, Task.effective_client_tags(task), task_platforms, only_detect_platform=True
        )
        client_tags = ctc.Tag.Query(task.requirements.client_tags)
        if client_tags and not task.requirements.host and matched_task_platform is None:
            raise cls.NoPlatform
        return matched_task_platform or task_platform

    @classmethod
    def task_to_execute(cls, client, jobs_ids, free, logger=None):
        """
        The main MAGIC method - selects a next task(s) to be executed by the client.

        :param client: client object
        :param jobs_ids: list of new jobs ids
        :param free: free computing resources on the client
        :type free: sandbox.web.api.v1.schemas.client.ComputingResources
        :param logger: logging facility to log operations into.
        :return: a tuple of task ID to execute, its status, platform to be used for it execution and job_id,
                 or None if the jobs ids cannot be used for getting tasks
        """
        if logger is None:
            logger = logging
        started = time.time()
        settings = common_config.Registry()
        logging.info("Looking for task to execute for %r with jobs %r", client.hostname, jobs_ids)
        if client.hardware.disk_free < settings.client.min_disk_space:
            logger.info(
                "Client '%s' has not enough disk space to execute any task (%s < %s).",
                client.hostname,
                common_format.size2str(client.hardware.disk_free << 20),
                common_format.size2str(settings.client.min_disk_space << 20)
            )
            return

        client_arch = common_platform.get_arch_from_platform(
            next(iter(set(ctc.Tag.Group.OS) & set(client.pure_tags)), None)
        )
        client_platforms = [client_arch] + common_platform.SUB_PLATFORMS[client_arch]

        def _post_execute(task_status):
            if ctc.Tag.POSTEXECUTE not in client.tags or ctc.Tag.Group.SERVICE in client.tags:
                return
            locked_task_ids = mapping.OAuthCache.locked_task_ids()
            if client_arch == ctm.OSFamily.default():
                pending = set(mapping.Task.objects(
                    mapping.Q(
                        execution__status=task_status,
                        requirements__tasks_resource__id__exists=False
                    ) | mapping.Q(
                        execution__status=task_status,
                        requirements__tasks_resource__id__exists=True,
                        requirements__platform__in=client_platforms + [ctm.OSFamily.ANY, "", None]
                    )
                ).fast_scalar("id"))
            else:
                pending = set(mapping.Task.objects(
                    execution__status=task_status,
                    requirements__tasks_resource__id__exists=True,
                    requirements__platform__in=client_platforms
                ).fast_scalar("id"))
            if not pending:
                return
            pending -= set(locked_task_ids)
            return next(iter(pending), None)

        releasing = _post_execute(ctt.Status.RELEASING)
        if releasing:
            yield Task.get(releasing, lite=True), None, jobs_ids[0]
            return

        resources = cls.qclient.resources(secondary=True)
        resources_on_client = dict(mapping.Resource.objects(
            hosts_states__host=client.hostname,
            hosts_states__state=ctr.HostState.OK,
            id__in=resources
        ).read_preference(pymongo.ReadPreference.SECONDARY).fast_scalar("id", "size"))
        capabilities = qtypes.ComputingResources(
            disk_space=client.hardware.disk_free,
            resources=resources_on_client,
            cores=client.hardware.cpu.cores,
            ram=client.hardware.ram,
            slots=client.info["system"].get("total_slots", 1)
        )
        host_info = qtypes.HostInfo(
            capabilities=capabilities,
            tags=tuple(map(
                str,
                (ctc.Tag.Group.expand(
                    ctc.Tag.Group.PURPOSE, ctc.Tag.Group.DISK, ctc.Tag.MULTISLOT
                ) | {ctc.Tag.Group.LINUX, ctc.Tag.Group.OSX, ctc.Tag.WINDOWS}) & set(client.tags)
            )),
            free=qtypes.ComputingResources(**(dict(free))) if free else qtypes.ComputingResources(),
        )

        pre_read_only = settings_controller.Settings.mode() != settings_controller.Settings.OperationMode.NORMAL
        started = started, time.time()
        task_to_execute = cls.qclient.task_to_execute(client.hostname, host_info)
        task_to_execute_it = cls.qclient.task_to_execute_it(client.hostname, host_info, secondary=None)
        got_a_job = False
        job_conflict = False
        try:
            task_to_execute.next()
        except StopIteration:
            pass
        else:
            result = None
            while jobs_ids:
                item = task_to_execute_it.send(result)
                if item is None:
                    break
                tid, score = item
                timer = common_context.Timer().start()
                try:
                    status = mapping.Task.objects.fast_scalar("execution__status").with_id(tid)
                    if status != ctt.Status.ENQUEUED:
                        logger.warning("Task #%s has invalid status: %s", tid, status)
                        continue
                    task = Task.get(tid, lite=True)
                except Task.NotExists:
                    logger.warning("Task #%s does not exist, remove it from Q", tid)
                    cls.qclient.push(tid, None, None)
                    continue

                # Don't start new tasks if pre-read-only-mode on
                if pre_read_only:
                    continue

                with timer["filters"]:
                    # Don't take the task if client doesn't match requirements
                    filter_results = list(common_itertools.chain(
                        Task.get_client_filters_info(task, client),
                        Task.check_available_space_filter(task, client)
                    ))
                    if not all(filter_results):
                        logger.warning(
                            "Cannot take task #%s due: %s (%s)", tid, filter_results, timer
                        )
                        continue

                with timer["get_task_platform"]:
                    # Checking tag/platform requirements
                    try:
                        task_platform = cls.get_task_platform(client, task)
                    except cls.NoPlatform:
                        logger.warning(
                            "Cannot take task #%s due to NoPlatform (%s)", tid, timer
                        )
                        continue

                with timer["commit"]:
                    # Report back to Q that task is actually taken
                    result = task_to_execute.send((tid, jobs_ids[0]))
                    if result == qtypes.QueueIterationResult.SKIP_JOB:
                        logger.warning("Conflict occurred for job %s while committing task #%s", jobs_ids[0], tid)
                        jobs_ids.pop(0)
                        job_conflict = True
                        continue
                    if result != qtypes.QueueIterationResult.ACCEPTED:
                        continue
                    result = qtypes.QueueIterationResult.NEXT_TASK

                now = time.time()
                logger.info(
                    "Task #%s assigned on host %r, arch %r. "
                    "Host's score is %s. "
                    "Spent (SQ): %.3fs (%.3fs). "
                    "Taken in %s",
                    task.id, client.hostname, task_platform or client.platform,
                    score,
                    now - started[0], now - started[1],
                    timer
                )
                got_a_job = True
                yield task, task_platform, jobs_ids.pop(0)
            else:
                task_to_execute_it.send(qtypes.QueueIterationResult.ACCEPTED)
            task_to_execute.send((None, None))
            task_to_execute.send(task_to_execute_it.wait(timeout=qconfig.Registry().serviceq.client.timeout))

        if not got_a_job:
            if job_conflict:
                yield None
                return
            # FIXME: SANDBOX-7243: Temporary process STOPPING tasks after any pending execution
            stopping = _post_execute(ctt.Status.STOPPING)
            if stopping:
                yield Task.get(stopping), None, jobs_ids[0]
            else:
                logger.info("No tasks for host '%s'.", client.hostname)

    @staticmethod
    def _list_not_ready_task_dependencies(task_id):
        task = mapping.Task.objects(id=task_id).only("requirements.resources").first()
        return list(mapping.Resource.objects(
            id__in=task.requirements.resources if task else [],
            state__ne=ctr.State.READY
        ).fast_scalar("id", "task_id", "state"))

    @classmethod
    def finalize_enqueue_task(cls, task, cache=None, logger=None, reset_lock=False):
        # type: (mapping.Task, TaskQueue.HostsCache, logging.Logger, bool) -> TaskQueue.EnqueueResult
        """ Check task dependant resources and try to push task to queue """
        if logger is None:
            logger = logging

        cache = cache or cls.hosts_cache()

        if common_config.Registry().common.installation == ctm.Installation.TEST:
            # To avoid failed tests
            cache = cls.HostsCache()

        with common_context.Timer() as timer:
            logger.debug("Enqueuing task #%s", task.id)
            if task.execution.time.started is None:
                task.execution.time.started = dt.datetime.utcfromtimestamp(0)
                task.execution.time.finished = dt.datetime.utcfromtimestamp(0)
            task.execution.host = ""
            task.execution.disk_usage.last = 0
            try:
                with timer["check_deps"]:
                    no_deps = cls._list_not_ready_task_dependencies(task.id)
                    iteration = Task.iteration(task)
                if no_deps:
                    event = None
                    broken = next(
                        (
                            (rid, tid)
                            for rid, tid, state in no_deps
                            if state != ctr.State.NOT_READY
                        ),
                        None
                    )
                    if broken:
                        Task.set_info(
                            task,
                            "Required resource(s) missing (resource:{}, task:{}).\nSee 'Depends on tasks' tab."
                            .format(*broken),
                            do_escape=False,
                            logger=logger
                        )
                        event = "No resource:{}".format(broken[0])
                    Task.set_status(
                        task,
                        ctt.Status.WAIT_RES if not broken else ctt.Status.NO_RES,
                        event=event,
                        wait_targets={"resources": [item[0] for item in no_deps]},
                        logger=logger
                    )
                elif iteration > common_config.Registry().server.services.tasks_enqueuer.maximum_allowed_restarts:
                    Task.set_status(
                        task,
                        ctt.Status.STOPPED,
                        event="The task exceeded maximum allowed amount of restarts.",
                        logger=logger
                    )
                else:
                    with timer["cache"]:
                        try:
                            enqueue_parameters, client_tags = cls.update_task_execution_hosts_list(
                                task, cache=cache, logger=logger, enqueue_task=False
                            )
                        except Exception as ex:
                            logger.exception("Error in updating hosts list for task #%s. Exception: %s", task.id, ex)
                            return cls.EnqueueResult.FAILED
                    if task.execution.status != ctt.Status.EXCEPTION:
                        with timer["update"]:
                            trigger_controller.TaskStatusNotifierTrigger.create_from_task(task.id)
                            Task.save(
                                task,
                                save_condition={
                                    "execution__status": task.execution.status, "lock_host": task.lock_host
                                },
                                logger=logger
                            )
                        with timer["set_status"] as set_status_timer:
                            Task.set_status(
                                task, ctt.Status.ENQUEUED,
                                event="Effective client tags: {}".format(client_tags),
                                timer=set_status_timer,
                                set_last_action_user=True
                            )
                        with timer["push_to_q"]:
                            try:
                                cls.add(*enqueue_parameters, score=task.score)
                            except (
                                qerrors.QException, jerrors.Timeout,
                                jerrors.HandshakeError, jerrors.CallError,
                                jerrors.ServerUnavailable, jsocket.EOF,
                                socket.error
                            ) as ex:
                                logger.error("Error while adding task #%s to the queue: %s", task.id, ex)
                                if isinstance(ex, qerrors.QAlreadyExecuting):
                                    return cls.EnqueueResult.FAILED
                        if common_config.Registry().common.installation == ctm.Installation.LOCAL:
                            try:
                                logger.debug(
                                    "Waking up client via socket at port %s", common_config.Registry().client.port
                                )
                                socket.create_connection(("127.0.0.1", common_config.Registry().client.port)).close()
                            except Exception as ex:
                                logger.warning("Error waking up locally running client instance: %s", ex)
                if "update" not in timer:
                    with timer["update"]:
                        Task.save(task, logger=logger)
                logger.info(
                    "Task #%s has been switched to '%s' (iteration %d) totally in %s",
                    task.id, task.execution.status, iteration, timer
                )
                return cls.EnqueueResult.SUCCESS
            except common_errors.UpdateConflict as ex:
                logger.warning("Error enqueuing task #%s: %s", task.id, ex)
            except Exception as ex:
                msg = "Error enqueuing task #{}: {}. Will try to enqueue it again a bit later.".format(
                    task.id, ex)
                Task.set_info(task, msg, logger=logger)
                logger.exception(msg)

                if reset_lock:
                    mapping.Task.objects(id=task.id, execution__status=task.execution.status).update(
                        set__execution__description=task.execution.description,
                        set__lock_host=""
                    )
        return cls.EnqueueResult.FAILED

    @classmethod
    def enqueue_task(cls, task, event=None, logger=None, sync=False):
        """
        Enqueue given task. If necessary change task status to ENQUEUING to do some preparing work.

        :param task: task object
        :param event: string with description of event that cause task enqueuing
        :param logger: logging facility to log operations into.
        :return: True if task was successfully enqueued, False otherwise
        """

        if logger is None:
            logger = logging
        request = Task.request
        old_status = task.model.execution.status
        logger.info(
            "Enqueuing task #%s (current status is %r, host: %r)", task.id, old_status, task.model.requirements.host
        )
        if not ctt.Status.can_switch(old_status, ctt.Status.ENQUEUING):
            return False

        # noinspection PyUnusedLocal
        enqueue_result = EnqueueResult(ctt.Status.ENQUEUING)
        with cls.qclient.lock(str(task.model.id)):
            task.model.reload()
            if task.model.execution.status != old_status:
                logger.info(
                    "Task %s changed status on waiting lock. Old status: %s, new status: %s",
                    task.id, old_status, task.model.execution.status
                )
                return False
            permissions_are_ok, error_desc = Task.check_user_permission(task.model, request)
            if not permissions_are_ok:
                raise common_errors.AuthorizationError(error_desc)

            if old_status == ctt.Status.DRAFT:
                try:
                    if request and request.user.login != task.model.author:
                        logging.info(
                            "Author mismatch: #%s is being enqueued by %r while its author is %r",
                            task.id, request.user.login, task.model.author
                        )

                    mapping.Task.objects.filter(id=task.id).update(
                        set__execution__auto_restart__left=task.model.max_restarts
                    )
                    task.model.reload()
                    enqueue_result = task.on_enqueue()
                except:
                    ex = sys.exc_value
                    logger.error("Error while enqueuing task #%s: %s", task.id, ex)
                    Task.audit(mapping.Audit(
                        task_id=task.id,
                        author=task.model.author,
                        content="Enqueuing failed: {}".format(ex),
                    ))

                    task.model.reload()
                    task.save_trace(exc=ex, save=False, update_gsid=False)
                    task.model.save()
                    raise

                else:
                    common_statistics.Signaler().push(dict(
                        type=cts.SignalType.TASK_CREATION,
                        date=task.model.time.created,
                        timestamp=task.model.time.created,
                        author=task.model.author,
                        owner=task.model.owner or ctu.OTHERS_GROUP.name,
                        task_type=task.model.type,
                        unique_key=task.model.unique_key or "",
                        hints=task.model.hints,
                        tags=task.model.tags,
                        task_id=task.id,
                        parent_id=task.model.parent_id,
                        scheduler_id=task.model.scheduler,
                        sdk_type=task.sdk_type,
                    ))

            else:
                try:
                    enqueue_result = task.on_enqueue()
                except:
                    ex = sys.exc_value
                    logger.error("Error while enqueuing task #%s: %s", task.id, ex)
                    task.set_status(
                        new_status=ctt.Status.EXCEPTION,
                        event="Enqueuing failed: {}".format(ex),
                        force=True
                    )
                    task.model.reload()
                    task.save_trace(exc=ex, save=False, update_gsid=False)
                    task.model.save()
                    raise

            if task.needs_container and task.needs_container is not True:
                platform_error = None
                try:
                    res_id, platform, _ = task.get_container_platform()
                except common_errors.ResourceNotFound:
                    platform_error = "Container resource #{} does not exist".format(task.needs_container)
                else:
                    if not platform:
                        platform_error = "Container resource #{} has no 'platform' attribute".format(res_id)
                if platform_error:
                    task.set_status(
                        new_status=ctt.Status.EXCEPTION,
                        event=platform_error,
                        force=True
                    )
                    raise common_errors.TaskError(platform_error)

            logger.info("New status for task #%s: %s", task.id, enqueue_result.new_status)

            settings = common_config.Registry()

            if not settings.server.sync_enqueuing or not sync or enqueue_result.new_status != ctt.Status.ENQUEUING:
                queue_put_result = cls.EnqueueResult.FAILED
            else:
                try:
                    task.save(save_condition={"execution__status": task.status})
                except:
                    logger.warning(
                        "Task #%s enqueueing failed. "
                        "Possible race condition (2) with another master host, "
                        "current status is %r (was %r)",
                        task.id, task.model.execution.status, old_status
                    )
                    return False

                queue_put_result = cls.finalize_enqueue_task(task.model, logger=logger)

            if queue_put_result == cls.EnqueueResult.FAILED:
                try:
                    task.set_status(
                        new_status=ctt.Status.ENQUEUING,
                        event=event,
                        set_last_action_user=True
                    )
                except common_errors.UpdateConflict:
                    task.model.reload()
                    logger.warning(
                        "Task #%s enqueueing failed. "
                        "Possible race condition (2) with another master host, "
                        "current status is %r (was %r)",
                        task.id, task.model.execution.status, old_status
                    )
                    return False
                except Exception:
                    logger.exception("Error while enqueueing of task #%s", task.id)
                    raise

                if enqueue_result.new_status != ctt.Status.ENQUEUING:
                    task.model.reload()
                    task.set_status(
                        new_status=enqueue_result.new_status, wait_targets=enqueue_result.wait_targets
                    )

                try:
                    trigger_controller.TaskStatusNotifierTrigger.create_from_task(task.id)
                    task.save()
                except:
                    pass

        if task.status == ctt.Status.ENQUEUING:
            try:
                cls.qclient.prequeue_push(task.id)
            except (qerrors.QTimeout, qerrors.QRetry, jerrors.RPCError, jsocket.EOF, socket.error) as ex:
                # Don't log traceback, but sent signal anyway.
                logger.error("Error pushing task #%s to prequeue: %s(%s)", task.id, ex.__class__.__name__, ex)
                common_statistics.Signaler().push(dict(
                    type=cts.SignalType.EXCEPTION_STATISTICS,
                    timestamp=dt.datetime.utcnow(),
                    exc_type=ex.__class__.__name__,
                    client_tags=settings.client.tags,
                    client_id=settings.this.id,
                    component=ctm.Component.SERVICEQ,
                ))
        return True

    @classmethod
    def set_priority(cls, request, task, priority, user=None):
        user = user or request.user
        if not user_controller.user_has_permission(user, (task.model.author, task.model.owner)):
            raise ValueError("User {!s} not allowed to change priority for task {!r}".format(
                user.login, task
            ))

        if priority > user_controller.Group.allowed_priority(request, task.model.owner, user):
            raise ValueError("Can not increase task #{} owned by {!r} priority {!r} for user {!s}".format(
                task.id,
                task.model.owner,
                ctt.Priority.make(task.model.priority),
                (user or request.user).login
            ))

        def task_tree(task_id):
            for subtask_gen in it.imap(task_tree, mapping.Task.objects.filter(parent_id=task_id).scalar("id")):
                for subtask_id in subtask_gen:
                    yield subtask_id
            yield task_id

        task_ids = list(task_tree(task.id))
        priority = int(priority)
        cls.qclient.sync([(tid, priority, None, None, None) for tid in task_ids])
        mapping.Task.objects(id__in=task_ids).update(set__priority=priority)

        content = "Priority changed to {!r}".format(ctt.Priority.make(priority))
        for tid in task_ids:
            audit_item = mapping.Audit(
                task_id=tid,
                content=content,
            )
            Task.audit(audit_item)

    @classmethod
    def _owners_rating(cls, owners_rating):
        quotas_by_pools = {}
        for pool, rating in owners_rating.items():
            quotas = {}
            index = 1
            for owner, item in rating:
                if not item.queue_size:
                    quotas[owner] = (None, item)
                else:
                    quotas[owner] = (index, item)
                    index += 1
            quotas_by_pools[pool] = (index - 1, quotas)
        return quotas_by_pools

    @classmethod
    def owners_rating(cls, owner=None):
        owners_rating = cls.qclient.owners_rating_by_pools(owner=owner, secondary=True)
        return cls._owners_rating(owners_rating)

    @classmethod
    def multiple_owners_quota(cls, owners=()):
        quotas_by_pools = cls.qclient.multiple_owners_quota_by_pools(owners=owners, secondary=True)
        return {pool: {q[0]: q[1] for q in quotas} for pool, quotas in quotas_by_pools.items()}

    @common_patterns.classproperty
    @common_patterns.ttl_cache(3600)
    def quota_pools(cls):
        return cls.qclient.quota_pools(secondary=True)

    @classmethod
    def get_enriched_client_tags(cls, client_tags):  # type: (ctc.Tag.Query) -> iter(set)
        platform_tags = {ctc.Tag.Group.LINUX, ctc.Tag.Group.OSX, ctc.Tag.WINDOWS}
        disk_tags = ctc.Tag.Group.DISK
        for pos_tags, neg_tags in ctc.Tag.Query.predicates(client_tags):
            if ctc.Tag.LXC in pos_tags or ctc.Tag.PORTOD in pos_tags:
                pos_tags.add(ctc.Tag.Group.LINUX)
            elif pos_tags & ctc.Tag.Group.LINUX:
                pos_tags.add(ctc.Tag.Group.LINUX)
            if pos_tags & ctc.Tag.Group.OSX:
                pos_tags.add(ctc.Tag.Group.OSX)
            if pos_tags & neg_tags or len(pos_tags & platform_tags) > 1 or len(pos_tags & disk_tags) > 1:
                continue
            if pos_tags & platform_tags:
                target_tags_sets = [pos_tags]
            else:
                target_tags_sets = [pos_tags | {platform} for platform in platform_tags]
            for tags in target_tags_sets:
                if ctc.Tag.Group.LINUX in tags and not tags & disk_tags:
                    for disk in disk_tags:
                        yield tags | {disk}
                else:
                    yield tags
