import datetime as dt

import six
from bson import json_util

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.template as ctte

import sandbox.taskbox.model.bridge as model_bridge
import sandbox.taskbox.model.update as model_update

from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.controller import resource as resource_controller


class TemplateWrapper(object):
    def __new__(cls, model, tasks_resource_id=None, tasks_resource_filter=None):
        if cls is not TemplateWrapper:
            return object.__new__(cls)

        tasks_resource_meta = Template.tasks_resource_meta(
            tasks_resource_id=tasks_resource_id, tasks_resource_filter=tasks_resource_filter, model=model
        )

        if Template.is_taskboxed_task(tasks_resource_meta):
            wrapper = TaskboxTemplateWrapper.__new__(TaskboxTemplateWrapper, model)
        else:
            from sandbox import sdk2
            if model.task.type in sdk2.Task:
                wrapper = NewTemplateWrapper.__new__(NewTemplateWrapper, model)
            else:
                wrapper = OldTemplateWrapper.__new__(OldTemplateWrapper, model)
        wrapper.tasks_resource = tasks_resource_meta
        return wrapper

    @property
    def empty_task(self):
        task = Template.empty_task(self.tasks_resource)
        task.type = self.model.task.type
        return task

    @classmethod
    def create_wrapper_from_task(cls, template_model, task_model):
        """
        Create wrapper from task

        :param template_model: mapping.TaskTemplate
        :param task: controller.TaskWrapper
        """

        tasks_resource = None
        if task_model.requirements and task_model.requirements.tasks_resource:
            tasks_resource = task_model.requirements.tasks_resource.id

        return TemplateWrapper(template_model, tasks_resource_id=tasks_resource)

    def __init__(self, model, *args, **kwargs):
        assert isinstance(model, mapping.TaskTemplate), "Model must be instance of mapping.TaskTemplate"
        self._model = model
        self._rest = None

    def init_model(self):
        """ Init base fields """

        self.model.time.created = dt.datetime.utcnow()
        self.model.time.updated = dt.datetime.utcnow()
        self.model.status = ctte.Status.READY

    @common.utils.singleton_property
    def model(self):
        """ Return mapping.TaskTemplate model """
        return self._model

    @property
    def tasks_resource(self):
        return self.__tasks_resource

    @tasks_resource.setter
    def tasks_resource(self, value):
        self.__tasks_resource = value

    def input_parameters_meta(self):
        """ Get input parameters meta """
        return self.model.task.input_parameters

    def common_parameters_meta(self):
        """ Get common parameters meta """
        return self.model.task.common_parameters

    def requirements_meta(self):
        """ Get requirements meta """
        return self.model.task.requirements_meta

    def update_input_parameters_meta(self):
        """ Set input parameters meta """
        self.model.task.input_parameters = self.input_parameters_meta()

    def update_common_parameter_meta(self):
        """ Set common parameters meta """
        self.model.task.common_parameters = self.common_parameters_meta()

    def update_requirements_meta(self):
        """ Set requirements meta """
        self.model.task.requirements = self.requirements_meta()

    def create(self):
        raise NotImplemented("Method create not implemented for TemplateWrapper")

    def save(self):
        self.model.save()

    @classmethod
    def convert_input_parameter_meta(cls, task_parameter_meta):
        """
        Convert ParametersMeta from task to template
        :param task_parameter_meta: mapping.ParametersMeta.ParameterMeta

        :rtype mapping.Template.Task.ParameterMeta
        """
        parameter_meta = mapping.TaskTemplate.Task.ParameterMeta()

        parameter_meta.name = task_parameter_meta.name
        parameter_meta.required = task_parameter_meta.required
        parameter_meta.title = task_parameter_meta.title
        parameter_meta.description = task_parameter_meta.description
        parameter_meta.modifiers = task_parameter_meta.modifiers
        parameter_meta.context = task_parameter_meta.context
        parameter_meta.output = task_parameter_meta.output
        parameter_meta.type = task_parameter_meta.type
        parameter_meta.sub_fields = task_parameter_meta.sub_fields
        parameter_meta.complex = task_parameter_meta.complex
        parameter_meta.default = task_parameter_meta.default
        parameter_meta.do_not_copy = task_parameter_meta.do_not_copy

        return parameter_meta

    @classmethod
    def convert_input_parameters_meta(cls, task):
        pass

    def convert_common_parameters_meta(cls, task):
        pass

    def convert_requirements_meta(cls, task):
        pass

    def copy_from_task(self, task):
        """
        Fill parameters and requirements from task

        :param task: controller.TaskWrapper
        :rtype controller.TemplateWrapper
        """

        self.model.task.input_parameters = self.convert_input_parameters_meta(task)
        self.model.task.common_parameters = self.convert_common_parameters_meta(task)
        self.model.task.requirements = self.convert_requirements_meta(task)
        return self

    def update_meta(self):
        """ Update meta """
        self.update_input_parameters_meta()
        self.update_common_parameter_meta()
        self.update_requirements_meta()

    def _requirements_update_dict(self, data):
        return data.get("requirements", {})

    def _common_parameter_update_dict(self, data):
        return data.get("parameters", {}).get("common", {})

    def _custom_parameter_update_dict(self, data):
        return data.get("parameters", {}).get("input", {})

    def _check_updating_value(self, name, updating_dict):
        return "value" in updating_dict.get(name, {})

    def _check_updating_meta(self, field, updating_dict):
        return "meta" in updating_dict.get(field.name, {})

    def _make_common_task_request(self, fields, req_data, data):
        for field in fields:
            if field.name in req_data:
                data[field.name] = req_data[field.name]
                continue

            if field.default_from_code:
                continue

            if field.filter:
                res = resource_controller.Resource.last_resource(None, field.filter)
                if res is not None:
                    data[field.name] = res.id
                    continue

            data[field.name] = Template.decode_parameter(field)

    def _make_custom_task_request(self, fields, req_data, data):
        req_data_dict = {d["name"]: d["value"] for d in req_data}
        for field in fields:
            if field.name in req_data_dict:
                data.append({"name": field.name, "value": req_data_dict[field.name]})
                continue

            if field.default_from_code:
                continue

            if field.filter:
                res = resource_controller.Resource.last_resource(None, field.filter)
                if res is not None:
                    data.append({"name": field.name, "value": res.id})
                    continue

            data.append({"name": field.name, "value": Template.decode_parameter(field)})

    def create_task_request(self, req_data):
        data = {"requirements": {}, "custom_fields": []}
        self._make_common_task_request(
            self.model.task.requirements, req_data.get("requirements", {}), data["requirements"]
        )
        self._make_common_task_request(self.model.task.common_parameters, req_data, data)
        self._make_custom_task_request(
            self.model.task.input_parameters, req_data.get("custom_fields", []), data["custom_fields"]
        )
        data["type"] = self.model.task.type

        for notification in data.get("notifications", []):
            for idx, recipient in enumerate(notification["recipients"]):
                if recipient == "<owner>":
                    notification["recipients"][idx] = data.get("owner", "")
                elif recipient == "<author>":
                    notification["recipients"][idx] = Template.request.user.login

        return data

    def _create_task_update_data(self, data):
        task_data = {"requirements": {}, "custom_fields": {}}
        for name, requirement in self._requirements_update_dict(data).iteritems():
            if self._check_updating_value(name, self._requirements_update_dict(data)):
                task_data["requirements"][name] = requirement["value"]

        for name, common_parameter in self._common_parameter_update_dict(data).iteritems():
            if self._check_updating_value(name, self._common_parameter_update_dict(data)):
                task_data[name] = common_parameter["value"]

        for name, custom_parameter in self._custom_parameter_update_dict(data).iteritems():
            if self._check_updating_value(name, self._custom_parameter_update_dict(data)):
                task_data["custom_fields"][name] = custom_parameter["value"]

        return task_data

    def _merge_common_fields(self, updated_fields, old_fields_dict, updating_dict):
        new_fields = []
        for field in updated_fields:
            old_field = old_fields_dict.get(field.name)

            if old_field is not None:
                field.hide = old_field.hide
                field.filter = old_field.filter
                field.default_from_code = old_field.default_from_code

                if not self._check_updating_value(field.name, updating_dict):
                    field.value = old_field.value

            if self._check_updating_meta(field, updating_dict):
                meta = updating_dict[field.name]["meta"]
                for updated_meta_field in ("hide", "default_from_code"):
                    if updated_meta_field in meta:
                        setattr(field, updated_meta_field, meta[updated_meta_field])
                if "filter" in meta:
                    field.filter = common.api.filter_query(
                        meta["filter"], resource_controller.Resource.LIST_QUERY_MAP
                    )
            if field.filter:
                try:
                    resource_controller.Resource.last_resource(resource_id=None, filter=field.filter)
                except Exception as ex:
                    raise ValueError(ex.message)
            new_fields.append(field)
        return new_fields

    def make_audit(self, old_parameters, new_parameters, update_dict, audit_array):
        old_parameters_dict = Template.parameters_dict(old_parameters)
        new_parameters_dict = Template.parameters_dict(new_parameters)
        for name in set(new_parameters_dict) - set(old_parameters_dict):
            audit_array.append(mapping.TemplateAudit.ChangedField(old=None, new=new_parameters_dict[name]))

        for parameter in old_parameters:
            if parameter.name not in new_parameters_dict:
                audit_array.append(mapping.TemplateAudit.ChangedField(old=parameter, new=None))
            elif parameter.name in update_dict:
                audit_array.append(
                    mapping.TemplateAudit.ChangedField(old=parameter, new=new_parameters_dict[parameter.name])
                )

    def _setup_notifications(self, common_parameters, data):
        from sandbox.yasandbox.controller import Task
        if not self._check_updating_value("notifications", data):
            return

        value = data["notifications"]["value"]
        if value is not None:
            notifications = Task.notifications(value)
            new_notifications = []
            for notification in notifications:
                new_notifications.append({
                    "transport": notification.transport,
                    "statuses": notification.statuses,
                    "recipients": notification.recipients,
                    "check_status": notification.check_status,
                    "juggler_tags": notification.juggler_tags
                })
            value = new_notifications
        Template.parameters_dict(common_parameters)["notifications"].value = value

    def update_template(self, data, audit):
        from sandbox.yasandbox.controller import TaskWrapper
        task_update = self._create_task_update_data(data)

        # Fill task custom fields by current template values
        current_task_update = self.create_task_request({})
        for parameter in current_task_update["custom_fields"]:
            name = parameter["name"]
            if name not in task_update["custom_fields"]:
                task_update["custom_fields"][name] = parameter["value"]

        tw = TaskWrapper(self.empty_task)
        model_update.LocalUpdate.update_common_fields(task_update, tw.model)
        tw._tb.update_parameters((task_update["custom_fields"], {}, {}))

        requirements_update_dict = self._requirements_update_dict(data)
        common_parameters_update_dict = self._common_parameter_update_dict(data)
        custom_parameters_update_dict = self._custom_parameter_update_dict(data)

        updated_requirements = self.convert_requirements_meta(tw)
        updated_common_parameters = self.convert_common_parameters_meta(tw)
        self._setup_notifications(updated_common_parameters, common_parameters_update_dict)
        updated_custom_parameters = self.convert_input_parameters_meta(tw)

        new_requirements = self._merge_common_fields(
            updated_requirements, Template.parameters_dict(self.model.task.requirements), requirements_update_dict
        )
        new_common_parameters = self._merge_common_fields(
            updated_common_parameters, Template.parameters_dict(self.model.task.common_parameters),
            common_parameters_update_dict
        )
        new_custom_parameters = self._merge_common_fields(
            updated_custom_parameters, Template.parameters_dict(self.model.task.input_parameters),
            custom_parameters_update_dict
        )

        self.make_audit(self.model.task.requirements, new_requirements, requirements_update_dict, audit.requirements)
        self.make_audit(
            self.model.task.common_parameters, new_common_parameters,
            common_parameters_update_dict, audit.common_parameters
        )
        self.make_audit(
            self.model.task.input_parameters, new_custom_parameters,
            custom_parameters_update_dict, audit.input_parameters
        )

        self.model.task.requirements = new_requirements
        self.model.task.common_parameters = new_common_parameters
        self.model.task.input_parameters = new_custom_parameters


class NewTemplateWrapper(TemplateWrapper):
    def __init__(self, model, **kwargs):
        super(NewTemplateWrapper, self).__init__(model, **kwargs)
        self._tb = self._create_task_bridge()

    def create(self):
        self.init_model()
        self._tb = self._create_task_bridge()
        return self

    def init_model(self):
        super(NewTemplateWrapper, self).init_model()
        self.update_meta()

    def input_parameters_meta(self):
        params = self._tb.template_parameters_meta()
        return params

    def common_parameters_meta(self):
        return self._tb.template_common_parameters_meta()

    def requirements_meta(self):
        requirements = self._tb.template_requirements_meta()
        for parameter_meta in requirements:
            if parameter_meta.name in ("ram", "disk_space") and parameter_meta.default is not None:
                parameter_meta.default <<= 20
        return requirements

    def _create_task_bridge(self):
        request_id = None if Template.request is None else Template.request.id[:8]
        return model_bridge.LocalTaskBridge(self.model.task, request_id, setup_agentr=None)

    @classmethod
    def convert_input_parameters_meta(cls, task):
        input_parameters_meta = []
        input_values = task._input_parameters_raw_values()
        for pm in task.parameters_meta.params:
            if pm.output or not pm.name or pm.title is None:
                continue
            parameter_meta = cls.convert_input_parameter_meta(pm)
            parameter_meta.default_from_code = False
            parameter_meta.hide = False
            parameter_meta.filter = None
            parameter_meta.value = input_values.get(parameter_meta.name)
            input_parameters_meta.append(parameter_meta)
        return input_parameters_meta

    @classmethod
    def convert_common_parameters_meta(cls, task):
        from sandbox import sdk2
        common_parameters_meta = task._tb.template_common_parameters_meta()
        for parameter_meta in common_parameters_meta:
            if parameter_meta.name == "notifications":
                continue

            name = sdk2.Task.Parameters.__parameters_database_map__.get(parameter_meta.name) or parameter_meta.name
            value = getattr(task.model, name)

            if value not in ("", None) and parameter_meta.complex:
                value = value.to_json()

            parameter_meta.default_from_code = False
            parameter_meta.hide = False
            parameter_meta.filter = None
            parameter_meta.value = value
        return common_parameters_meta

    @classmethod
    def convert_requirements_meta(cls, task):
        requirements_meta = task._tb.template_requirements_meta()
        for parameter_meta in requirements_meta:
            value = getattr(task.model.requirements, parameter_meta.name)
            if parameter_meta.name == "tasks_resource" and value is not None:
                value = value.id
            elif parameter_meta.name in ("ram", "disk_space"):
                if value is not None:
                    value <<= 20
                if parameter_meta.default is not None:
                    parameter_meta.default <<= 20
            elif value not in ("", None) and parameter_meta.complex:
                if parameter_meta.name == "ramdrive" and value is not None and value.size is not None:
                    value.size <<= 20
                if isinstance(value, list):
                    value = json_util.dumps([item.to_mongo() for item in value])
                else:
                    value = value.to_json()

            parameter_meta.default_from_code = False
            parameter_meta.hide = False
            parameter_meta.filter = None
            parameter_meta.value = value

        return requirements_meta


class TaskboxTemplateWrapper(NewTemplateWrapper):

    def _create_task_bridge(self):
        request_id = None if Template.request is None else Template.request.id[:8]
        return model_bridge.RemoteTaskBridge(self.empty_task, request_id)


class OldTemplateWrapper(TemplateWrapper):
    def __init__(self, model, **kwargs):
        raise AttributeError("SDK1 Tasks are not supported in templates")

    @common.utils.singleton_property
    def cls(self):
        import sandbox.yasandbox.proxy.task
        return sandbox.yasandbox.proxy.task.getTaskClass(self._model.task.type)


class Template(object):
    Model = mapping.TaskTemplate

    @classmethod
    def is_taskboxed_task(cls, tasks_resource_meta):
        return tasks_resource_meta is not None and tasks_resource_meta.taskbox_enabled

    @classmethod
    def initialize(cls):
        cls.Model.ensure_indexes()
        mapping.TemplateAudit.ensure_indexes()

    @common.utils.classproperty
    def request(self):
        return getattr(mapping.base.tls, "request", None)

    @classmethod
    def copy(cls, source, alias, shared_with=None):
        model = mapping.TaskTemplate()
        model.author = cls.request.user.login
        model.description = source.description
        model.shared_with = shared_with
        model.task = source.task
        model.alias = alias
        model.time.created = dt.datetime.utcnow()
        model.time.updated = dt.datetime.utcnow()

        if source.status == ctte.Status.DELETED:
            model.status = ctte.Status.READY
        else:
            model.status = source.status

        return TemplateWrapper(model)

    @classmethod
    def requirements_dict_from_task(cls, task):
        requirements_dict = {}
        if getattr(task, "requirements", None) is not None:
            for requirement in task.requirements:
                requirements_dict[requirement.name] = requirement
        return requirements_dict

    @classmethod
    def parameters_dict(cls, parameters):
        return {parameter.name: parameter for parameter in parameters}

    @classmethod
    def decode_parameter(cls, field):
        value = field.value
        value = model_update.decode_parameter(value, field.complex)
        # Multi-select field must be a list of strings (selected items), but it is false for sdk1 tasks.
        if field.type == ctt.ParameterType.MULTISELECT and isinstance(value, six.string_types):
            value = value.split()
        return value

    @classmethod
    def tasks_resource_meta(cls, tasks_resource_id=None, tasks_resource_filter=None, model=None):
        tasks_resource = None
        tasks_resource_meta = None

        if tasks_resource_id is not None or tasks_resource_filter is not None:
            tasks_resource_filter = common.api.filter_query(
                tasks_resource_filter, resource_controller.Resource.LIST_QUERY_MAP
            )
            tasks_resource = resource_controller.Resource.last_resource(
                resource_id=tasks_resource_id,
                filter=tasks_resource_filter
            )
        else:
            if model is not None:
                resource_model = Template.requirements_dict_from_task(model.task).get("tasks_resource")
                if resource_model is not None:
                    tasks_resource = resource_controller.Resource.last_resource(
                        resource_id=resource_model.value,
                        filter=resource_model.filter
                    )

        if tasks_resource is not None:
            tasks_resource_meta = mapping.Task.Requirements.TasksResource()
            model_update.ServerSideUpdate.updated_tasks_resource_model(tasks_resource_meta, tasks_resource)
        return tasks_resource_meta

    @classmethod
    def empty_task(cls, tasks_resource):
        task = mapping.Task(id=0)
        task.requirements = mapping.Task.Requirements()
        task.requirements.tasks_resource = tasks_resource
        return task
