import logging
import datetime as dt
import functools as ft

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn

from sandbox.yasandbox.database import mapping

from . import notification as notification_controller


class TimeTrigger(object):
    class Exception(Exception):
        pass

    class AlreadyExists(Exception):
        pass

    class NotExists(Exception):
        pass

    Model = mapping.TimeTrigger

    @classmethod
    def initialize(cls):
        """ Initialize db for TimeTrigger model. """
        cls.Model.ensure_indexes()

    @classmethod
    def count(cls):
        """ Get number of triggers. """
        return cls.Model.objects().count()

    @classmethod
    def active_triggers(cls):
        """
        Get list of triggered triggers.
        :rtype: list of mapping.TimeTrigger
        """
        return cls.Model.objects(
            activated=True,
            time__lte=dt.datetime.utcnow(),
            source__nin=mapping.OAuthCache.locked_task_ids()
        )

    @classmethod
    def create(cls, model):
        """
        Create new trigger.

        :param model: TimeTrigger model
        :rtype: mapping.TimeTrigger
        """
        while True:
            try:
                return model.save(force_insert=True)
            except mapping.NotUniqueError:
                obj = cls.get(model.source)
                if obj.token != model.token:
                    # TODO: sessions are unique to task_id, so there can be at most one at the moment
                    actual_token = next(iter(
                        mapping.OAuthCache.objects(
                            token__in=(obj.token, model.token)
                        ).order_by("-validated").scalar("token")
                    ), None)
                    if model.token != actual_token:
                        raise cls.AlreadyExists("Time trigger for task #{} already exists".format(model.source))
                cls.delete(obj)

    @classmethod
    def get(cls, source):
        """
        Get trigger by source.

        :param source: trigger id
        :rtype: mapping.TimeTrigger
        """
        model = cls.Model.objects.with_id(source)
        if not model:
            raise cls.NotExists("Time trigger for task #{} does not exist".format(source))
        return model

    @classmethod
    def update(cls, model):
        """
        Save trigger to db.

        :param model: TimeTrigger model
        :rtype: mapping.TimeTrigger
        """
        return model.save()

    @classmethod
    def delete(cls, model):
        """
        Delete trigger.

        :param model: TimeTrigger model
        """
        model.delete()


class TaskTrigger(object):
    class Exception(Exception):
        pass

    class AlreadyExists(Exception):
        pass

    class NotExists(Exception):
        pass

    name = "Task trigger"
    Model = None  # must be set by subclass

    @classmethod
    def initialize(cls):
        cls.Model.ensure_indexes()

    @classmethod
    def count(cls):
        """ Get number of triggers. """
        return cls.Model.objects().count()

    @classmethod
    def create(cls, model):
        """
        Create new trigger.

        :param model: trigger object
        """
        while True:
            try:
                return model.save(force_insert=True)
            except mapping.NotUniqueError:
                obj = cls.get(model.source)
                if obj.token != model.token:
                    actual_token = next(iter(
                        mapping.OAuthCache.objects(
                            token__in=(obj.token, model.token)
                        ).order_by("-validated").scalar("token")
                    ), None)
                    if model.token != actual_token:
                        raise cls.AlreadyExists("{} for task #{} already exists".format(cls.name, model.source))
                cls.delete(obj)

    @classmethod
    def get(cls, source):
        """
        Get trigger by source.

        :param source: trigger object
        """
        model = cls.Model.objects.with_id(source)
        if not model:
            raise cls.NotExists("{} for task #{} does not exist".format(cls.name, source))
        return model

    @classmethod
    def update(cls, model):
        """
        Save trigger to db.

        :param model: trigger object
        """
        return model.save()

    @classmethod
    def delete(cls, model):
        """
        Delete trigger.

        :param model: trigger object
        """
        model.delete()


class TaskStatusTrigger(TaskTrigger):
    name = "Status trigger"
    Model = mapping.TaskStatusTrigger

    class AlreadyExists(TaskTrigger.AlreadyExists):
        pass

    class NotExists(TaskTrigger.NotExists):
        pass

    @classmethod
    def list(cls, targets=None, statuses=None):
        """
        Get list of triggers.

        :param targets: list of task ids
        :param statuses: list of task statuses
        :rtype: list of mapping.TaskStatusTrigger
        """
        query = {}
        if targets:
            query["targets__in"] = targets
        if statuses:
            query["statuses__in"] = statuses
        return cls.Model.objects(**query)

    @staticmethod
    def get_not_ready_targets(targets, statuses):
        not_ready_targets = []

        for tid, status in mapping.Task.objects(id__in=targets).scalar("id", "execution__status"):
            if status in statuses:
                # target is already ready
                continue
            not_ready_targets.append(tid)

        return not_ready_targets


class TaskOutputTrigger(TaskTrigger):
    name = "Output trigger"
    Model = mapping.TaskOutputTrigger

    class AlreadyExists(TaskTrigger.AlreadyExists):
        pass

    class NotExists(TaskTrigger.NotExists):
        pass

    @classmethod
    def list(cls, targets=None, any_params=False):
        """
        Get list of triggers.

        :param targets: list of pairs (task_id, field)
        :param any_params: if True match either of targets instead of all of them
        :rtype: list of mapping.TaskOutputTrigger
        """
        query = {}
        if targets:
            param_op = '__in' if any_params else '__all'
            query["targets__" + param_op] = [
                {"target": target, "field": field}
                for target, field in targets
            ]
        return cls.Model.objects(**query)


class TaskStatusNotifierTrigger(object):
    Model = mapping.TaskStatusNotificationTrigger
    notification_resolver = {}

    @classmethod
    def initialize(cls):
        """ Initialize db for TaskStatusNotificationTrigger model. """
        cls.Model.ensure_indexes()
        cls.notification_resolver = {
            ctn.Transport.EMAIL: cls._email_resolver,
            ctn.Transport.TELEGRAM: notification_controller.Notification.telegram_recipients,
            ctn.Transport.Q: notification_controller.Notification.q_recipients,
            ctn.Transport.JUGGLER: notification_controller.Notification.juggler_recipients,
        }

    @common.utils.classproperty
    def _email_resolver(self):
        return ft.partial(notification_controller.Notification.recipients, resolve_groups=True, exclude_robots=True)

    @classmethod
    def create_from_task(cls, task_id):
        cls.Model.objects(source=task_id).delete()
        for notification in mapping.Task.objects.scalar("notifications").with_id(task_id):
            if notification.transport in cls.notification_resolver:
                cls.append(task_id, notification)

    @classmethod
    def append(cls, task_id, notification, resolver=None):
        recipients = (resolver or cls.notification_resolver[notification.transport])(*notification.recipients)
        statuses = [_ for _ in notification.statuses if _ not in ctt.Status.Group.NONWAITABLE]
        if not recipients or not statuses:
            return

        try:
            cls.Model(
                source=task_id,
                statuses=statuses,
                recipients=recipients,
                transport=notification.transport,
                check_status=notification.check_status,
                juggler_tags=notification.juggler_tags
            ).save()
        except mapping.NotUniqueError:
            pass


def delete_wait_triggers(task_id):
    logging.warning("Removing all triggers for task #%s", task_id)
    mapping.TimeTrigger.objects(source=task_id).delete()
    mapping.TaskOutputTrigger.objects(source=task_id).delete()
    mapping.TaskStatusTrigger.objects(source=task_id).delete()
