import re
import uuid
import hashlib
import logging
import datetime as dt
import collections

import six

from sandbox.common import fs as common_fs
from sandbox.common import abc as common_abc
from sandbox.common import auth as common_auth
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import itertools as common_it
from sandbox.common import patterns as common_patterns
from sandbox.common.types import misc as ctm
from sandbox.common.types import user as ctu
from sandbox.common.types import task as ctt

from sandbox.yasandbox.database import mapping


def user_has_permission(user, owners, groups_cache=None):
    # type: (str or mapping.User, iter, dict or None) -> bool

    settings = common_config.Registry()
    if not settings.server.auth.enabled:
        return True
    if user and isinstance(user, six.string_types):
        if user == User.anonymous.login:
            return False
        user = mapping.User.objects.with_id(user)
    if not user:
        return False
    if user.super_user:
        return True
    groups = Group.get_user_groups(user, groups_cache=groups_cache)
    if (set([user.login] + groups) | Group.public_groups()) & set(owners or []):
        return True
    return False


def user_has_right_to_act_on_behalf_of(user, requested_login):  # type: (mapping.User, str) -> bool
    if user.login == requested_login or User.has_roles(user, [ctu.Role.TASKS_SUDO]):
        return True
    owned_robots = mapping.RobotOwner.objects.fast_scalar("robots").with_id(user.login) or []
    return requested_login in owned_robots


def validate_credentials(credentials, groups=None):
    """
    Check whether the credentials is a valid user(s) or group(s)

    :param credentials: name or list of names
    :param groups: validate: False - only users, True - only groups, None - both
    :raises Exception: if credentials is not valid
    """
    settings = common_config.Registry()
    if not settings.server.auth.enabled:
        return
    for c in common_it.chain(credentials):
        if (groups is False or not Group.exists(c)) and (groups is True or not User.validate(c)):
            raise Exception("Invalid credential '{}'".format(c))


class OAuthCache(object):
    Model = mapping.OAuthCache

    class SessionExpired(Exception):
        """
        Raises if task session is expired.
        It especially differs with :class:`common.proxy.ReliableServerProxy.SessionExpired`
        """

    _validation_ttl = 10
    _ssh_key_ttl = 7200

    @classmethod
    def initialize(cls):
        cls.Model.ensure_indexes()

        settings = common_config.Registry().server.auth
        cls._ssh_key_ttl = settings.ssh_key.token_ttl * 60
        cls._validation_ttl = settings.oauth.validation_ttl

    @classmethod
    def get(cls, token):
        now = dt.datetime.utcnow()
        token = cls.Model.objects(token=token).first()
        if not token:
            return
        if not token.source or not token.login:
            token.delete()
            return
        if not token.ttl:
            return token if token.validated + dt.timedelta(minutes=cls._validation_ttl) > now else None
        cls.validate_token(token)
        if token.validated + dt.timedelta(seconds=token.ttl / 2) < now:
            token.update(set__validated=now)
        return token

    @classmethod
    def validate_token(cls, token):
        if token.validated + dt.timedelta(seconds=token.ttl) < dt.datetime.utcnow():
            if token.source.startswith(ctu.TokenSource.CLIENT):
                raise cls.SessionExpired("Session {!r} for task #{} on client '{}' has been expired".format(
                    token.token, token.task_id, token.source.split(":")[1]
                ))
            raise cls.SessionExpired("Session {}:{} has been expired".format(
                token.source, token.app_id
            ))

        if token.expired:
            raise cls.SessionExpired("Session {}/#{} is no longer valid".format(token.token, token.task_id))

        return token

    @classmethod
    def is_valid_token(cls, token):
        try:
            cls.validate_token(token)
        except cls.SessionExpired:
            return False
        else:
            return True

    @classmethod
    def refresh(cls, login, token=None, source=ctu.TokenSource.PASSPORT, app_id=None):
        now = dt.datetime.utcnow()
        if not token:
            if source != ctu.TokenSource.SSH_KEY:
                raise ValueError("Token can be generated only for SSH key authentication")

            query = {"login": login, "source": source}
            if app_id:
                query["app_id"] = app_id
            cache = cls.Model.objects(**query).first()
            if not cache or cache.ttl and cache.validated + dt.timedelta(seconds=cache.ttl) < now:
                token = uuid.uuid4().hex
                cache = cls.Model(token=token)
                # Also, drop other caches with the same source and application ID
                cls.Model.objects(**query).delete()
        else:
            cache = cls.Model.objects.with_id(token) or cls.Model(token=token)

        cache.login = login
        cache.source = source
        cache.validated = now
        if app_id:
            cache.app_id = app_id
        if source == ctu.TokenSource.SSH_KEY:
            cache.ttl = cls._ssh_key_ttl

        try:
            cache.save()
        except mapping.NotUniqueError:
            pass
        return cache

    @classmethod
    def refresh_all(cls, logins=None, valid_until=None):
        """
        Refresh specified users oauth token TTLs to eliminate Blackbox blackouts )

        :param logins: list of user logins or `None` for all.
        :param valid_until: time when tokens expire
        :return: Amount of updated documents
        """
        kwargs = {"source": ctu.TokenSource.PASSPORT}
        if logins:
            kwargs["login__in"] = common_it.chain(logins)
        if valid_until is None:
            valid_until = dt.datetime.utcnow()
        return cls.Model.objects(**kwargs).update(set__validated=valid_until)

    @classmethod
    def expire(cls, session):
        if session.expired:
            return  # Already expired, nothing to do

        logging.info(
            "Session %s (task: #%s, client: %r) change: %s -> EXPIRED",
            session.token, session.task_id, session.client_id, session.state
        )
        session.state = str(ctt.SessionState.EXPIRED)
        session.save()

    @classmethod
    def abort(cls, session, reason):
        logging.info(
            "Session %s (task: #%s, client: %r) change: %s -> ABORTED (reason: %s)",
            session.token, session.task_id, session.client_id, session.state, reason
        )

        if session.state != ctt.SessionState.ABORTED and reason == ctt.Status.STOPPING:
            session.ttl += common_config.Registry().common.task.execution.terminate_timeout

        session.state = str(ctt.SessionState.ABORTED)
        session.abort_reason = str(reason)
        session.save()

    @classmethod
    def update(cls, task, **kwargs):
        update_data = mapping.OAuthCache.UpdateData(**kwargs)
        update_kws = {
            "set__state": ctt.SessionState.UPDATED,
            "set__update_data": update_data
        }

        if update_data.kill_timeout:
            update_kws["inc__ttl"] = update_data.kill_timeout - task.kill_timeout

        updated = mapping.OAuthCache.objects(
            task_id=task.id, state=ctt.SessionState.ACTIVE
        ).update_one(**update_kws)

        return updated

    @classmethod
    def commit(cls, session):

        updated = mapping.OAuthCache.objects(
            token=session.token, state=ctt.SessionState.UPDATED
        ).update_one(
            set__state=ctt.SessionState.ACTIVE,
            set__update_data=None
        )
        return updated


class ValidatedUser(object):
    def __init__(self, user, is_dismissed=False):
        self.user = user
        self.is_dismissed = is_dismissed

    def __nonzero__(self):
        return bool(self.user and not self.is_dismissed)


class User(object):
    Model = mapping.User

    class StaffInfo(common_patterns.Abstract):
        """ Specification for UI presentation class. """
        __slots__ = ("login", "is_dismissed", "is_robot", "telegram_login", "uid")
        __defs__ = (None, False, False, None, None)

    STAFF_CHECK_TTL = 1  # day
    LONG_TERM_STAFF_TTL = 3  # days
    BLACKBOX_CHECK_PERIOD = 10  # minutes

    STAFF_API_INTERVAL = 1  # seconds
    STAFF_API_TIMEOUT = 30  # seconds
    STAFF_API_MAX_TIMEOUT = 60  # seconds

    PARAM_NAMES = {
        'owner', 'author', 'task_type', 'status', 'host', 'descr_mask', 'show_childs', 'hidden', 'important_only', 'p',
        'state', 'omit_failed', 'attr_name', 'attr_value', 'resource_type', 'order_by', 'id'
    }

    @classmethod
    def initialize(cls):
        """
        initialize db for User model
        """
        cls.Model.ensure_indexes()

    @common_patterns.singleton_classproperty
    def anonymous(cls):
        return cls.Model(
            login=ctu.ANONYMOUS_LOGIN,
            super_user=not common_config.Registry().server.auth.enabled,
        )

    @common_patterns.classproperty
    def service_user(cls):
        group = mapping.Group.objects.with_id(common_config.Registry().common.service_group)
        service_user = group and next(iter(map(
            lambda _: _.login,
            mapping.User.objects(login__in=group.users, robot=True, super_user=True),
        )), None) or ctu.ANONYMOUS_LOGIN
        if service_user != ctu.ANONYMOUS_LOGIN:
            cls.service_user = service_user
        return service_user

    @classmethod
    def has_roles(cls, user, roles):
        # type: (mapping.User, list[ctu.Role]) -> bool
        """ Check that user has roles """

        if user.super_user:
            return True
        user_roles = set(user.roles)
        if ctu.Role.ADMINISTRATOR in user_roles:
            return True

        for role in roles:
            if role not in user_roles:
                return False
        return True

    @staticmethod
    def create(model):
        """
        create new user with specified name

        :param model: user model
        :return model: User model
        """
        try:
            return model.save(force_insert=True)
        except mapping.NotUniqueError:
            raise Exception('User "{}" already exists'.format(model.login))

    @classmethod
    def get(cls, login):  # type: (str) -> mapping.User
        return cls.Model.objects(login=login).first()

    @classmethod
    def add_allowed_api_method(cls, login, method_name):
        """
        allow user to execute restricted api method
        """
        cls.Model.objects(login=login).update(add_to_set__allowed_api_methods=method_name)

    @classmethod
    def can_execute_api_method(cls, login, method_name):
        """
        check ability to execute restricted api method.
        Actually, this method is unused - it is referenced only in tests.
        """
        return bool(cls.Model.objects(
            login=login,
            allowed_api_methods__in=[method_name]
        ).count())

    @classmethod
    def valid(cls, login):
        """
        Returns user object if its marked as valid, i.e., validated recently.
        :param login: user name
        """
        return cls.Model.objects(
            login=login,
            staff_validate_timestamp__gt=(dt.datetime.utcnow() - dt.timedelta(days=cls.STAFF_CHECK_TTL))
        ).first()

    @common_patterns.singleton_classproperty
    def staff_api(cls):
        settings = common_config.Registry()
        staff_url = settings.server.auth.staff.url
        staff_token = common_fs.read_settings_value_from_file(settings.server.auth.oauth.token)

        rest_client = common_rest.Client(base_url=staff_url, auth=common_auth.OAuth(staff_token), logger=logging)
        rest_client.DEFAULT_INTERVAL = cls.STAFF_API_INTERVAL
        rest_client.DEFAULT_TIMEOUT = cls.STAFF_API_TIMEOUT
        rest_client.MAX_TIMEOUT = cls.STAFF_API_MAX_TIMEOUT
        rest_client.reset()

        rest_client = rest_client << common_rest.Client.HEADERS({
            "Accept-Encoding": "identity",  # SANDBOX-6056
        })

        return rest_client

    @classmethod
    def validate_login(cls, login=None, uid=None):
        """
        Get Staff info of the given user. Return `None` if the user is not on Staff.

        :param login: Staff login
        :param uid: Passport user id
        :return: `StaffInfo` instance or `None`
        """
        if login is None and uid is None:
            return None
        fields = ("login", "official.is_dismissed", "official.is_robot", "accounts", "uid")
        query = {
            "_one": 1,
            "_fields": ",".join(fields),
        }
        if login is not None:
            query["login"] = login
        if uid is not None:
            query["uid"] = uid
        try:
            person = cls.staff_api.persons.read(**query)
        except common_rest.Client.HTTPError:
            return None

        telegram_login = None
        for account in person["accounts"]:
            if account["type"] == "telegram":
                telegram_login = account["value"]
                break

        return cls.StaffInfo(
            login=person["login"],
            is_dismissed=person["official"]["is_dismissed"],
            is_robot=person["official"]["is_robot"],
            telegram_login=telegram_login,
            uid=person["uid"],
        )

    @classmethod
    def cached_user(cls, user, now, avoid_staff_validation):
        if not user:
            return None
        user_svt = user.staff_validate_timestamp
        if avoid_staff_validation or (user_svt and user_svt > now - dt.timedelta(days=cls.STAFF_CHECK_TTL)):
            return ValidatedUser(user)
        else:
            return None

    @classmethod
    def user_from_validation_info(cls, validation_info, user=None, now=None):
        if now is None:
            now = dt.datetime.utcnow()
        if not validation_info:
            return ValidatedUser(None)
        if validation_info.is_dismissed:
            return ValidatedUser(None, is_dismissed=True)
        if not user:
            user = cls.Model(login=validation_info.login)
        user.staff_validate_timestamp = now
        user.robot = validation_info.is_robot
        user.telegram_login = validation_info.telegram_login
        user.uid = validation_info.uid
        user.save()
        return ValidatedUser(user)

    @classmethod
    def validate_from_uid(cls, uid, avoid_staff_validation=False):
        """
        Check uid of user on staff if needed and mark it as valid on success. Create new user if not registered yet.

        :param uid:   user uid
        :param avoid_staff_validation: do not validate uid on staff if ttl expired; use for trusted users.
        :return:        :class:`mapping.User` in case the user is valid or `None` otherwise.
        """
        settings = common_config.Registry()
        if not settings.server.auth.enabled:
            return ValidatedUser(cls.anonymous)
        now = dt.datetime.utcnow()
        user = cls.Model.objects(uid=uid).first()
        validated_user = cls.cached_user(user, now, avoid_staff_validation)
        if validated_user:
            return validated_user
        validation_info = cls.validate_login(uid=uid)
        return cls.user_from_validation_info(validation_info, user=user, now=now)

    @classmethod
    def validate_user(cls, login, user, avoid_staff_validation=False):
        """
        Check user on staff if needed and mark it as valid on success. Create new user if not registered yet.

        :param user: user model with fields login and staff_validate_timestamp
        :param avoid_staff_validation: do not validate login on staff if ttl expired; use for trusted logins.
        :return: ValidatedUser class with mapping.User in case the user is valid or `None` otherwise
        """
        now = dt.datetime.utcnow()
        validated_user = cls.cached_user(user, now, avoid_staff_validation)
        if validated_user:
            return validated_user
        validation_info = cls.validate_login(login)
        return cls.user_from_validation_info(validation_info, user=user, now=now)

    @classmethod
    def validate(cls, login, avoid_staff_validation=False):
        """
        Check login on staff if needed and mark it as valid on success. Create new user if not registered yet.

        :param login:   user login
        :param avoid_staff_validation: do not validate login on staff if ttl expired; use for trusted logins.
        :return: ValidatedUser class with mapping.User in case the user is valid or `None` otherwise
        """
        settings = common_config.Registry()
        if not settings.server.auth.enabled or login == ctu.ANONYMOUS_LOGIN:
            return ValidatedUser(cls.anonymous)

        user = cls.get(login)
        return cls.validate_user(login, user, avoid_staff_validation=avoid_staff_validation)

    @classmethod
    def validated(cls, login, is_robot=False):  # type: (str, bool) -> None
        """
        Mark as valid on success. Create new user if not registered yet.

        :param login:    user login
        :param is_robot: whether user is robot
        :return:         `None`
        """
        if not cls.Model.objects(login=login).update(
            set__staff_validate_timestamp=dt.datetime.utcnow(),
            set__robot=is_robot
        ):
            cls.Model(
                login=login,
                staff_validate_timestamp=dt.datetime.utcnow(),
                robot=is_robot
            ).save()

    @classmethod
    @common_patterns.ttl_cache(300)
    def is_super(cls, login):
        """
        Check if user with specified login is superuser.

        :param login: user login
        :return: True if user is superuser, False - otherwise
        :rtype: bool
        """
        if not common_config.Registry().server.auth.enabled:
            return True
        user = cls.Model.objects(login=login).first()
        return bool(user) and user.super_user

    @staticmethod
    def _get_hashed_sid(sid):
        return hashlib.sha256(sid.encode('utf-8')).hexdigest()

    @classmethod
    def get_login_by_sid(cls, sid, need_check=True):
        """ Determines user's login by session ID cache. """
        if need_check:
            query = cls.Model.objects(
                session__id=cls._get_hashed_sid(sid),
                session__updated__gt=(dt.datetime.utcnow() - dt.timedelta(minutes=cls.BLACKBOX_CHECK_PERIOD))
            )
        else:
            query = cls.Model.objects(session__id=cls._get_hashed_sid(sid))
        return query.scalar("login").first()

    @classmethod
    def _filter_params(cls, params):
        """
        Filters parameters for filtering session documents.
        Only parameters presented in PARAM_NAMES is allowed.

        :params params: dict with query
        :return: filtered dict query
        :rtype: dict
        """
        return {key: value for key, value in params.iteritems() if key in cls.PARAM_NAMES}

    @classmethod
    def update_path_params(cls, login, path, params):
        cls.Model.objects(login=login).update(**{'set__preferences__{}'.format(path): cls._filter_params(params)})

    @classmethod
    def get_path_params(cls, login, path):
        return cls.Model.objects(login=login).only('preferences.{}'.format(path)).first().preferences.get(path, {})

    @classmethod
    def set_session_id(cls, login, sid):
        """ Updates user's session ID cache. """
        cls.Model.objects(login=login).update(set__session=cls.Model.Session(id=cls._get_hashed_sid(sid)))

    @classmethod
    def refresh_staff(cls, logins=None, valid_until=None):
        """
        Refresh staff validation TTLs

        :param logins: list of user logins or `None` for all.
        :param valid_until: time when validation expires
        :return: Amount of updated documents
        """
        query = {}
        if logins:
            query["login__in"] = common_it.chain(logins)
        if valid_until is None:
            valid_until = dt.datetime.utcnow()
        return cls.Model.objects(**query).update(set__staff_validate_timestamp=valid_until)

    @common_patterns.singleton_classproperty
    def uids(cls):
        return dict(cls.Model.objects(uid__exists=True).fast_scalar("uid", "login"))

    @classmethod
    def uid_to_login(cls, uid):
        """
        Get user login by uid

        :param uid: passport uid
        :return: staff login
        """

        if uid in cls.uids:
            return cls.uids[uid]

        user = cls.staff_api.persons.read(
            _one=1,
            _fields="login",
            uid=uid
        )["login"]

        cls.uids[uid] = user
        return user

    @classmethod
    @common_patterns.ttl_cache(3600)
    def check_telegram_username(cls, telegram_username):
        """
        :param telegram_username: telegram username
        :return: True, if telegram_username is bound to active staff member
        """
        if common_config.Registry().common.installation in ctm.Installation.Group.LOCAL:
            return True

        result = cls.staff_api.persons.read(
            params={
                "accounts.type": "telegram",
                "accounts.value": telegram_username,
            }
        ).get("result", [])
        return bool(result and not result[0]["official"]["is_dismissed"])


class Group(object):
    Model = mapping.Group
    NAME_CHECK_RE = re.compile(r"^[A-Z0-9_-]+$", re.IGNORECASE)
    ABC_CHECK_RE = NAME_CHECK_RE

    class AlreadyExists(ValueError):
        pass

    @common_patterns.singleton_classproperty
    def su(cls):
        return cls.Model(
            name="SU",
            priority_limits=cls.Model.PriorityLimits(**dict(zip(
                ("ui", "api"),
                map(int, ctu.SU_PRIORITY_LIMITS)
            )))
        )

    @common_patterns.singleton_classproperty
    def regular(cls):
        return cls.Model(
            name="REGULAR",
            priority_limits=cls.Model.PriorityLimits(**dict(zip(
                ("ui", "api"),
                map(int, ctu.DEFAULT_PRIORITY_LIMITS)
            )))
        )

    @common_patterns.singleton_classproperty
    def anonymous(cls):
        return cls.Model(
            name=ctu.OTHERS_GROUP.name,
            priority_limits=cls.Model.PriorityLimits(**dict(zip(
                ("ui", "api"),
                map(int, ctu.OTHERS_GROUP.priority_limits)
            )))
        )

    @common_patterns.classproperty
    def service_group(cls):
        return common_config.Registry().common.service_group

    @classmethod
    def _validate_base_fields(cls, model):  # type: (mapping.Group) -> None
        model.name = model.name.strip().upper()
        if not model.name:
            raise ValueError("Empty group name")
        if not cls.NAME_CHECK_RE.match(model.name):
            raise ValueError(
                "Group name '{}' does not match regular expression '{}'".format(model.name, cls.NAME_CHECK_RE.pattern)
            )
        if not model.users:
            raise ValueError("Empty users list")

    @staticmethod
    def _validate_users_on_staff(users, cache=None):
        settings = common_config.Registry()
        if not settings.server.auth.enabled:
            return
        humans = 0
        validated_users = []

        for login in users:
            if cache is not None:
                validated_user = cache.validate_user(login)
            else:
                validated_user = User.validate(login)
            if validated_user:
                humans += int(not validated_user.user.robot)
                validated_users.append(login)
        if not humans:
            raise ValueError("Group must contain at least one real user")
        return validated_users

    @classmethod
    def initialize(cls):
        """
        initialize db for Group model
        """
        cls.Model.ensure_indexes()

    @classmethod
    def create(cls, model):  # type: (mapping.Group) -> mapping.Group

        cls._validate_base_fields(model)
        cls._validate_users_on_staff(model.users)
        if model.email:
            model.email = model.email.strip()

        model.abc = (model.abc or "").lower().strip() or None
        if model.abc and not cls.validate_abc(model.abc):
            raise ValueError("Invalid ABC service: {}".format(model.abc))

        try:
            if model.name == cls.anonymous.name:
                raise mapping.NotUniqueError()
            return model.save(force_insert=True)
        except mapping.NotUniqueError:
            raise cls.AlreadyExists("Group '{}' already exists".format(model.name))

    @classmethod
    def edit(cls, model, cache=None):
        """
        edit group model

        :param model: Group model
        :param cache: Group cache
        :return model: Group model
        """
        cls._validate_base_fields(model)
        model.users = cls._validate_users_on_staff(model.users, cache=cache)
        if model.email:
            model.email = model.email.strip()
        if not cls.Model.objects.with_id(model.name):
            raise ValueError('Group "{}" does not exist'.format(model.name))

        changed_fields = model._get_changed_fields()

        if "abc" in changed_fields:
            model.abc = (model.abc or "").lower().strip() or None
            if model.abc and not cls.validate_abc(model.abc):
                raise ValueError("Invalid ABC service: {}".format(model.abc))

        return model.save()

    @classmethod
    def delete(cls, group):
        """
        delete group

        :param group: group name
        """
        if not cls.Model.objects.with_id(group):
            raise ValueError('Group "{}" does not exist'.format(group))
        cls.Model.objects(name=group).delete()

    @classmethod
    def get(cls, group):
        """
        get group

        :param group: group name
        :return model: Group model
        """
        if group == cls.anonymous.name:
            return cls.anonymous
        obj = cls.Model.objects.with_id(group)
        if not obj:
            raise ValueError('Group "{}" does not exist'.format(group))
        return obj

    @classmethod
    def exists(cls, group):
        return group == cls.anonymous.name or bool(cls.Model.objects.with_id(group))

    @classmethod
    def list(cls):
        """
        list all groups

        :return list: Group models list
        """
        return cls.Model.objects.order_by('+id')

    @classmethod
    def unpack(cls, lst):
        """
        Given a list of users and groups, return a set of users, including group members
        """
        groups = cls.Model.objects(name__in=lst).scalar("name", "users")

        result = set(lst)
        for group_name, group_users in groups:
            result.discard(group_name)
            result.update(group_users)

        return result

    @classmethod
    def get_user_groups(cls, user, objects=False, include_public=True, groups_cache=None):
        if isinstance(user, mapping.User):
            if groups_cache is None:
                result = user.groups if objects else user.groups_names
            else:
                result = groups_cache.get(user.login, [])
        else:
            if groups_cache is None:
                data = cls.Model.objects(users=user)
                result = list(data if objects else data.fast_scalar("name"))
            else:
                result = groups_cache.get(user, [])

        # try to get list of user groups and fallback to anonymous if list is empty
        if not result or user == User.anonymous.login:
            result += [cls.anonymous if objects else cls.anonymous.name]
        if include_public:
            result.extend(list(Group.public_groups(objects=objects)))
        return result

    @classmethod
    def add_user(cls, group, user):
        """
        add user to group

        :param group: group name
        :param user: user name
        """
        if not cls.Model.objects(name=group).update(add_to_set__users=user):
            raise ValueError('Group "{}" does not exist'.format(group))

    @classmethod
    def remove_user(cls, group, user):
        """
        remove user from group

        :param group: group name
        :param user: user name
        """
        if not cls.Model.objects(name=group, users=user).update(pull__users=user):
            if not cls.Model.objects.with_id(group):
                raise ValueError('Group "{}" does not exist'.format(group))
            else:
                raise ValueError('User "{}" not in group "{}"'.format(user, group))

    @classmethod
    def change_email(cls, group, email):
        """
        change group email

        :param group: group name
        :param email: new group email
        """
        if not cls.Model.objects(name=group).update(set__email=email):
            raise ValueError('Group "{}" does not exist'.format(group))

    @staticmethod
    def get_view_url(group):
        return '/sandbox/admin/groups/view?group_name={}'.format(group)

    @classmethod
    def allowed_priority(cls, request, owner, user=None, attr=None, ssh_key=None):
        """ Returns maximum allowed priority for the given request and task owner. """
        user = user or request.user
        ssh_key = request.token_source == ctu.TokenSource.SSH_KEY if ssh_key is None else ssh_key
        attr = attr or ("ui" if request.source == request.Source.WEB or ssh_key else "api")
        try:
            if user.super_user:
                grp = cls.su
            elif owner == user.login:
                # TODO: SANDBOX-2545 This code allows priority SERVICE:NORMAL for tasks, which was created
                # TODO: by real people, but without any group specified. This actually required by only one
                # TODO: case - upload files from console via "REMOTE_COPY_RESOURCE" task.
                # TODO: After the task creation for upload will not be required, this hack can be dropped.
                grp = cls.anonymous if user.robot else cls.regular
            else:
                grp = cls.get(owner)
        except ValueError:
            grp = cls.anonymous

        pl = getattr(grp.priority_limits or cls.regular.priority_limits, attr)
        if pl is None:
            pl = getattr(ctu.DEFAULT_PRIORITY_LIMITS, attr)
        else:
            pl = ctt.Priority.make(pl)

        return pl.prev if ssh_key else pl

    @classmethod
    def get_source_content(cls, source, cache=None):
        content_getter = cls.SYNC_SOURCES.get(source.source)
        content = []
        if content_getter:
            for group_name in (filter(None, (g.strip() for g in source.group.split(",")))):
                if cache is not None:
                    logins = cache.content(source.source, group_name)
                else:
                    logins = set(content_getter(group_name) or [])
                content.append(mapping.Group.SyncSource.SyncSourceContent(name=group_name, logins=list(logins)))
        return content

    @classmethod
    def sync(cls, group, cache=None):
        for source in group.sources:
            source.content = cls.get_source_content(source, cache=cache)
        updated_users = group.get_users()
        old_users = set(group.users)
        group.users = list(updated_users)
        cls.edit(group, cache=None)
        diff_removed = old_users - updated_users
        diff_added = updated_users - old_users
        if diff_removed or diff_added:
            logging.debug("Group %s sync finished:\nAdded %s\nRemoved %s", group.name, diff_added, diff_removed)

    @classmethod
    def check_abc(cls, task):
        message = ""
        if not getattr(task, "owner", False):
            message = "[Warning]: Task owner was not set! Please, set owner."
        else:
            owner = task.owner
            group = mapping.Group.objects(name=owner).first()
            if not group:
                message = "[Warning]: Task owner is not a group, but a single user. Please, use group for task owner."
            else:
                if not getattr(group, "abc", ""):
                    message = "[Warning]: Current owner does not have an ABC group. Please, set the abc group."
        return message

    @staticmethod
    def _validate_user(name, cache=None):
        if cache is not None:
            validation_info = cache.validate_user(name)
        else:
            validation_info = User.validate(name)
        if validation_info.user:
            return [name]
        elif validation_info.is_dismissed:
            return []
        return None

    @classmethod
    def validate_abc(cls, service):
        """
        Test existence of an ABC service.

        :param service: ABC service slug (e.g. `sandbox`)
        :rtype: bool
        """
        if common_config.Registry().common.installation == ctm.Installation.LOCAL:
            return True
        return common_abc.abc_service_id(service) is not None

    @staticmethod
    @common_patterns.ttl_cache(60)
    def _oauth_token():
        config = common_config.Registry()
        return common_fs.read_settings_value_from_file(
            config.server.auth.oauth.token, ignore_file_existence=True
        )

    @staticmethod
    def staff_group_content(name, cache=None):
        match = Group._validate_user(name, cache=cache)
        if match is not None:
            return match

        config = common_config.Registry()
        staff_api = common_rest.Client(
            config.server.auth.staff.url,
            Group._oauth_token() or common_auth.NoAuth()
        )

        def getter(gname):
            logins = map(
                lambda _: _["person"]["login"],
                staff_api.groupmembership[{
                    "group.url": gname,
                    "person.official.is_dismissed": False,
                    "_fields": "person.login"
                }]["result"]
            )
            map(
                lambda _: logins.extend(getter(_["url"])),
                staff_api.groups[{
                    "parent.url": gname,
                    "_fields": "url"
                }]["result"]
            )
            return logins
        return getter(name)

    @staticmethod
    def rb_group_content(name, cache=None):
        match = Group._validate_user(name, cache=cache)
        if match is not None:
            return match

        config = common_config.Registry()
        arc_api = common_rest.Client(
            config.server.services.group_synchronizer.arc_api_url,
            Group._oauth_token() or common_auth.NoAuth()
        )

        response = arc_api.groups[name].members.read()
        return [user["name"] for user in response.get("data", [])]

    @staticmethod
    def abc_group_content(name, cache=None):
        """
        ABC sync syntax:

        "service" - ABC service
        "service/scope" - ABC service with specific scope e.g. "sandbox/development"
        "@user" - a specific user
        """

        if name.startswith("@"):
            match = Group._validate_user(name[1:], cache=cache)
            return match or []
        return common_abc.abc_group_content(name)

    SYNC_SOURCES = {
        ctu.GroupSource.STAFF: staff_group_content.__func__,
        ctu.GroupSource.RB: rb_group_content.__func__,
        ctu.GroupSource.ABC: abc_group_content.__func__,
        ctu.GroupSource.USER: _validate_user.__func__
    }

    @classmethod
    def check_abc_setting_permissions(cls, user, abc_group):
        if user.super_user:
            return True
        users = common_abc.users_by_service_and_role(abc_group, [common_abc.RESOURCE_MANAGER, common_abc.PRODUCT_HEAD])
        return user.login in users

    @classmethod
    @common_patterns.ttl_cache(3600)
    def public_groups(cls, objects=False):
        query = cls.Model.objects(public=True)
        if objects:
            return set(query)
        else:
            return set(query.fast_scalar("name"))

    @classmethod
    def normalize_sync_sources(cls, sources):
        for sync in sources:
            entities = set(x.strip() for x in sync.group.split(","))
            entities.discard(ctu.ANONYMOUS_LOGIN)
            sync.group = ", ".join(sorted(entities))


class GroupCache(object):
    ShotUser = collections.namedtuple("ShotUser", "login staff_validate_timestamp robot")

    def __init__(self):
        self.users = {}
        self.validated_users = {}
        self.source_cache = {source: {} for source in Group.SYNC_SOURCES.keys()}

    def initialize(self):
        self.users = {
            user.login: user for user in
            map(
                lambda _: self.ShotUser(*_),
                mapping.User.objects.fast_scalar(*self.ShotUser._fields)
            )
        }

    def validate_user(self, login):
        if login not in self.validated_users:
            if login in self.users:
                now = dt.datetime.utcnow()
                validated_user = User.cached_user(self.users[login], now, False)
                if validated_user:
                    self.validated_users[login] = validated_user
                else:
                    self.validated_users[login] = User.validate(login)
            else:
                self.validated_users[login] = User.validate(login)
        return self.validated_users[login]

    def content(self, source, group_name):
        content_getter = Group.SYNC_SOURCES.get(source)
        if not content_getter:
            return set()
        if group_name not in self.source_cache[source]:
            self.source_cache[source][group_name] = set(content_getter(group_name, cache=self) or [])
        return self.source_cache[source][group_name]
