import itertools as it
import functools as ft

from sandbox import common
from sandbox.yasandbox.database import mapping

from . import user as user_controller


class Vault(object):
    class Exception(Exception):
        pass

    class AlreadyExists(Exception):
        pass

    class NotExists(Exception):
        pass

    class NotAllowed(Exception):
        pass

    class EncryptionKeyNotDefined(Exception):
        pass

    class TaskIsNotExecuting(Exception):
        pass

    class InvalidFields(Exception):
        pass

    class UnknownUser(Exception):
        pass

    class KeyAmbiguity(Exception):
        pass

    @common.utils.singleton_classproperty
    def encryption_key(cls):  # type: () -> bytes or None
        return common.utils.read_settings_value_from_file(
            common.config.Registry().server.encryption_key,
            ignore_file_existence=True, binary=True
        )

    @classmethod
    def __encrypt(cls, data):
        """
        Encrypt model's data before saving it into the database (SANDBOX-4661)
        :param data: secret data
        :type model: str
        :return: encrypted data string
        :rtype: str
        """

        if cls.encryption_key:
            data = common.crypto.AES(cls.encryption_key).encrypt(data, use_base64=False, use_salt=True)

        return data

    @classmethod
    def __decrypt(cls, data):
        """
        Decrypt model's data after extraction from the database before giving it away (SANDBOX-4661)
        :param data: secret data
        :type data: str
        :return: decrypted data string
        :rtype: str
        """

        decrypted = data
        if cls.encryption_key:
            decrypted = common.crypto.AES(cls.encryption_key).decrypt(data, use_base64=False)
        return decrypted

    @classmethod
    def data_length(cls, data):
        """
        Decrypt secret data to figure how long it is.
        Only used to show length hint in secret edit window

        :param data: secret data
        :return: length of data after decryption
        :rtype: int
        """

        return len(cls.__decrypt(data))

    @staticmethod
    def _get_name(user):
        return user.login if isinstance(user, mapping.User) else user.name

    @classmethod
    def _check_access(cls, user, model, only_owner=False):
        settings = common.config.Registry()
        if not settings.server.auth.enabled:
            return True
        if isinstance(user, mapping.User) and user.super_user and not user.robot:
            return True
        if isinstance(user, mapping.User):
            user_groups = list(user_controller.Group.get_user_groups(user.login))
        else:
            user_groups = []
        allowed_users = list(model.allowed_users) if not only_owner else []
        return bool(set([cls._get_name(user)] + user_groups).intersection([model.owner] + allowed_users))

    @staticmethod
    def initialize():
        """
        initialize db for Vault model
        """
        mapping.Vault.ensure_indexes()

    @staticmethod
    def list(owner=None):
        """
        List vault items available for a user, if given; do not unencrypt secrets.

        :param owner: login of owner of vault items to return, if None then return all items
        :return list: Vault models list
        """
        query = {}
        if owner:
            owners = it.chain(user_controller.Group.get_user_groups(owner), (owner,))
            query['owner__in'] = owners
        return mapping.Vault.objects(**query).order_by('+owner', '+name')

    @classmethod
    def create(cls, model):
        """
        create new vault item with specified name

        :param model: Vault model
        :return model: Vault model
        """

        model.data = cls.__encrypt(model.data)
        try:
            return model.save(force_insert=True)
        except mapping.NotUniqueError:
            raise Vault.AlreadyExists('Vault item "{}" owned by user "{}" already exists'.format(
                model.name,
                model.owner
            ))

    @classmethod
    def get_by_id(cls, user, item_id):
        """
        get Vault model by id

        :param user: user object from request
        :return model: Vault model
        """
        model = mapping.Vault.objects.with_id(item_id)
        if not model:
            raise Vault.NotExists('Vault item #{} does not exist'.format(item_id))
        if not cls._check_access(user, model):
            raise Vault.NotAllowed('User "{}" not allowed to get vault item #{}'.format(
                cls._get_name(user),
                item_id
            ))

        model.data = cls.__decrypt(model.data)
        return model

    @classmethod
    def get(cls, user, owner, name):
        """
        get Vault model by owner and name

        :param user: user object from request
        :param owner: login of owner of vault item to return
        :param name: name of vault item
        :return model: Vault model
        """
        if owner:
            model = mapping.Vault.objects(owner=owner, name=name).first()
        else:
            models = filter(ft.partial(cls._check_access, user), mapping.Vault.objects(name=name))
            if len(models) > 1:
                raise cls.KeyAmbiguity(
                    'Ambiguity between key {!r} owners {!r}'.format(name, [_.owner for _ in models])
                )
            model = models[0] if models else None
        if not model:
            raise Vault.NotExists('Vault item {!r} owned by {!r} does not exist'.format(name, owner))
        if not cls._check_access(user, model):
            raise Vault.NotAllowed('User {!r} not allowed to get vault item {!r} owned by {!r}'.format(
                cls._get_name(user),
                name,
                owner
            ))

        model.data = cls.__decrypt(model.data)
        return model

    @classmethod
    def encrypt_data(cls, item, task, key, use_base64=False):
        """
        get Vault data by item and task object, encrypted by secret key generated for the given client.

        :param item: vault item to be encrypted
        :param task: task object, which is requested the data
        :param key: encryption key
        :param use_base64: if True, additionally encode result with base64
        :return str: encrypted data
        """
        settings = common.config.Registry()
        if not settings.server.auth.enabled:
            owner = None
            author = None
        else:
            try:
                owner = user_controller.Group.get(task.owner)
            except ValueError:
                owner = user_controller.User.get(task.owner)
                if not owner:
                    raise ValueError("Unknown task #{} owner '{}'".format(task.id, task.owner))
            author = user_controller.User.get(task.author)
            if not author:
                raise ValueError("Unknown task #{} author '{}'".format(task.id, task.author))
        if not (cls._check_access(owner, item) or cls._check_access(author, item)):
            raise Vault.NotAllowed(
                "Task owner {!r} or author {!r} is not allowed to get vault item {!r} owned by {!r}".format(
                    task.owner, task.author, item.name, item.owner
                )
            )
        if not key:
            raise Vault.EncryptionKeyNotDefined(
                'Vault encryption key is not defined for the task #{} session.'.format(task.id)
            )

        data = cls.__decrypt(item.data)
        return common.crypto.AES(key).encrypt(data, use_base64)

    @classmethod
    def update(cls, user, model, encrypt=True):
        """
        update Vault item

        :param user: user object from request
        :param model: Vault model
        :param encrypt: whether to encrypt data or not (for partial updates this is oftentimes set to False)
        :return model: Vault model
        """
        if not model.name:
            raise Vault.InvalidFields('Empty vault item name')
        user_controller.validate_credentials(model.allowed_users)
        if not cls._check_access(user, model, only_owner=True):
            raise Vault.NotAllowed('User "{}" not allowed to update vault item #{}'.format(
                cls._get_name(user),
                model.id
            ))

        if encrypt:
            model.data = cls.__encrypt(model.data)
        return model.save()

    @classmethod
    def delete(cls, user, model):
        """
        delete vault item

        :param model: Vault model
        """
        if not cls._check_access(user, model, only_owner=True):
            raise Vault.NotAllowed('User "{}" not allowed to delete vault item #{}'.format(
                cls._get_name(user),
                model.id
            ))
        model.delete()
