from __future__ import unicode_literals

import os
import datetime as dt
import contextlib

from sandbox import common
from sandbox.common.types import misc as ctm
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.controller import vault as controller


class Token(common.patterns.NamedTuple):
    __slots__ = ("secret", "signature", "token")

    class Error(Exception):
        pass

    @property
    def request(self):
        """
        Return a dictionary suitable for `tokenized_requests` parameter in Yav API
        :rtype: dict
        """
        req = {"token": self.token}
        if self.signature:
            req["signature"] = self.signature
        if self.secret.version_uuid:
            req["secret_version"] = self.secret.version_uuid
        return req

    def to_model(self, author):
        settings = common.config.Registry()
        is_local = (settings.common.installation == ctm.Installation.LOCAL)

        key = controller.Vault.encryption_key
        if not key:
            msg = "Encryption key is not configured."
            if is_local:
                msg += " See https://wiki.yandex-team.ru/sandbox/yav/#local-sandbox for details."
            raise self.Error(msg)

        token = self.token.encode("utf-8")
        encrypted_token = common.crypto.AES(key).encrypt(token, use_base64=True, use_salt=True)
        return mapping.YavToken(
            author=author,
            secret_uuid=self.secret.secret_uuid,
            signature=self.signature,
            token=encrypted_token
        )

    @classmethod
    def decrypt_token(cls, secret, yav_token):
        """
        Get and decrypt delegation token for the given secret.

        :param secret: yav secret
        :type secret: common.yav.Secret
        :param yav_token: yav token
        :type yav_token: tuple(secret_uuid, signature, token)
        :returns: delegation token
        :rtype: Token
        :raises `Token.Error`: if tokens can't be decrypted
        """

        secret_uuid, signature, token = yav_token
        decrypted_token = common.crypto.AES(controller.Vault.encryption_key).decrypt(token, use_base64=True)
        if decrypted_token is None:
            raise cls.Error("delegation token could not be decrypted")
        decrypted_token = decrypted_token.decode("utf-8")
        return cls(secret=secret, signature=signature, token=decrypted_token)

    @classmethod
    def get(cls, secrets):
        """
        Get and decrypt delegation tokens for the given secrets.

        :param secrets: yav secrets
        :type secrets: List[common.yav.Secret]
        :returns: delegation tokens
        :rtype: List[Token]
        :raises `Token.Error`: if tokens are not found for some of the secrets
        """

        secret_uuids = {s.secret_uuid for s in secrets}
        tokens = (
            mapping.YavToken.objects
            .fast_scalar("secret_uuid", "signature", "token")
            .filter(secret_uuid__in=secret_uuids)
        )

        data = {}
        for token in tokens:
            data[token[0]] = token

        missing = secret_uuids - set(data)
        if missing:
            raise cls.Error("undelegated secrets: {}".format(", ".join(missing)))

        result = []
        for secret in secrets:
            result.append(cls.decrypt_token(secret, data[secret.secret_uuid]))

        return result


class SecretInfo(common.patterns.Abstract):
    __slots__ = ("id", "name", "comment", "author", "updated_at", "delegated", "versions")
    __defs__ = (None, None, None, None, None, None, None)


class Yav(object):

    class Error(Exception):
        pass

    def __init__(self, service_ticket, user_ticket=None, consumer=None):
        settings = common.config.Registry()

        self._app_id = settings.common.tvm.apps.sandbox_production  # default for both production and local Sandbox
        if settings.common.installation == ctm.Installation.PRE_PRODUCTION:
            self._app_id = settings.common.tvm.apps.sandbox_testing

        self._service_ticket = service_ticket
        self._user_ticket = user_ticket
        self._consumer = consumer

    @classmethod
    @contextlib.contextmanager
    def _yav_exception_handler(cls):
        try:
            yield
        except common.rest.Client.HTTPError as exc:
            try:
                resp = exc.response.json()
                msg = resp["message"]
            except (ValueError, KeyError):
                msg = exc.response.response.text
            raise cls.Error("HTTP {} from Yav: {}".format(exc.status, msg))

    def _auth(self):
        settings = common.config.Registry()

        # Use TVM for non-local installations
        if settings.common.installation != ctm.Installation.LOCAL:
            return common.auth.TVMSession(service_ticket=self._service_ticket, user_ticket=self._user_ticket)

        # The environment variable should be set by ServiceAPI runner (see serviceapi/__main__.py)
        oauth_token = os.environ.get(ctm.OAUTH_TOKEN_ENV_NAME)

        if not oauth_token:
            # Fallback
            oauth_token = common.utils.read_settings_value_from_file(
                settings.server.auth.oauth.token, ignore_file_existence=True
            )

        if not oauth_token:
            raise self.Error(
                "Could not get an OAuth token. "
                "See https://wiki.yandex-team.ru/sandbox/yav/#local-sandbox for details."
            )
        return common.auth.OAuth(oauth_token)

    @common.utils.singleton_property
    def _client(self):
        settings = common.config.Registry()
        return common.rest.Client(settings.server.yav.url, auth=self._auth())

    def create_token(self, secret):
        """
        Create delegation token in Yav for the given secret. Requires TVM user ticket.

        :param secret: yav secret
        :type secret: `Secret`
        :returns: delegation token
        :rtype: `Token`
        """
        assert self._user_ticket, "user ticket is required for this method"

        settings = common.config.Registry()

        data = {
            "comment": "Token for Sandbox",
        }
        if settings.common.installation != ctm.Installation.LOCAL:
            data["tvm_client_id"] = self._app_id

        with self._yav_exception_handler():
            resp = self._client.secrets[secret.secret_uuid].tokens(**data)

        if resp.get("status") != "ok":
            msg = resp.get("message", "Unknown error")
            raise self.Error(msg)

        return Token(secret, None, resp["token"])

    def check_revoked(self, secret, yav_token):
        """
        Get and decrypt delegation token for the given secret.

        :param secret: yav secret
        :type secret: common.yav.Secret
        :param yav_token: yav token
        :type yav_token: tuple(secret_uuid, signature, token)
        :returns: token was revoked
        :rtype: bool
        :raises `Token.Error`: if tokens can't be decrypted
        """
        token = Token.decrypt_token(secret, yav_token)
        info = self._client.tokens.info.create(token.request)
        return info["token_info"]["state_name"] != "normal"

    def get_by_tokens(self, tokens, uid=None):  # type: (list[Token], str or None) -> dict[str, common.yav.Data]
        """
        Fetch secret values by their delegation tokens.

        :param tokens: delegation tokens
        :param uid: optional Staff uid to test that the user is allowed to use the tokens.
        :returns: secret values
        :raises `Yav.Error`: if some secrets could not be fetched
        """

        if not tokens:
            return {}

        reqs = [token.request for token in tokens]

        for req in reqs:
            if uid is not None:
                req["uid"] = uid
            req["service_ticket"] = self._service_ticket

        params = {}
        if self._consumer:
            params["consumer"] = self._consumer

        with self._yav_exception_handler():
            response = self._client.tokens.create(data={"tokenized_requests": reqs}, params=params)

        secrets = {}
        errors = []

        for secret, token in zip(response.get("secrets", []), tokens):
            if secret["status"] == "ok":
                secrets[token.secret] = common.yav.Data.from_dict(secret)
            else:
                errors.append("{}: code='{}', message='{}'".format(
                    token.secret,
                    secret.get("code", "unknown"),
                    secret.get("message")
                ))

        if errors:
            raise self.Error("; ".join(errors))

        return secrets

    def suggest(self, query, uuids=None, page=0, limit=10):
        """
        Search Yav secrets available to the current user

        :param query: search query
        :param uuids: Yav secrets uuids
        :param page: page number to return starting from 0
        :param limit: page size to return
        :rtype: List[dict]
        :raises `Yav.Error`: for any Yav-side error
        """
        assert self._user_ticket, "user ticket is required for this method"

        if uuids:
            secrets = self.secrets(uuids, delegation=True)
        else:
            with self._yav_exception_handler():
                response = self._client.secrets.read(
                    query=query,
                    query_type="prefix",
                    role="READER",
                    order_by="name",
                    asc=1,
                    page=page,
                    page_size=limit,
                )

            if response.get("status") != "ok":
                msg = response.get("message", "Unknown error")
                raise self.Error(msg)

            secrets = {}
            for secret in response.get("secrets", []):
                secrets[secret["uuid"]] = SecretInfo(
                    id=secret["uuid"],
                    name=secret["name"],
                    comment=secret.get("comment", ""),
                    author=secret["creator_login"],
                    updated_at=dt.datetime.utcfromtimestamp(secret["updated_at"]),
                )

            delegated = set(
                mapping.YavToken.objects.fast_scalar("secret_uuid").filter(
                    secret_uuid__in=[s.id for s in secrets.itervalues()]
                )
            )
            for secret in secrets.values():
                secret.delegated = (secret.id in delegated)

        return secrets.values()

    def secrets(self, uuids, versions=False, delegation=False, raise_on_errors=True):
        """
        Fetch info for the given Yav secrets uuids

        :param uuids: Yav secrets uuids
        :type uuids: List[str]
        :param versions: Return secrets versions
        :type versions: bool
        :param delegation: Return delegation flag
        :type delegation: bool
        :param raise_on_errors: Raise exception on any error from Yav API,
                                otherwise ignore them and return available secrets
        :type raise_on_errors: bool
        :rtype: Dict[str, SecretInfo]
        :raises `Yav.Error`: for any Yav-side error
        """
        assert self._user_ticket, "user ticket is required for this method"

        secrets = {}

        for uuid in uuids:

            # noinspection PyBroadException
            try:
                with self._yav_exception_handler():
                    resp = self._client.secrets[uuid].read()

                if resp.get("status") != "ok":
                    msg = resp.get("message", "Unknown error")
                    raise self.Error(msg)

            except Exception:
                if raise_on_errors:
                    raise
                continue

            secret = resp["secret"]

            info = SecretInfo(
                id=secret["uuid"],
                name=secret["name"],
                comment=secret.get("comment", ""),
                author=secret["creator_login"],
                updated_at=dt.datetime.utcfromtimestamp(secret["updated_at"]),
            )

            if versions:
                info.versions = [
                    {
                        "version": version["version"],
                        "author": version["creator_login"],
                        "comment": version.get("comment", ""),
                        "created_at": dt.datetime.utcfromtimestamp(version["created_at"]),
                        "keys_": version["keys"]
                    }
                    for version in secret["secret_versions"]
                ]

            secrets[info.id] = info

        if delegation and secrets:
            delegated = set(mapping.YavToken.objects.fast_scalar("secret_uuid").filter(secret_uuid__in=list(secrets)))
            for secret in secrets.values():
                secret.delegated = (secret.id in delegated)

        return secrets
