import re
import inspect
import datetime as dt
import collections

import requests
from infi.clickhouse_orm import database

from sandbox.common import patterns as common_patterns

from sandbox.yasandbox.database.clickhouse import models
from sandbox.yasandbox.database.clickhouse import exceptions


class DistributedDatabase(database.Database):
    """
    Wrapper around `infi.clickhouse_orm.database.Database`, which lets one run
    DDL (cluster-wide) queries for creating and dropping databases and distributed tables.
    Works only with successors of `sandbox.yasandbox.database.clickhouse.models.DistributedModel`
    """

    #: Default HTTP port for querying ClickHouse
    DB_PORT = 8123

    #: When a replica goes offline, do not query it for this amount of seconds at all.
    OFFLINE_REPLICA_TIMEOUT = 600

    #: A hack to figure ClickHouse's real HTTP port by sending a request to TCP port for binary client
    PORT_RE = re.compile("You must use port (\d+)")

    def __init__(
        self, db_name, cluster_name, db_url="http://localhost:8123/",
        username=None, password=None, readonly=False, autocreate=False, logger=None
    ):
        import logging
        self.cluster_name = cluster_name
        self.logger = (logger or logging).getLogger("clickhouse")
        self.db_exists = False
        super(DistributedDatabase, self).__init__(db_name, db_url, username, password, readonly, autocreate)
        self.db_exists = False
        self.shards = collections.Counter()

    @staticmethod
    def __test_address(host_name, port):
        try:
            requests.get("http://{}:{}".format(host_name, port)).raise_for_status()
            return True
        except requests.ConnectionError:
            return False

    @staticmethod
    def __extract_http_port(host_name, port):
        try:
            data = requests.get("http://{}:{}".format(host_name, port)).text
            match = re.search(DistributedDatabase.PORT_RE, data)
            return match and int(match.group(1))
        except requests.ConnectionError:
            return None

    def __wrap_db(self, db, instances):
        for name, meth in inspect.getmembers(db, inspect.ismethod):

            def wrapper(f):
                def inner(*a, **kw):
                    try:
                        return f(*a, **kw)
                    except requests.ConnectionError as exc:
                        self.logger.error(
                            "Kicking replica %s from list of alive replicas", db.db_url
                        )
                        instances.remove(db)
                        raise exceptions.ReplicaUnavailableException(exc)

                return inner

            setattr(db, name, wrapper(meth))

    @property
    @common_patterns.ttl_cache(OFFLINE_REPLICA_TIMEOUT)
    def instances(self):
        dbs = []
        self.shards.clear()

        for row in self.select(
            "SELECT host_name, shard_num, port FROM system.clusters WHERE cluster='{}'".format(self.cluster_name)
        ):
            host_name, port = row.host_name, self.DB_PORT
            if not self.__test_address(host_name, port):
                port = self.__extract_http_port(host_name, row.port)
                if port is None or not self.__test_address(host_name, port):
                    self.logger.error(
                        "No replica available on %s (either of %s ports), "
                        "assuming it stays offline for the next %s seconds",
                        host_name, list({self.DB_PORT, port} - {None}), self.OFFLINE_REPLICA_TIMEOUT
                    )
                    continue

            username, password = None, None
            if self.request_session.auth is not None:
                username, password = self.request_session.auth

            db = database.Database(
                self.db_name,
                "http://{}:{}".format(host_name, port),
                username=username,
                password=password,
                autocreate=False
            )

            self.__wrap_db(db, dbs)

            db.shard_num = row.shard_num
            self.shards[row.shard_num] += 1
            dbs.append(db)

        if not dbs:
            raise RuntimeError("There are no available replicas in ClickHouse cluster, go figure what's wrong")

        return dbs

    def create_database(self):
        for db in self.instances:
            db.db_exists = False
            try:
                db.create_database()
            except exceptions.ReplicaUnavailableException as ex:
                self.logger.error("Database creation failed, replica is treated as dead: %s", ex)

        self.db_exists = True

    def drop_database(self):
        for db in self.instances:
            try:
                db.drop_database()
            except exceptions.ReplicaUnavailableException as ex:
                self.logger.error("Database drop failed, replica is treated as dead: %s", ex)

    def create_table(self, model_class):
        if model_class.is_system_model():
            raise exceptions.DatabaseException("You can't create system table")
        if not issubclass(model_class, models.DistributedModel):
            raise exceptions.DatabaseException(
                "You can only create tables with DistributedDatabase's help for DistributedModel (sub)classes"
            )

        for db in self.instances:
            alive = True
            for statement in model_class.create_table_sql(db, self.cluster_name):
                if not alive:
                    continue

                try:
                    db._send(statement)
                except exceptions.ReplicaUnavailableException as ex:
                    self.logger.error("Table creation failed, replica is treated as dead: %s", ex)
                    alive = False

        model_class.init_auto_enums(self)

    def rename_table(self, model_class):
        """
        Rename the respective table into <table_name>__old using a distributed (ON CLUSTER) DDL query.
        Suffix becomes __old__<date>__<time> if such a table already exists.

        :param model_class: ClickHouse model class
        """

        suffix = "__old"
        signal_name = model_class.underlying_table_name()

        old_already_exists = bool(list(self.select(
            "SELECT * FROM system.tables WHERE database = '{db}' AND name LIKE '{name}{suffix}'".format(
                db=self.db_name,
                name=model_class.table_name(),
                suffix=suffix
            )
        )))
        if old_already_exists:
            suffix = "__old__{}".format(dt.datetime.now().strftime("%Y_%m_%d__%H_%M_%S"))

        self.logger.info(
            "%s: renaming distributed and replicated tables to %s",
            signal_name, "__".join((model_class.table_name(), suffix))
        )

        for table in (model_class.table_name(), model_class.underlying_table_name()):
            self._send(
                "RENAME TABLE `{db}`.`{table}` TO `{db}`.`{table}{suffix}` ON CLUSTER {cluster}".format(
                    db=self.db_name, table=table, suffix=suffix, cluster=self.cluster_name
                )
            )
        self.logger.info("%s: done renaming tables!", signal_name)

    def drop_table(self, model_class):
        if model_class.system:
            raise exceptions.DatabaseException("You can't drop system table")

        if not issubclass(model_class, models.DistributedModel):
            raise exceptions.DatabaseException(
                "You can only drop tables with DistributedDatabase's help for DistributedModel (sub)classes"
            )

        for statement in model_class.drop_table_sql(self.db_name):
            for db in self.instances:
                try:
                    db._send(statement)
                except exceptions.ReplicaUnavailableException as ex:
                    self.logger.error("Table drop failed, replica is treated as dead: %s", ex)

    def insert(self, model_instances, batch_size=None):
        kws = {}
        if batch_size is not None:
            kws.update(batch_size=batch_size)
        super(DistributedDatabase, self).insert(model_instances, **kws)
