import re
import itertools as it
import collections

import enum

import infi.clickhouse_orm.utils

from infi.clickhouse_orm import database
from infi.clickhouse_orm.fields import (
    StringField,
    Int8Field, Int16Field, Int32Field, Int64Field,
    BaseEnumField, Enum8Field, Enum16Field,
    DateField, DateTimeField, ArrayField,
    Float32Field, NullableField, LowCardinalityField,
)

from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.database.clickhouse import exceptions


TOLERABLE_ERRORS = (
    341,  # Alter is not finished because some replicas are inactive right now; Alter will be done asynchronously.
)


class EnumClass(enum.Enum):
    __allow_synonyms__ = False  # https://a.yandex-team.ru/review/913807


class AutoEnumField(BaseEnumField):
    ENUM_STUB = EnumClass("EnumStub", ["_"], start=0)
    ENUM_RE = re.compile(r"'([\w\-.]+)' = (\d+)")

    def __init__(self, default=None, alias=None, materialized=None, readonly=None):
        super(AutoEnumField, self).__init__(self.ENUM_STUB, default, alias, materialized, readonly)

    def get_sql(self, db=None):
        return super(AutoEnumField, self).get_sql(False, db=db)

    @property
    def db_type(self):
        assert len(self.enum_cls) < 32768, "Maximum enum size exceeded"
        return "Enum16"

    @staticmethod
    def to_python(value, _):
        return value

    @staticmethod
    def to_db_string(value, quote=True):
        return infi.clickhouse_orm.utils.escape(value, quote)

    def save_to_mongo(self, full_name, logger):
        mapping.AutoEnum.objects(name=full_name).update(
            set__values=[_.name for _ in self.enum_cls if _.name != "_"],
            upsert=True
        )
        logger.debug("%s: saved %d values to MongoDB", full_name, len(self.enum_cls))

    def load_from_mongo(self, full_name, logger):
        enum_state = mapping.AutoEnum.objects.with_id(full_name)
        if enum_state is None or not enum_state.values:
            logger.debug("%s: nothing is loaded from MongoDB", full_name)
            return False

        self.enum_cls = EnumClass("AutoEnum", enum_state.values)
        logger.debug("%s: loaded %d values from MongoDB", full_name, len(enum_state.values))
        return True

    def update_enum(self, db, model, field_name, items, alter_table=True, force=False):
        full_enum_name = ".".join((db.db_name, model.underlying_table_name(), field_name))

        if self.enum_cls is self.ENUM_STUB:
            enum_cls = EnumClass("AutoEnum", list(items)) if items else self.enum_cls
            db.logger.debug("Creating a new enum for %s", full_enum_name)
        else:
            enum_names = [_.name for _ in self.enum_cls if _.name != "_"]
            db.logger.debug(
                "Checking if enum %s needs updating (%d values exist already)", full_enum_name, len(enum_names)
            )
            new_items = set(items) - set(enum_names)
            enum_cls = EnumClass("AutoEnum", list(it.chain(enum_names, new_items))) if new_items else self.enum_cls
            if new_items:
                db.logger.info(
                    "New enum items %s for %s, current enum %s, new enum %s",
                    new_items, full_enum_name, list(self.enum_cls), list(enum_cls)
                )
        if enum_cls is not self.enum_cls or force:
            db.logger.debug("Enum %s update detected", full_enum_name)
            self.enum_cls = enum_cls
            well_done = True
            if alter_table:
                db.logger.debug("About to start altering tables for %s", full_enum_name)
                for table_name in (model.table_name(), model.underlying_table_name()):
                    full_table_name = "{}.{}".format(db.db_name, table_name)
                    shards = set()
                    fails = collections.Counter()
                    for db_instance in db.instances:
                        if table_name != model.table_name() and db_instance.shard_num in shards:
                            continue
                        field_sql = self.get_sql()
                        db.logger.info(
                            "Altering `%s.%s` to `%s` on %s",
                            full_table_name, field_name, field_sql, db_instance.db_url
                        )

                        try:
                            db_instance._send("ALTER TABLE {} MODIFY COLUMN `{}` {}".format(
                                full_table_name, field_name, field_sql
                            ))

                        except exceptions.ReplicaUnavailableException as ex:
                            well_done = False
                            db.logger.error("Alter failed, replica is treated as dead: %s", ex)

                        except database.DatabaseException as ex:
                            well_done = False
                            fatal = ex.code not in TOLERABLE_ERRORS
                            logging_method = db.logger.warning if fatal else db.logger.error
                            fails[db_instance.shard_num] += 1
                            failed_replicas = fails[db_instance.shard_num]
                            total_replicas = db.shards[db_instance.shard_num]
                            logging_method(
                                "Error occurred while altering table on replica %s from %s: %s. Fail count: %r",
                                failed_replicas, total_replicas, ex, fails
                            )
                            if fatal and (table_name == model.table_name() or failed_replicas >= total_replicas):
                                raise

                        shards.add(db_instance.shard_num)
            if well_done:
                self.save_to_mongo(full_enum_name, db.logger)

            return True
        return False

    @classmethod
    def parse_enum(cls, enum_type):
        members = {}
        for match in cls.ENUM_RE.finditer(enum_type):
            members[match.group(1)] = int(match.group(2))
        return members


def wrap_enum(class_):
    """
    Convert common.utils.Enum to a class with fields and __getattr__ and __getitem__ methods
    which fields.Enum*Field wants to see. This allows for reusing original enums
    without field modifications and order preservation.

    :type class_: Iterable
    """
    return EnumClass(class_.__name__, sorted(class_))


__all__ = [
    "StringField",
    "Int8Field", "Int16Field", "Int32Field", "Int64Field",
    "Enum8Field", "Enum16Field", "AutoEnumField",
    "DateField", "DateTimeField", "ArrayField",
    "Float32Field", "NullableField", "LowCardinalityField",
    "wrap_enum"
]
