"""
The module should be on the high level of modules dependencies hierarchy, so it should not depend on any
Sandbox package(s) except of `common`, only from third-party libraries and Skynet API.
"""

import collections

import six
from six.moves.urllib import parse as urlparse

import pymongo
from pymongo.errors import BSONError
from pymongo.errors import AutoReconnect
from pymongo.errors import ConnectionFailure
from pymongo.errors import OperationFailure
from pymongo.errors import DocumentTooLarge
from pymongo.read_preferences import ReadPreference

import mongoengine as me
import mongoengine.connection as mec


class FlexoDocument(me.Document):  # noqa
    meta = {
        "abstract": True,
        "strict": False,
        "auto_create_index": False
    }


class FlexoEmbeddedDocument(me.EmbeddedDocument):  # noqa
    meta = {
        "abstract": True,
        "strict": False,
    }


# Modern mongoengine fails to load documents with unknown fields in the database
# Patch it here to avoid modifying every single model
me.Document = FlexoDocument  # noqa
me.EmbeddedDocument = FlexoEmbeddedDocument  # noqa

from mongoengine import Q
# noinspection PyUnresolvedReferences
from mongoengine import MongoEngineConnectionError as ConnectionError
from mongoengine.errors import OperationError
# noinspection PyUnresolvedReferences
from mongoengine.errors import SaveConditionError
from mongoengine.errors import InvalidDocumentError
from mongoengine.errors import NotUniqueError
from mongoengine.errors import ValidationError
from mongoengine.errors import DoesNotExist
from mongoengine.context_managers import switch_db

from sandbox.common import fs
from sandbox.common import config
from sandbox.common import itertools
from sandbox.common.types.task import Priority
from sandbox.common.types import database as ctd

from . import base
from .autoenum import AutoEnum
from .client import Client
from .notification import Notification, UINotification
from .task_status_event import TaskStatusEvent
from .resource import Resource, ResourceCache, ResourceLink, ResourceMeta, Bucket
from .resource import ResourceForBackup, ResourcesToWarehouse
from .scheduler import Scheduler
from .semaphore import Semaphore, SemaphoreAudit
from .service import Service
from .settings import Settings
from .state import State
from .statistics import Statistics, Weather
from .task import Task, Template, Audit, TaskTagCache, ClientTagsToHostsCache, ParametersMeta
from .template import TaskTemplate, TemplateAudit
from .trigger import TimeTrigger, TaskStatusTrigger, TaskOutputTrigger, TaskStatusNotificationTrigger
from .user import User, Group, OAuthCache, RobotOwner
from .vault import Vault
from .yav import YavToken
from .abcd import ABCDAccount

#: Default connections pool size for pymongo
PYMONGO_POOL_SIZE = 10

#: Database-specific object ID type. Currently integer.
ObjectId = int

# database errors that should be retried
DB_TEMPORARY_ERRORS = (
    "write results unavailable",
    "could not contact primary",
    "(not master)"
)

# Database errors that should result in user error
DB_QUERY_ERRORS = (
    "command document too large",
    "document after update is larger",
    "Exceeded memory limit",
    "Sort operation used more than the maximum",
)

# noinspection PyUnresolvedReferences
__all__ = [
    "base",

    "ReadPreference",
    "OperationFailure", "BSONError", "DocumentTooLarge", "AutoReconnect", "ConnectionFailure", "ConnectionError",
    "OperationError", "SaveConditionError", "InvalidDocumentError", "NotUniqueError", "ValidationError",
    "DoesNotExist",

    "ObjectId", "Q", "switch_db",

    "TaskTemplate", "TemplateAudit",
    "Task", "Audit", "Priority", "Client", "Resource", "ResourceCache", "ResourceLink", "Scheduler", "ResourceMeta",
    "Template",
    "TaskTagCache", "ClientTagsToHostsCache", "ParametersMeta",
    "User", "Group", "OAuthCache", "RobotOwner",
    "Settings", "Notification", "UINotification", "Statistics", "Weather", "Service", "State",
    "YavToken", "Vault",
    "TimeTrigger", "TaskStatusTrigger", "TaskOutputTrigger", "TaskStatusNotificationTrigger",
    "Semaphore", "SemaphoreAudit",
    "AutoEnum",
    "ABCDAccount", "Bucket",
    "ResourceForBackup", "ResourcesToWarehouse",
    "TaskStatusEvent"
]


def ensure_connection(uri=None, max_pool_size=None, **kws):
    """
    Establishes two connections (read-write and read-only) to the database if its not established yet.
    :param uri: Custom connection URI for read-write and read-only connections.
    :param max_pool_size: Maximum connection pool size.
    :param kws: Additional connection parameters.
    :return Tuple with `pymongo` connection object and connection URI determined.
    """

    if uri is None:
        uri = config.Registry().server.mongodb.connection_url

    if uri.startswith("file://"):
        uri = fs.to_path(uri)
        with open(uri) as conf_file:
            uri = conf_file.read().strip()
    parsed = urlparse.urlparse(uri)
    if not parsed.scheme or parsed.scheme != "mongodb" or not parsed.path:
        raise Exception("Invalid database URI: {}".format(uri))

    Connection = collections.namedtuple("Connection", ("connection", "uri"))
    Connections = collections.namedtuple("Connections", ("rw", "rwp", "ro"))

    connections = []
    for alias, pref in (
        (ctd.ReadPreference.SECONDARY, pymongo.ReadPreference.SECONDARY),
        (ctd.ReadPreference.PRIMARY_PREFERRED, pymongo.ReadPreference.PRIMARY_PREFERRED),
        (ctd.ReadPreference.PRIMARY, pymongo.ReadPreference.PRIMARY),
    ):
        connections.append(Connection(
            mec.connect(
                parsed.path.strip("/"),
                alias=alias,
                host=uri,
                read_preference=pref,
                maxPoolSize=PYMONGO_POOL_SIZE if max_pool_size is None else max_pool_size,
                **kws
            ),
            uri
        ))

    return Connections(*reversed(connections))


def get_connection(alias=ctd.ReadPreference.PRIMARY, reconnect=False):
    return mec.get_connection(alias, reconnect=reconnect)


def disconnect():
    """ Disconnects both read-write and read-only connections. """
    for alias in ctd.ReadPreference:
        mec.disconnect(alias)
        mec._connection_settings.pop(alias, None)

    def clear_caches_for(class_):
        if class_._meta["abstract"]:
            for c in class_.__subclasses__():
                clear_caches_for(c)
        else:
            class_._collection = None
            class_._rp2collection = {}

    # MongoEngine caches Collection instance, which has a MongoClient
    clear_caches_for(me.Document)


def to_dict(bd):
    """
    Convert BaseDict from mongoengine to python dict
    """
    return {
        k: to_dict(v)
        if isinstance(v, me.base.BaseDict)
        else to_list(v)
        if isinstance(v, me.base.BaseList)
        else v
        for k, v in six.iteritems(bd)
    }


def to_list(bl):
    """
    Convert BaseList from mongoengine to python list
    """
    return [
        to_dict(v)
        if isinstance(v, me.base.BaseDict)
        else to_list(v)
        if isinstance(v, me.base.BaseList)
        else v
        for v in bl
    ]


def is_query_error(e):
    if not isinstance(e, (OperationError, OperationFailure)):
        return False

    for msg in DB_QUERY_ERRORS:
        if msg in str(e):
            return True

    return False


def retry_temporary_errors(fn, warn_fn=None, timeout=600):
    """
    Retries database errors that occur for a short period of time such as when
    replica set primary is unavailable.
    """
    last_error = None
    for _ in itertools.progressive_yielder(.1, 3, timeout, sleep_first=False):
        try:
            result = fn()
            break
        except OperationError as ex:
            if any(msg in ex.message for msg in DB_TEMPORARY_ERRORS):
                if warn_fn:
                    warn_fn(ex)
                last_error = ex
                continue
            raise
    else:
        raise last_error
    return result
