import time
import calendar
import datetime as dt
import operator as op
import functools

import six.moves
from six.moves import cPickle

import mongoengine as me

from sandbox.common import config
from sandbox.common import patterns
from sandbox.common.types import client as ctc

from . import base


class Client(base.ConnectionSwitcherMixin, me.Document):
    """
    The class represents a mapping between Python object and database storage for "Client" entity,
    i.e., Sandbox agent information.
    """
    meta = {
        "indexes": [
            {"fields": ["tags", "hostname"]},
        ]
    }

    Reloading = ctc.ReloadCommand
    Tag = ctc.Tag

    @classmethod
    def tags_query(cls, tags):
        return functools.reduce(op.or_, (
            me.Q(**{
                key: value
                for key, value in (
                    (
                        "tags__all",
                        list(six.moves.filter(
                            lambda _: not _.startswith(cls.Tag.Query.NOT),
                            six.moves.map(str, i)
                        ))
                    ),
                    (
                        "tags__nin",
                        list(six.moves.map(
                            lambda _: _[len(cls.Tag.Query.NOT):],
                            six.moves.filter(
                                lambda _: _.startswith(cls.Tag.Query.NOT),
                                six.moves.map(str, i)
                            )
                        ))
                    ),
                )
                if value
            })
            for i in ctc.Tag.Query(tags)
        ), me.Q())

    @property
    def lxc(self):
        """ Checks that client has lxc tag """
        return ctc.Tag.LXC in self.tags_set

    @property
    def porto(self):
        """ Checks that client supports porto """
        return ctc.Tag.PORTOD in self.tags_set

    @property
    def multislot(self):
        """ Checks that client supports multislot execution """
        return ctc.Tag.MULTISLOT in self.tags_set

    @property
    def fqdn(self):
        return self.info.get("system", {}).get("fqdn")

    @property
    def available(self):
        """ Check that client is available by Wall-E """
        if self.alive:
            return ctc.ClientAvailability.ALIVE

        if ctc.Tag.MAINTENANCE in self.tags:
            return ctc.ClientAvailability.DEAD
        else:
            return ctc.ClientAvailability.UNKNOWN

    @property
    def alive(self):
        """ Checks last ping time """
        return time.time() - self.updated_timestamp <= config.Registry().server.web.mark_client_as_dead_after

    @patterns.singleton_property
    def updated_timestamp(self):
        return calendar.timegm(self.updated.timetuple())

    @patterns.singleton_property
    def info(self):
        return cPickle.loads(self.context)

    @patterns.singleton_property
    def platforms(self):
        if not self.lxc and not self.porto:
            return
        return list(ctc.ContainerPlatforms)

    @patterns.singleton_property
    def ncpu(self):
        """ Temporary for compatibility with proxy client """
        return self.hardware.cpu.cores

    @patterns.singleton_property
    def ram(self):
        """ Temporary for compatibility with proxy client """
        return self.hardware.ram

    @patterns.singleton_property
    def arch(self):
        """ Temporary for compatibility with proxy client """
        return self.platform

    @patterns.singleton_property
    def model(self):
        """ Temporary for compatibility with proxy client """
        return self.hardware.cpu.model

    @patterns.singleton_property
    def tags_set(self):
        return set(self.tags)

    #: The sub-document encapsulates all the discovered information about agent's hardware.
    class Hardware(me.EmbeddedDocument):
        #: The sub-document encapsulates information about CPU
        class CPU(me.EmbeddedDocument):
            #: CPU model, for example "e5-2660"
            model = me.StringField(required=True)
            #: Amount of CPU cores (including virtual in case of HyperThreading enabled).
            cores = me.IntField(min_value=0, required=True)

        #: CPU information
        cpu = me.EmbeddedDocumentField(CPU, required=True)
        #: Total amount of RAM
        ram = me.IntField(min_value=0, required=True)
        #: Available disk space for tasks execution.
        disk_free = me.IntField(min_value=0, required=True)

    class Command(me.EmbeddedDocument):
        command = me.StringField(choices=list(iter(ctc.ReloadCommand)))
        author = me.StringField(default="")
        comment = me.StringField(default="")

    #: First part of a fully qualified domain name, for example, "sandbox1234" for sandbox1234.search.yandex.net
    hostname = me.StringField(min_length=1, primary_key=True)
    #: Time of last information update.
    updated = me.DateTimeField(required=True, default=dt.datetime.utcnow)
    #: Agent's platform (i.e., operating system) information.
    platform = me.StringField(required=True)
    #: Agent's hardware information.
    hardware = me.EmbeddedDocumentField(Hardware, required=True)
    #: Stack of service commands for agent
    reloading = me.ListField(me.StringField(choices=list(iter(Reloading))))
    #: Stack of service commands for agent with meta information
    pending_commands = me.ListField(me.EmbeddedDocumentField(Command))
    #: More details about the client:
    #: - revisions of packages it's running on;
    #: - list of tasks it's currently executing;
    #: - system information: job slots amount, kernel version, LXC/Porto capabilities, etc
    #: - ...
    context = me.BinaryField()
    #: Tags, including tag groups
    tags = me.ListField(me.StringField())
    #: Tags only
    pure_tags = me.ListField(me.StringField())
    #: Revision to control session consistency.
    revision = me.IntField(required=True, default=0)
