import mongoengine as me
from datetime import datetime

import sandbox.common.types.notification as ctn

from . import base
from . import task


class Notification(base.ConnectionSwitcherMixin, me.Document):
    """
    The class represents a mapping between Python object and database storage for "Notification" entity
    """
    meta = {"indexes": ["author", "transport", "date", "sent", "host"]}

    # notification author
    author = me.StringField(required=True, default="sandbox")
    # creation date
    date = me.DateTimeField(required=True, default=datetime.now)
    # list of addresses
    send_to = me.ListField(me.StringField(regex="[^@<>]*$"), required=True)
    # notification subject
    subject = me.StringField(default="[SANDBOX] default subject")
    # notification body
    body = me.StringField(default="")
    # send notification with specified transport param
    transport = me.StringField(choices=list(ctn.Transport), required=True)
    # notification type: text or html
    type = me.StringField(choices=list(ctn.Type), required=True)
    # send status of notification
    sent = me.BooleanField(default=False)
    # sender script hostname
    host = me.StringField(default="")
    # notification headers
    headers = me.ListField(me.StringField())
    # notification content charset
    charset = me.StringField(choices=list(ctn.Charset), required=True)
    # notification view type: DEFAULT, EXECUTION_REPORT, RELEASE_REPORT
    view = me.StringField(choices=list(ctn.View), default=ctn.View.DEFAULT)
    # task_id to generate task execution or release report
    task_id = me.IntField()
    # stamp used by mailman to mark inconsistent notifications
    inconsistent = me.BooleanField(default=False)
    # execution info
    execution = me.EmbeddedDocumentField(task.Task.Execution, db_field="exc")
    # whether the notification is important (will be sent from sandbox-urgent@)
    urgent = me.BooleanField(default=False)
    # check status
    check_status = me.StringField(df_field="cs", choices=list(ctn.JugglerStatus))
    # tags for Juggler
    juggler_tags = me.ListField(me.StringField(min_length=1, max_length=128), db_field="jt")


class UINotification(base.ConnectionSwitcherMixin, me.Document):
    """
    UI notifications model.
    """
    content = me.StringField(required=True)
    severity = me.StringField(choices=list(ctn.Severity), default=ctn.Severity.INFO)
