from __future__ import division, unicode_literals

import hashlib
import datetime as dt
import collections
import re

from bson import json_util

import six

import pymongo
import mongoengine as me

import sandbox.common.types.resource as ctr

from . import abcd
from . import base


class AttributeValueField(me.base.BaseField):

    def to_python(self, value):
        if isinstance(value, six.text_type):
            return value
        elif isinstance(value, six.binary_type):
            try:
                value = value.decode("utf-8")
            except UnicodeError:
                pass
        return value

    def validate(self, value, clean=True):
        if not isinstance(value, (bool, float) + six.integer_types + six.string_types):
            self.error("Attribute value accepts only simple types: string, integer, float, boolean")


class ResourceMeta(base.ConnectionSwitcherMixin, me.Document):
    """ Parameters views' meta information. """

    meta = {"indexes": ["hash"]}

    class Meta(me.EmbeddedDocument):
        def __init__(self, *args, **kwargs):
            releaseable = kwargs.pop("releaseable", None)
            releasable = kwargs.pop("releasable", None)
            super(ResourceMeta.Meta, self).__init__(*args, **kwargs)
            self.releaseable = releaseable
            self.releasable = releasable

        class AttributesMeta(me.EmbeddedDocument):
            #: Attribute type
            class_name = me.StringField()
            #: Attribute description
            description = me.StringField()
            #: Attribute default
            default = AttributeValueField()
            #: Attribute is required
            required = me.BooleanField()
            #: Attribute value
            value = AttributeValueField()

        #: resource name
        name = me.StringField(required=True)
        #: resource can be released
        __releasable = me.BooleanField(db_field="rel")

        @property
        def releaseable(self):
            # TODO: Remove after update all taskbox binaries
            return self.__releasable

        @releaseable.setter
        def releaseable(self, value):
            # TODO: Remove after update all taskbox binaries
            if value is not None and self.__releasable is None:
                self.__releasable = value

        @property
        def releasable(self):
            return self.__releasable

        @releasable.setter
        def releasable(self, value):
            if value is not None:
                self.__releasable = value

        #: resource can be stored on any arch
        any_arch = me.BooleanField(db_field="arch")
        #: list of resource releasers
        releasers = me.ListField(me.StringField())
        #: resource of executable
        executable = me.BooleanField(db_field="exe")
        #: create backup for resource
        auto_backup = me.BooleanField(db_field="back")
        #: calc md5 hash of resource
        calc_md5 = me.BooleanField(db_field="md5")
        #: on restart policy
        on_restart = me.StringField(choices=list(iter(ctr.RestartPolicy)))
        #: release subscribers
        release_subscribers = me.ListField(me.StringField())
        #: share resource
        share = me.BooleanField()
        #: parent hash
        parent = me.StringField()
        #: info about resource
        default_attribute = me.StringField()
        #: resource attributes
        attributes = me.MapField(me.DictField(), db_field="attr")

    # resource meta
    resource_meta = me.EmbeddedDocumentField(Meta)
    #: parameters hash
    hash = me.StringField()

    @property
    def calculated_hash(self):
        return hashlib.md5(self.resource_meta.to_json(sort_keys=True)).hexdigest()

    def save(self, **kws):
        self.hash = self.calculated_hash
        super(ResourceMeta, self).save(**kws)


class Resource(base.ConnectionSwitcherMixin, me.Document, base.Aggregatable):
    """
    mapping for resource
    """
    meta = {"indexes": [
        "type",
        "state",
        "task_id",
        "time.accessed",
        "time.created",
        "time.expires",
        "hosts_states.host",
        "attributes.key",
        "mds.namespace",
    ]}

    State = ctr.State

    class Attribute(me.EmbeddedDocument):
        #: key
        key = me.StringField(db_field="k", required=True)
        #: value
        value = AttributeValueField(db_field="v", required=True)

    class HostState(me.EmbeddedDocument):
        State = ctr.HostState

        #: host name
        host = me.StringField(db_field="h", required=True)
        #: resource state
        state = me.StringField(db_field="st", choices=list(State), required=True)

    class Time(me.EmbeddedDocument):
        """ Time marks aggregate. """
        #: Date of resource registration.
        created = me.DateTimeField(db_field="ct", default=dt.datetime.utcnow, required=True)
        #: Date of last resource synchronization.
        accessed = me.DateTimeField(db_field="at", default=dt.datetime.utcnow, required=True)
        #: Estimated date of resource expiration. In case of `None` - never expires.
        expires = me.DateTimeField(db_field="ex")
        #: Date of last resource meta info update
        updated = me.DateTimeField(db_field="up", default=dt.datetime.utcnow, required=True)

    class MDS(me.EmbeddedDocument):
        #: S3 key for file or metadata
        key = me.StringField(db_field="k", required=True)
        #: S3 bucket
        namespace = me.StringField(db_field="n")
        #: backup S3 bucket
        backup_namespace = me.StringField(db_field="bn")

    class SystemAttributes(me.EmbeddedDocument):
        #: path to binary for linux platform
        linux_platform = me.StringField(db_field="l")
        #: path to binary for osx platform
        osx_platform = me.StringField(db_field="o")
        #: path to binary for osx with arm64 platform
        osx_arm_platform = me.StringField(db_field="oa")
        #: path to binary for osx with windows platform
        win_nt_platform = me.StringField(db_field="w")

    #: Object ID - atomically incremented positive integer, primary key.
    id = me.SequenceField(primary_key=True)
    #: resource type
    type = me.StringField(required=True)
    #: resource name
    name = me.StringField(required=True)
    #: resource path (relative)
    path = me.StringField(required=True)
    #: resource owner
    owner = me.StringField(required=True)
    #: associated task
    task_id = base.ReferenceField(db_field="tid", required=True)
    #: supported OS architectures
    arch = me.StringField(required=True)
    #: platforms description for resources
    system_attributes = me.EmbeddedDocumentField(SystemAttributes)
    #: Time marks
    time = me.EmbeddedDocumentField(Time, required=True)

    #: resource state
    state = me.StringField(choices=list(State), default=State.NOT_READY, required=True)

    #: whole resource size, in KiB
    size = me.IntField(min_value=0)
    #: whole resource hashsum
    md5 = me.StringField()
    #: resource download id
    skynet_id = me.StringField(db_field="skyid")

    #: resource user attributes
    attributes = me.ListField(me.EmbeddedDocumentField(Attribute), db_field="attrs")

    #: resource hosts states
    hosts_states = me.ListField(me.EmbeddedDocumentField(HostState), db_field="hosts")

    #: MDS metadata
    mds = me.EmbeddedDocumentField(MDS)

    #: force cleanup flag
    force_cleanup = me.BooleanField(db_field="fc")

    #: True if the resource consists of multiple files
    multifile = me.BooleanField(db_field="mf")
    #: executable bit
    executable = me.BooleanField(db_field="ex")

    # resource meta information
    resource_meta = me.ListField(me.StringField(), db_field="rmet")

    def attributes_dict(self, exclude=None):
        return {
            attr.key: attr.value.encode("utf-8") if isinstance(attr.value, six.text_type) else str(attr.value)
            for attr in self.attributes
            if exclude is None or attr.key not in exclude
        }

    def to_json(self, *args, **kwargs):
        use_db_field = kwargs.pop('use_db_field', True)
        data = self.to_mongo(use_db_field=use_db_field)
        if self.resource_meta:
            resouce_meta_dict = {obj.hash: obj for obj in ResourceMeta.objects(hash__in=self.resource_meta)}
            data["resource_meta_objects"] = [
                resouce_meta_dict[meta_hash].to_mongo(use_db_field=use_db_field)
                for meta_hash in self.resource_meta
            ]
        return json_util.dumps(data, *args, **kwargs)

    @classmethod
    def from_json(cls, json_data, created=False):
        data = json_util.loads(json_data)
        resource_meta_objects = None
        if data.get("resource_meta_objects"):
            resource_meta_objects = [
                ResourceMeta._from_son(meta, created=created)
                for meta in data.get("resource_meta_objects")
            ]
            data.pop("resource_meta_objects")
        result = cls._from_son(data, created=created)
        result.resource_meta_objects = resource_meta_objects
        return result

    @classmethod
    def ensure_indexes(cls):
        """
        The method will provide query indexes usage hints in  addition to indexes creation for the collection.
        """
        super(Resource, cls).ensure_indexes()
        db = cls._get_db()
        # Provide here all popular queries hints with sorting by ID, otherwise Mongo will perform full collection scan
        # because its query planner selects index by ID when sorting by that field.
        db.command(
            "planCacheSetFilter", cls._get_collection_name(),
            query={"type": "X", "state": "X"}, sort={"_id": pymongo.DESCENDING},
            indexes=[{"type": 1}, {"state": 1}]
        )
        db.command(
            "planCacheSetFilter", cls._get_collection_name(),
            query={"type": "X", "state": "X", "tid": "X"}, sort={"_id": pymongo.DESCENDING},
            indexes=[{"type": 1}, {"state": 1}, {"tid": 1}]
        )
        db.command(
            "planCacheSetFilter", cls._get_collection_name(),
            query={"type": "X", "state": "X", "attrs": "X"}, sort={"_id": pymongo.DESCENDING},
            indexes=[{"type": 1}, {"state": 1}, {"attrs": 1}]
        )
        db.command(
            "planCacheSetFilter", cls._get_collection_name(),
            query={"type": "X", "state": "X", "attrs.k": "X"}, sort={"_id": pymongo.DESCENDING},
            indexes=[{"type": 1}, {"state": 1}, {"attrs.k": 1}]
        )

    @classmethod
    def storage_excessive_redundancy(cls, host, hosts, limit):
        """
        Returns resources list which has more than 2 copies on storage hosts in form of
          [{'_id': <ID>, 'amount': <COPIES AMOUNT>, 'type': <TYPE>, 'size': <SIZE>}, ...]

        :param host:    Host to be checked.
        :param hosts:   Hosts list to check copies on __excluding__ host to check.
        :param limit:   Limit result set to given amount.
        :return:        List of dicts (see above) ordered by amount of copies.
        """
        pipeline = [
            {"$match": {"state": cls.State.READY, "hosts.h": host}},
            {"$project": {"_id": 1, "hosts": 1, "type": 1, "size": 1, "mds": 1}},
            {"$unwind": "$hosts"},
            {"$match": {"hosts.h": {"$in": list(hosts)}}},
            {"$group": {
                "_id": "$_id",
                "amount": {"$sum": 1},
                "type": {"$last": "$type"},
                "size": {"$last": "$size"},
                "atime": {"$last": "$time.at"},
                "mds": {"$last": "$mds"}
            }},
            {"$project": {
                "_id": 1, "type": 1, "size": 1, "mds": 1,
                "amount": {
                    "$cond": {
                        "if": {"$ne": ["$mds", None]},
                        "then": "$amount",
                        "else": {"$add": ["$amount", 1]}
                    }
                }
            }},
            {"$match": {"amount": {"$gt": 2}}},
            {"$sort": {"amount": pymongo.DESCENDING, "atime": pymongo.ASCENDING}},
        ]
        if limit:
            pipeline.append({"$limit": limit})
        return cls.aggregate(pipeline, allowDiskUse=True)

    @classmethod
    def client_excessive_redundancy(cls, host, copies, limit):
        """
        Returns resources list which has more than given amount of copies on any host in form of
          [{'_id': <ID>, 'amount': <COPIES AMOUNT>, 'type': <TYPE>, 'size': <SIZE>, 'atime': <LAST ACCESS TIME>}, ...]

        :param host:    Host to be checked.
        :param copies:  Required amount of copies to be available for each resource.
        :param limit:   Limit result set to given amount.
        :return:        List of dicts (see above) ordered by access time and amount of copies.
        """
        pipeline = [
            {"$match": {
                "state": cls.State.READY,
                "hosts": {
                    "$elemMatch": {"h": host, "st": cls.HostState.State.OK},
                },
            }},
            {"$project": {
                "hosts": 1,
                "type": 1,
                "size": 1,
                "atime": "$time.at",
                "mds": {"$cond": [{"$gt": ["$mds", None]}, 2, 0]},  # MDS stores resources with factor 2
            }},

            {"$unwind": "$hosts"},
            {"$match": {
                "hosts.st": cls.HostState.State.OK,
            }},
            {"$group": {
                "_id": "$_id",
                "hosts": {"$sum": 1},
                "mds": {"$last": "$mds"},
                "type": {"$last": "$type"},
                "size": {"$last": "$size"},
            }},

            {"$project": {
                "type": 1,
                "size": 1,
                "atime": 1,
                "amount": {"$sum": ["$mds", "$hosts"]},
            }},

            {"$match": {"amount": {"$gt": copies}}},
            {"$sort": {"atime": pymongo.ASCENDING, "amount": pymongo.DESCENDING}},
            {"$limit": limit},
        ]
        return cls.aggregate(pipeline, allowDiskUse=True)

    @classmethod
    def clients_insufficient_redundancy(cls, storages, hosts=None):
        """
        Lists resources for the whole cluster (clients only), which has redundancy problems, i.e.,
        has no backup on `H@SANDBOX_STORAGE` hosts group, while it was requested by adding
        `backup_task` attribute.

        :param storages:    A collection with storage hosts (`H@SANDBOX_STORAGE` hosts group).
        :param hosts:       Check only hosts specified.
        :return:            A list of named resource information tuples of following structure:
                            - 'id'   - resource ID,
                            - 'task' - value of 'backup_task' attribute,
                            - 'type' - resource type,
                            - 'size' - resource size in KiB,
                            - 'host' - one of hosts, which holds the resource.
        """

        tuple_t = collections.namedtuple("Resource", ["id", "task", "type", "size", "host"])
        query = {
            "state": cls.State.READY,
            "attrs.k": "backup_task",
            "mds": None,
            "hosts": {
                "$not": {
                    "$elemMatch": {
                        "h": {"$in": list(storages)},
                        "st": cls.HostState.State.OK
                    }
                }
            }
        }
        if hosts:
            query["hosts.h"] = {"$in": list(hosts)}
        cursor = cls._get_collection().find(
            query,
            ["hosts", "type", "size", "attrs"],
            sort=[("_id", pymongo.ASCENDING)],
        )
        for r in cursor:
            task = next((_["v"] for _ in r["attrs"] if _["k"] == "backup_task"), 0)
            yield tuple_t(
                r["_id"],
                (int(task) if task.isdigit() else 0) if isinstance(task, six.string_types) else task,
                r["type"],
                r["size"],
                next((_["h"] for _ in r.get("hosts", []) if _["st"] == cls.HostState.State.OK), None)
            )

    @classmethod
    def host_resources_stats(cls, host):
        """
        Returns total amount and size of resources registered on the given host,
        including resources, marked as DELETED or BROKEN.
        :param host:    Host to be counted.
        :return         Dict of tuples, where key is resource state and value is
                        a tuple with resources amount and total size in KiB for this resource type.
        """
        pipeline = [
            {'$match': {'hosts.h': host}},
            {'$project': {'_id': 0, 'state': 1, 'size': 1}},
            {'$group': {
                '_id': '$state',
                'amount': {'$sum': 1},
                'size': {'$sum': '$size'},
            }},
        ]
        return {r['_id']: (r['amount'], r['size']) for r in cls.aggregate(pipeline)}

    @classmethod
    def storage_top_usage(
        cls,
        storages, since=None, to=None, immortal=False, group_by='type', order_by='size', owner=None,
        state=State.READY,
    ):
        """
        List resources which have backup on `H@SANDBOX_STORAGE` hosts group, grouped by resource type or owner
        and ordered by sum of resources size

        :param storages:        A collection with storage hosts (`H@SANDBOX_STORAGE` hosts group).
        :param since:           Lower bound for query period of resource creation date. 30 days if not provided.
        :param to:              Upper bound for query period of resource creation date. Now if not provided.
        :param immortal:        Query resources only with `do_not_remove` flag.
        :param group_by:        Resource field to group size by.
        :param order_by:        Resource field to order the table by.
        :param owner:           Filter resources for given owner only.
        :param state:           Resources state to be checked
        :return:                A list of named resource information tuples of following structure:
                                - 'type' - resource type,
                                - 'size' - cumulative resource size in KiB,
                                - 'amount' - cumulative amount of resources.
        """

        hosts = list(storages)
        now = dt.datetime.now()
        match = {
            'hosts.st': cls.HostState.State.OK,
            'hosts.h': {'$in': hosts},
            'time.ct': {
                '$gt': since if since else now - dt.timedelta(days=30),
                '$lte': to if to else now
            },
        }
        if immortal:
            match['time.ex'] = {'$exists': False}
        if owner:
            match['owner'] = owner
        if state:
            match['state'] = state
        pipeline = [
            {'$match': match},
            {'$project': {'hosts.h': 1, group_by: 1, 'size': 1}},
            {'$unwind': '$hosts'},
            {'$match': {'hosts.h': {'$in': hosts}}},
            {'$group': {
                '_id': '$' + group_by,
                'size': {'$sum': '$size'},
                'amount': {'$sum': 1},
            }},
            {'$sort': {order_by: pymongo.DESCENDING}},
        ]

        tuple_t = collections.namedtuple('Resource', [group_by, 'size', 'amount'])
        for r in cls.aggregate(pipeline):
            yield tuple_t(r['_id'], r['size'], r['amount'])

    @classmethod
    def released_resources(
        cls,
        resource_type=None, release_status=None, include_broken=False, task_id=None, arch=None, limit=None, offset=None
    ):
        """
        list released resources with "type", releases status "status" for "task_id"
        ret:
        """
        match = {'state': {'$in': [cls.State.READY, cls.State.BROKEN]} if include_broken else cls.State.READY}
        if release_status:
            match['attrs'] = {'$elemMatch': {'k': 'released', 'v': release_status}}
        else:
            match['attrs'] = {'$elemMatch': {'k': 'released'}}
        if resource_type:
            match['type'] = resource_type
        if task_id:
            match['tid'] = task_id
        if arch:
            match['arch'] = {'$in': (arch, 'any')}

        pipeline = [
            {'$match': match},
            {'$group': {
                '_id': '$tid',
                'res': {'$addToSet': '$_id'},
            }},
            {'$sort': {'_id': pymongo.DESCENDING}},
        ]
        if offset:
            pipeline.append({'$skip': int(offset)})
        if limit:
            pipeline.append({'$limit': int(limit)})
        return [(r['_id'], r['res']) for r in cls.aggregate(pipeline)]

    @classmethod
    def hosters(cls):
        """ Returns a dict of known resource hosters with amount of resources per host. """
        pipeline = [
            {'$match': {'state': cls.State.READY}},
            {'$unwind': '$hosts'},
            {'$project': {'hosts.h': 1}},
            {'$group': {'_id': '$hosts.h', 'amount': {'$sum': 1}}},
        ]
        return {r['_id']: r['amount'] for r in cls.aggregate(pipeline)}

    @classmethod
    def ready_resources_size(cls):
        """
        Collect total size(KiBs) of READY resources per host

        :return: per host size of READY resources in Bytes
        :rtype: list of dicts with keys: _id - hostname, total_size - size of ready resources in KiBs
        """
        pipeline = [
            {"$match": {"state": "READY"}},
            {"$project": {"size": 1, "hosts.h": 1}},
            {"$unwind": "$hosts"},
            {"$group": {"_id": "$hosts.h", "total_size": {"$sum": "$size"}}}
        ]
        return cls.aggregate(pipeline)

    @classmethod
    def resources_per_state(cls, delta=dt.timedelta(hours=1)):
        """
        Get amount of resources per state since `since` param

        :param delta: timedelta object to bound minimal update time of resource
        :return: Per state amount of resources
        :rtype: dict
        """
        now = dt.datetime.utcnow()
        res = {st.lower(): 0 for st in cls.State}
        pipeline = [
            {"$match": {"time.at": {
                "$gte": now - delta - dt.timedelta(seconds=1),
                "$lt": now - dt.timedelta(seconds=1)}
            }},
            {"$group": {"_id": "$state", "amount": {"$sum": 1}}}
        ]
        res.update({doc["_id"].lower(): doc["amount"] for doc in cls.aggregate(pipeline)})
        return res

    @classmethod
    def last_resources(cls, group_by, resource_type=None, state=None, order_by="-_id"):
        """
        Returns latest resource identifier grouped by the field provided.

        :param group_by:        a field name to group results by.
        :param resource_type:   resource type to match if provided.
        :param state:           resource state to match if provided.
        :return:                Generator with resources' identifiers.
        :rtype: generator
        """
        query = {}
        if resource_type:
            if isinstance(resource_type, (list, tuple)):
                query['type'] = {'$in': resource_type}
            else:
                query['type'] = resource_type
        if state:
            if isinstance(state, (list, tuple)):
                query['state'] = {'$in': state}
            else:
                query['state'] = state

        pipeline = []
        if query:
            pipeline.append({"$match": query})

        order_by, order_direction = order_by[1:], (pymongo.DESCENDING if order_by[0] == '-' else pymongo.ASCENDING)
        pipeline.append({"$project": {group_by: 1, "_id": 1, order_by: 1}})
        pipeline.append({"$sort": {order_by: order_direction}})
        pipeline.append({"$group": {"_id": '$' + group_by, "identifier": {"$first": '$_id'}}})

        return ((row['_id'], row['identifier']) for row in cls.aggregate(pipeline))


class ResourceCache(base.ConnectionSwitcherMixin, me.Document):
    """
    Cache entry for resource query
    """

    #: TTL of the cache entry
    TTL = dt.timedelta(minutes=10)

    #: MD5 hash for resource list query
    query_hash = me.StringField(primary_key=True, db_field="qh", required=True)

    #: optimistic inclusive lower bound of resource ID for the query
    resource_id = base.ReferenceField(db_field="rid", required=True)

    #: timestamp of the last resource ID update
    updated = me.DateTimeField(db_field="ut", default=dt.datetime.utcnow, required=True)

    @classmethod
    def ttl_threshold(cls):
        """
        Return a datetime before which a cache entry is considered invalid
        :rtype: `datetime`
        """
        return dt.datetime.utcnow() - cls.TTL


class ResourceLink(base.ConnectionSwitcherMixin, me.Document):
    """
    mapping for temporary resource links
    """
    meta = {"indexes": [
        "resource_id",
        "accessed",
    ]}

    #: Link identifier
    id = me.StringField(primary_key=True)
    #: User's login, which can be identified by the link
    author = me.StringField(required=True)
    #: Resource id
    resource_id = me.IntField(required=True)
    #: Date of last link access.
    accessed = me.DateTimeField(db_field="at", default=dt.datetime.utcnow, required=True)


class Bucket(base.ConnectionSwitcherMixin, me.Document):
    """
    mapping for MDS-S3 bucket
    """

    meta = {"indexes": [
        "abc",
        "abcd_account.id",
        "abcd_account.folder_id",
        "free",
    ]}

    BUCKET_NAME_REGEXP = re.compile(r"^{}(\d+)(-([-\w]+))?$".format(ctr.BUCKETS_PREFIX))
    DEFAULT_ABCD_ACCOUNT = "default"

    #: bucket name
    name = me.StringField(primary_key=True)
    #: ABCD account specific fields
    abcd_account = me.EmbeddedDocumentField(abcd.ABCDAccount, db_field="abcd")
    #: ABC service slug the bucket is belong to
    abc = me.StringField()
    #: disable notifications about bucket exhausting
    ignore_bucket_exhaust = me.BooleanField(default=False)
    #: free space threshold in bytes, when decreasing it, removing of resources without TTL will be started
    lru_threshold = me.IntField(default=None, min_value=0)
    #: free space in bytes
    free = me.IntField(default=0)
    #: total space in bytes
    total = me.IntField(default=0)
    #: last update time
    updated_at = me.DateTimeField()

    @classmethod
    def abcd_account_name(cls, name):
        if name in ctr.SPECIAL_BUCKETS:
            return name[len(ctr.BUCKETS_PREFIX):]
        m = cls.BUCKET_NAME_REGEXP.match(name)
        if m is None:
            return None
        return m.groups()[2] or cls.DEFAULT_ABCD_ACCOUNT


class ResourceForBackup(base.ConnectionSwitcherMixin, me.Document):
    """
    Queue of resources to backup
    """
    meta = {"indexes": [
        "sources",
        "lock",
        "till",
    ]}

    #: resource id
    id = me.IntField(primary_key=True)
    #: resource size, in KiB
    size = me.IntField(min_value=0)
    #: source hosts
    sources = me.ListField(me.StringField())
    #: host that acquired lock
    lock = me.StringField()
    #: time of adding to queue
    added = me.DateTimeField(default=dt.datetime.utcnow, required=True)
    #: resource lock expiration time
    till = me.DateTimeField()


class ResourcesToWarehouse(base.ConnectionSwitcherMixin, me.Document):
    """
    Queue of resources to copy to warehouse S3 bucket
    """

    #: resource id
    id = me.IntField(primary_key=True)
    #: resource size, in KiB
    size = me.IntField(min_value=0)
    #: S3 bucket
    bucket = me.StringField(required=True)
    #: time of adding to queue
    added_at = me.DateTimeField(default=dt.datetime.utcnow, required=True)
