import datetime as dt

import mongoengine as me

import sandbox.common.types.scheduler as cts

from . import base
from . import task


class Scheduler(base.ConnectionSwitcherMixin, task.Template, me.Document):
    """ The class represents a mapping between Python object and database storage for "Scheduler" entity """

    meta = {
        "indexes": [
            "type",
            "owner",
            "author",
            "tags",
            "status",
        ],
    }

    Status = cts.Status

    class SchedulerNotifications(me.EmbeddedDocument):
        """ Full notification info for scheduler """

        class Notification(task.Template.BaseNotification):
            """ Notification info for scheduler. """

            # scheduler statuses for notification
            statuses = me.ListField(me.StringField(choices=list(cts.Status)), db_field="st", required=True)
            # resolved scheduler recipients
            resolved_recipients = me.ListField(me.StringField(), required=True)

        #: scheduler notification info
        notifications = me.ListField(me.EmbeddedDocumentField(Notification), db_field="sched_noti")
        #: time if last notification update
        updated = me.DateTimeField(db_field="up", default=dt.datetime.utcnow, required=True)

    class Time(me.EmbeddedDocument):
        """ The sub-document that encapsulates time data """

        #: Creation time.
        created = me.DateTimeField(db_field="ct", default=dt.datetime.utcnow, required=True)
        #: Update time.
        updated = me.DateTimeField(db_field="up", default=dt.datetime.utcnow, required=True)
        #: Next run time.
        next_run = me.DateTimeField()
        #: Last run time.
        last_run = me.DateTimeField()
        #: Check time
        checked = me.DateTimeField()

    class Plan(me.EmbeddedDocument):
        """ Task planning data """

        #: Days of week to run mask
        days_of_week = me.IntField(required=True, default=127)
        #: Retry mode
        retry = me.StringField(choices=list(cts.Retry), default=cts.Retry.NO, required=True)
        #: Time interval in seconds between task runs
        interval = me.IntField(required=True, default=60 * 60)
        #: Time interval in seconds to run next task after failed task
        retry_interval = me.IntField(required=True, default=60 * 60)
        #: Start mode
        start_mode = me.StringField(choices=list(cts.StartMode), default=cts.StartMode.IMMEDIATELY, required=True)
        #: First task start time
        start_time = me.DateTimeField(required=False)
        #: Repetition mode
        repetition = me.StringField(choices=list(cts.Repetition), default=cts.Repetition.NO, required=True)
        #: Schedule tasks sequentially
        sequential_run = me.BooleanField(default=False, required=True)

    #: Object ID - atomically incremented positive integer, primary key.
    id = me.SequenceField(primary_key=True)

    #: Task planning parameters
    plan = me.EmbeddedDocumentField(Plan, default=Plan, required=True)
    #: Current scheduler's state.
    status = me.StringField(choices=list(Status), default=Status.STOPPED, required=True)
    #: The time border for the last task state information message to be sent
    last_notification_time = me.DateTimeField(default=dt.datetime.utcnow, required=True)

    #: Time data
    time = me.EmbeddedDocumentField(Time, default=Time, required=True)
    #: Current executing tasks ids
    cur_tasks_ids = me.ListField(me.IntField(), db_field="tids")

    #: scheduler notification info
    scheduler_notifications = me.EmbeddedDocumentField(SchedulerNotifications)
