import datetime as dt
import mongoengine as me

from . import base


class Semaphore(base.ConnectionSwitcherMixin, me.Document):
    """
    Mapping class between Python objects and 'semaphore' collection in database.
    """
    meta = {
        "indexes": [
            {"fields": ["owner", "shared"]},
            "name", "auto"
        ]
    }

    #: Semaphore time marks
    class Time(me.EmbeddedDocument):
        """
        Semaphore time marks
        """
        #: creation date and time
        created = me.DateTimeField(required=True, default=dt.datetime.utcnow)
        #: last change time
        updated = me.DateTimeField(required=True, default=dt.datetime.utcnow)

    #: primary key
    id = me.SequenceField(primary_key=True)

    #: name
    name = me.StringField(required=True, unique=True)

    #: group owns the semaphore
    owner = me.StringField(required=True)

    #: list of groups which owned tasks that allowed to require this semaphore
    shared = me.ListField(me.StringField())

    #: time marks
    time = me.EmbeddedDocumentField(Time, required=True)

    #: shows that the semaphore created automatically
    auto = me.BooleanField(default=False)

    #: maximum capacity
    capacity = me.IntField(default=1, min_value=0)

    #: shows that the semaphore is available to acquire by everyone
    public = me.BooleanField(default=False)


class SemaphoreAudit(base.ConnectionSwitcherMixin, me.Document):
    """
    The class represents a mapping between Python object and database storage for semaphore audit entity
    """
    meta = {"indexes": ["semaphore_id", "time", "author"]}

    #: Id of semaphore
    semaphore_id = me.IntField(required=True)
    #: Date and time of the entry
    time = me.DateTimeField(required=True, default=dt.datetime.utcnow)
    #: Entry description
    description = me.StringField()
    #: The entry author
    author = me.StringField()
    #: The host, which added the entry
    target = me.StringField()
    #: Ip of remote host that requested changes
    source = me.StringField()
