import datetime as dt

import mongoengine as me

from sandbox.common import enum

from . import base


class Settings(base.ConnectionSwitcherMixin, me.Document):
    """
    Misc collection to store various cluster-wide Sandbox runtime-defined settings.
    Actually, this collection stores only one document with various runtime-specific data.
    """

    class OperationMode(enum.Enum):
        """ Server operation mode enumeration. """
        NORMAL = None
        READ_ONLY = None
        PRE_READ_ONLY = None

    class Time(me.EmbeddedDocument):
        """ Sub-document encapsulates all the database timestamps. """
        #: Database initialization date
        created = me.DateTimeField(required=True, default=dt.datetime.utcnow)
        #: Last update execution.
        updated = me.DateTimeField(required=True, default=dt.datetime.utcnow)
        #: Last usage time. Updated each minute by running server.
        used = me.DateTimeField(required=True, default=dt.datetime.utcnow)

    class Updates(me.EmbeddedDocument):
        """ Sub-document encapsulates information about database updates. """

        class Appiled(me.EmbeddedDocument):
            """ This one aggregates lists of applied updates by stages. """
            pre = me.ListField(me.StringField())
            main = me.ListField(me.StringField())
            post = me.ListField(me.StringField())

        #: Lists collection of applied database updates.
        applied = me.EmbeddedDocumentField(Appiled, required=True)
        #: A host name, which currently executes updates.
        executor = me.StringField()

    #: Current server operation mode.
    operation_mode = me.StringField(choices=list(OperationMode), required=True, default=OperationMode.NORMAL)
    operation_mode_set_by = me.StringField()
    #: Various timestamps.
    time = me.EmbeddedDocumentField(Time, required=True)
    #: Database updates information
    updates = me.EmbeddedDocumentField(Updates, required=True)
